# -*- coding: utf-8 -*-

import logging
import waffle

from django.conf import settings

from idm.core.management.base import IdmBaseCommand
from idm.core.models import SystemMetainfo
from ids.registry import registry

log = logging.getLogger(__name__)

SUMMARY_TEMPLATE = 'Не удалось перезапросить роль в системе {system_name}'

DESCRIPTION_TEMPLATE = """
    Привет!

    IDM не смог перезапросить следующие роли из-за ошибки в системе {system_name}

    Если не устранить причину ошибки, то это может привести к проблемам с доступом."""

COMMENT_TEMPLATE = """Привет!

На данный момент IDM не смог перезапросить следующие роли из-за ошибки в системе {system_name}:

{links}

Если не устранить причину ошибки, то это может привести к проблемам с доступом.

Если тебе нужна помощь, обращайся к дежурному по IDM https://nda.ya.ru/t/bzb9j7w75DbGFy"""


ROLE_LINK_TEMPLATE = "{i}. https://idm.yandex-team.ru/system/{system_slug}#role={role_id},f-role-id={role_id}"

INVISIBLE_ROLE_LINKS_TEMPLATE = """\n
    <{{ Еще {len_invisible_links} ролей
        {invisible_links}
    }}>
    """

MAX_LINK_NUMBER = 20


class Command(IdmBaseCommand):
    help = 'Сообщает системам о неудачном пересмотре ролей'
    intranet_only = True

    def idm_handle(self, *args, **options):
        if not waffle.switch_is_active('create_issue_of_failed_role_review'):
            log.info('Switch idm_create_issue_of_failed_role_review is not active')
            return

        system_meta_info_with_failed_roles = SystemMetainfo.objects.filter(
            roles_failed_on_last_review__isnull=False
        ).select_related('system')

        rep_issues = registry.get_repository(
            'startrek2',
            'issues',
            user_agent=settings.IDM_ROBOT_USERNAME,
            oauth_token=settings.IDM_TRACKER_OAUTH_TOKEN,
            timeout=settings.IDM_TRACKER_TIMEOUT,
        )

        for system_meta_info in system_meta_info_with_failed_roles:
            system = system_meta_info.system

            visible_links = []
            invisible_links = []

            for i, role_id in enumerate(system_meta_info.roles_failed_on_last_review):
                container = visible_links
                if i >= MAX_LINK_NUMBER:
                    container = invisible_links
                container.append(ROLE_LINK_TEMPLATE.format(i=i+1, role_id=role_id, system_slug=system.slug))
            links_text = "\n".join(visible_links)
            if invisible_links:
                links_text += INVISIBLE_ROLE_LINKS_TEMPLATE.format(
                    len_invisible_links=len(invisible_links),
                    invisible_links="\n".join(invisible_links)
                )

            query = f'Queue: {settings.IDM_FAILED_REVIEW_QUEUE} and Resolution: empty() and Tags: "{system.slug}"'
            issues = rep_issues.client.issues.find(
                query=query
            )
            if issues:
                issue = issues[0]
            else:
                issue = rep_issues.client.issues.create(
                    queue=settings.IDM_FAILED_REVIEW_QUEUE,
                    summary=SUMMARY_TEMPLATE.format(system_name=system.get_name()),
                    description=DESCRIPTION_TEMPLATE.format(system_name=system.get_name()),
                    tags=[system.slug],
                )
            responsible_for_system = [user.username for user in system.get_responsibles()]
            issue.comments.create(
                text=COMMENT_TEMPLATE.format(links=links_text, system_name=system.get_name()),
                summonees=responsible_for_system,
            )
