# -*- coding: utf-8 -*-

import logging

from django.conf import settings
from django.db.models import DurationField, ExpressionWrapper, F, Q
from django.utils import timezone
import waffle

from idm.core.constants.email_templates import EXPIRATION_ROLES_TEMPLATES
from idm.core.constants.role import ROLE_STATE
from idm.core.management.base import IdmBaseCommand
from idm.core.models import Role
from idm.framework.requester import requesterify
from idm.users.models import User

log = logging.getLogger(__name__)


class Command(IdmBaseCommand):
    help = 'Переводим истекающие роли в статус expiring и отсылаем письма'
    intranet_only = True

    def idm_handle(self, *args, **options):
        if not waffle.switch_is_active('find_expiring_roles'):
            log.info('Switch find_expiring_roles is not active')
            return

        common_q = Q(
            state=ROLE_STATE.GRANTED,
            expire_at__lt=timezone.now() + timezone.timedelta(days=settings.IDM_EXPIRING_DAYS),
            system__is_active=True,
            system__is_broken=False,
        )

        expiring_with_ttl_date = (
            Role.objects
            .annotate(
                duration_from_ttl_date=ExpressionWrapper(F('ttl_date') - F('added'), output_field=DurationField())
            )
            .filter(
                common_q,
                ttl_date__isnull=False,
                duration_from_ttl_date__gte=timezone.timedelta(days=settings.IDM_EXPIRING_DAYS),
            )
            .select_related('system', 'node')
        )

        expiring_with_ttl_days = (
            Role.objects
            .filter(
                common_q,
                ttl_days__isnull=False,
                ttl_days__gte=settings.IDM_EXPIRING_DAYS,
            )
            .select_related('system', 'node')
        )

        for role in expiring_with_ttl_date | expiring_with_ttl_days:
            role.set_state(ROLE_STATE.EXPIRING, requester=requesterify(User.objects.get_idm_robot()))
            Role.objects.send_reminder(role.get_subject(), [role], EXPIRATION_ROLES_TEMPLATES, expiring=True)
