# -*- coding: utf-8 -*-


import logging

from django.core.management.base import BaseCommand
from django.db import connection

from idm.core.constants.role import ROLE_STATE
from idm.core.models import Role

log = logging.getLogger(__name__)

find_duplicated_sql = '''
SELECT role1.id, role2.id
FROM upravlyator_role as role1
JOIN upravlyator_role as role2
ON (role1.id < role2.id AND
    role1.system_id = role2.system_id AND
    COALESCE(role1.user_id, -1) = COALESCE(role2.user_id, -1) AND
    COALESCE(role1.group_id, -1) = COALESCE(role2.group_id, -1) AND
    role1.node_id = role2.node_id AND
    COALESCE(role1.parent_id, -1) = COALESCE(role2.parent_id, -1) AND
    COALESCE(role1.fields_data, '') = COALESCE(role2.fields_data, '')
)
WHERE (role1.state IN (%s) AND
      role2.state IN (%s)
);'''


def get_duplicated_roles():
    states_to_check = ROLE_STATE.RETURNABLE_STATES
    states_representation = ', '.join(("'%s'" % state) for state in states_to_check)

    with connection.cursor() as cursor:
        cursor.execute(find_duplicated_sql % (states_representation, states_representation))
        rows = cursor.fetchall()
    return rows


def group_duplicates(rows):
    groups = []
    for id1, id2 in rows:
        for group in groups:
            if id1 in group or id2 in group:
                group |= {id1, id2}
                break
        else:
            groups.append({id1, id2})
    return groups


def decide_group(group):
    # active returnable в порядке увеличения "значимости"
    states_by_priority = ['need_request', 'imported', 'rerequested', 'review_request', 'onhold', 'granted']

    group_roles = Role.objects.filter(pk__in=group)
    active_role_pk = None
    potential_roles = group_roles.filter(state__in=states_by_priority)
    if potential_roles.exists():
        active_role = max(potential_roles, key=lambda role: (states_by_priority.index(role.state), role.pk))
        active_role_pk = active_role.pk

    if active_role_pk is None:
        active_role_pk = max(group)

    roles_to_deprive = group_roles.exclude(pk=active_role_pk)
    return active_role_pk, roles_to_deprive


class Command(BaseCommand):
    help = """Одноразовая команда, отзывает задублированные роли. Пуша в систему при этом не должно происходить."""

    def handle(self, *args, **options):
        rows = get_duplicated_roles()
        log.info('Found %d rows', len(rows))

        groups = group_duplicates(rows)
        log.info('Found %d groups of duplicating roles', len(groups))

        total_roles = Role.objects.filter(state__in=ROLE_STATE.RETURNABLE_STATES).count()
        deleted_roles = 0
        for group in groups:
            active_role_pk, roles_to_deprive = decide_group(group)
            comment = 'Отзыв задублированной роли в пользу роли с id=%d' % active_role_pk
            for role in roles_to_deprive:
                try:
                    log.info('Trying to deprive role %d', role.pk)

                    role.no_email = True
                    role.save(update_fields=('no_email',))
                    role.deprive_or_decline(None, comment=comment, bypass_checks=True, deprive_all=True)

                    deleted_roles += 1
                    log.info('Successfully deprived duplicating role with id=%d', role.pk)
                except Exception:
                    log.exception('Could not deprive role with id=%d', role.pk)

            log.info('#returnable roles: %d', total_roles)
            log.info('#groups: %d', len(groups))
            log.info('#deleted roles: %d', deleted_roles)
