# -*- coding: utf-8 -*-


import logging

from django.core.management.base import CommandError

from idm.core.management.base import IdmBaseCommand
from idm.utils.celery import parse_message
from idm.utils.mongo import get_mongo_db


log = logging.getLogger(__name__)


class Command(IdmBaseCommand):
    help = """Одноразовая команда для удаления из mongo тасок определённого типа"""

    def add_arguments(self, parser):
        super(Command, self).add_arguments(parser)
        parser.add_argument(
            '--tasks_type',
            required=True,
            action='store',
            dest='tasks_type',
        )
        parser.add_argument(
            '--ignore_error_decode_body',
            required=False,
            default=False,
            action='store',
            dest='ignore_error',
        )

    def idm_handle(self, *args, **options):
        tasks_type = options['tasks_type']
        ignore_error = options['ignore_error']
        log.info('Start remove tasks from mongo for %s, ignore error: %s', tasks_type, ignore_error)

        objects = []
        error_messages = []

        mongo_db = get_mongo_db()
        for message in mongo_db.messages.find():
            _id = message['_id']
            try:
                body = parse_message(message)
            except Exception:
                if ignore_error:
                    error_messages.append(message)
                    continue
                else:
                    raise CommandError('Error while parse message: %s' % message)
            if body.get('task') == tasks_type:
                objects.append(_id)
        log.info('Found %s tasks to remove, errors: %s. Starting to remove', len(objects), len(error_messages))

        LIMIT = 1000
        for offset in range(0, len(objects), LIMIT):
            ids = objects[offset:offset + LIMIT]
            mongo_db.messages.remove({'_id': {'$in': ids}})
        log.info('All tasks have been removed')
