# -*- coding: utf-8 -*-


import logging

from django.db import connection, transaction

from idm.core.management.base import IdmBaseCommand
from idm.core.models import Role, RoleNode
from idm.framework.requester import requesterify
from idm.users.models import User

log = logging.getLogger(__name__)

COUNT_CLOSURES_SQL = """
SELECT COUNT(*)
FROM upravlyator_rolenodeclosure closure
JOIN upravlyator_rolenode node ON closure.child_id = node.id
WHERE node.system_id = %s;
"""

FIX_CLOSURE_TREE_SQL = """
WITH RECURSIVE rec AS (
  SELECT
    parent_id AS ancestor,
    id AS descendant,
    ARRAY[id] AS descendants
  FROM upravlyator_rolenode WHERE system_id=%s AND parent_id IS NOT NULL

  UNION ALL

  SELECT
    parent_id AS ancestor,
    descendant,
    descendants || ancestor AS descendants
  FROM upravlyator_rolenode
  JOIN rec ON id = ancestor
  WHERE not (ancestor = any(descendants)) AND system_id=%s AND parent_id IS NOT NULL
),
self_rec AS (
  SELECT id AS ancestor, id AS descendant, ARRAY[]::integer[] AS descendants
  FROM upravlyator_rolenode WHERE system_id=%s
),
all_rec AS (
    SELECT * FROM rec
    UNION ALL
    SELECT * FROM self_rec
)
INSERT INTO upravlyator_rolenodeclosure (depth, child_id, parent_id)
SELECT
  COALESCE(array_length(descendants, 1), 0), -- длина пустого массива - NULL
  descendant,
  ancestor
FROM all_rec
LEFT JOIN upravlyator_rolenodeclosure closure
    ON ancestor=closure.parent_id AND descendant=closure.child_id
WHERE closure.id IS NULL;
"""

FIX_PATHS_RECURSIVE_PART = """
WITH RECURSIVE rec AS (
  SELECT
    node.id,
    node.system_id,
    node.state,
    0::bigint as level,
    '/' AS slug_path,
    '/' AS value_path
  FROM upravlyator_rolenode node
  WHERE level = 0 AND system_id = %s

  UNION ALL

  SELECT
    node.id AS id,
    node.system_id,
    node.state,
    node.level AS level,
    rec.slug_path || node.slug || '/' AS slug_path,
    CASE node.level %% 2  -- отлавливать багу с форматированием было не очень весело
      WHEN 0 THEN rec.value_path || node.slug || '/'
      ELSE rec.value_path
    END AS value_path
  FROM upravlyator_rolenode node
  JOIN rec ON node.parent_id = rec.id
  WHERE node.level = rec.level + 1 AND node.system_id = %s
)
"""

FIX_PATHS_PART_1 = """
SELECT node.id, rec.id
FROM upravlyator_rolenode node 
JOIN rec ON node.slug_path = rec.slug_path AND node.system_id = rec.system_id AND node.id <> rec.id
WHERE  node.state = 'active' AND rec.state = 'active';
"""

FIX_PATHS_PART_2 = """
UPDATE upravlyator_rolenode node
SET slug_path = rec.slug_path, value_path = rec.value_path
FROM rec
WHERE node.id = rec.id AND node.system_id = %s
AND (node.slug_path <> rec.slug_path OR node.value_path <> rec.value_path)
RETURNING node.id;
"""


class Command(IdmBaseCommand):
    help = """Одноразовая команда, создаёт отсутствующие записи в таблице rolenodeclosure и чинит slug_path и value_path"""

    def add_arguments(self, parser):
        super(Command, self).add_arguments(parser)
        parser.add_argument(
            '--system', action='store', dest='system', type='system', required=True
        )

    def has_no_roles(self, rolenode):
        return not Role.objects.filter(
            node__in=rolenode.get_descendants(include_self=True),
            is_active=True
        ).exists()

    @transaction.atomic
    def idm_handle(self, *args, **options):
        system = options.get('system')

        # Находим недостающие записи и создаём их
        with connection.cursor() as cursor:
            cursor.execute(COUNT_CLOSURES_SQL, [system.pk])
            before = cursor.fetchone()[0]
            cursor.execute(FIX_CLOSURE_TREE_SQL, [system.pk]*3)
            cursor.execute(COUNT_CLOSURES_SQL, [system.pk])
            after = cursor.fetchone()[0]
            print('{} rolenode closure records were created'.format(after-before))

        # Если есть 2 узла с одинаковым slug_path, то удаляем тот, на котором нет ролей
        with connection.cursor() as cursor:
            cursor.execute(FIX_PATHS_RECURSIVE_PART + FIX_PATHS_PART_1, [system.pk] * 2)
            requester = requesterify(User.objects.get_idm_robot())
            for pks in cursor.fetchall():
                active_nodes = RoleNode.objects.filter(pk__in=pks, state='active')
                if len(active_nodes) == 2:
                    nodes_with_no_roles = list(filter(self.has_no_roles, active_nodes))
                    if nodes_with_no_roles:
                        nodes_with_no_roles[0].mark_depriving()
                        nodes_with_no_roles[0].deprive(requester)
                    else:
                        print('Both %d and %d rolenodes have roles' % pks)

        # Перепросчитываем slug_path и value_path у всех узлов в системе
        with connection.cursor() as cursor:
            cursor.execute(FIX_PATHS_RECURSIVE_PART + FIX_PATHS_PART_2, [system.pk] * 3)
            updated_nodes = [x[0] for x in cursor.fetchall()]
        print('Slug_path, value_path were recalculated for %s rolenodes' % len(updated_nodes))
