# -*- coding: utf-8 -*-


import json
from optparse import make_option

import pytz
from blackbox import JsonBlackbox as Blackbox
from django.conf import settings
from django.core.management.base import BaseCommand
from django.utils import timezone

from idm.core.constants.passport_login import CREATED_FLAG_REASON
from idm.core.models import UserPassportLogin
from idm.sync.passport import _get_current_ip

OUR_DELTA = 10.0


class Command(BaseCommand):
    help = """Простановка статусов создания паспортных логинов."""
    option_list = BaseCommand.option_list + (
        make_option(
            '--our_logins',
            help='Путь до json-файла с точно нашими логинами.'
        ),
    )

    def handle(self, *args, **options):
        our_logins_path = options.get('our_logins')
        if our_logins_path is not None:
            our_logins = set(json.load(open(our_logins_path)))
        else:
            our_logins = set()

        blackbox = Blackbox(settings.BLACKBOX_PASSPORT_URL)

        info = []
        for login in UserPassportLogin.objects.all():
            info.append((login.login, login.added))

        print('Getting account info from blackbox:')
        for i, item in enumerate(info):
            if i % 500 == 0:
                print('#{}/{}'.format(i, len(info)))
            login, addition = item
            response = blackbox.userinfo(
                login=login,
                userip=_get_current_ip(),
                dbfields=['account_info.reg_date.uid']
            )['users'][0]
            if 'dbfields' not in response:
                info[i] = (login, addition, None)
                continue
            date_string = response['dbfields']['account_info.reg_date.uid']
            registration = timezone.datetime.strptime(date_string, '%Y-%m-%d  %X')
            info[i] = (login, addition, registration)

        info = [
            (login, add, pytz.utc.localize(reg) - timezone.timedelta(hours=3))
            for (login, add, reg) in info
            if reg is not None
        ]
        deltas = {item[0]: (item[2] - item[1]).total_seconds() for item in info if item[0]}
        maybe_logins = {
            login
            for login, delta in deltas.items()
            if login not in our_logins
            and abs(delta) < OUR_DELTA
            and login.startswith('yndx-')
        }

        for_sure = (
            UserPassportLogin.objects
            .filter(login__in=our_logins)
            .update(created_by_idm=True, created_flag_reason=CREATED_FLAG_REASON.HISTORY)
        )
        predicted = (
            UserPassportLogin.objects
            .filter(login__in=maybe_logins)
            .update(created_by_idm=True, created_flag_reason=CREATED_FLAG_REASON.DELTA)
        )
        print('Logins marked as ours. For sure: {}, according to delta: {}'.format(for_sure, predicted))
