# -*- coding: utf-8 -*-


import logging
from optparse import make_option

from django.conf import settings
from idm.core.management.base import IdmBaseCommand

from idm.core.models import System
from idm.utils.lock import lock
from idm.core.tasks import SyncGroupMembershipSystemRelations

log = logging.getLogger(__name__)


class Command(IdmBaseCommand):
    help = '''Сверяем членства пользователей в группах для систем с политиками AWARE_OF_MEMBERSHIPS,
    создаем, разрешаем неконсистентности.'''
    intranet_only = True

    def add_arguments(self, parser):
        super(Command, self).add_arguments(parser)
        parser.add_argument(
            '--system', type='system', required=False
        )
        parser.add_argument(
            '--check-only', default=False, help='Только проверить расхождения, не разрешая их.'
        )
        parser.add_argument(
            '--resolve-only', default=False, help='Только разрешить расхождения, не создавая новых'
        )

    def process_system(self, system, check_only, resolve_only):
        log.info('Start processing GroupMembershipInconsistencies for %s (check_only: %s, resolve_only: %s)',
                 system.slug, check_only, resolve_only)
        SyncGroupMembershipSystemRelations.apply_async(
            kwargs={'system_id': system.pk,
                    'check_only': check_only,
                    'resolve_only': resolve_only,
                    },
            countdown=settings.IDM_PLUGIN_TASK_COUNTDOWN
        )
        log.info('GroupMembershipInconsistencies for %s have been processed', system.slug)

    def idm_handle(self, *args, **options):
        system = options.get('system')
        check_only = options.get('check_only', False)
        resolve_only = options.get('resolve_only', False)
        if system:
            self.process_system(system, check_only, resolve_only)
        else:
            log.info('Start GroupMembershipInconsistencies process')
            systems = System.objects.get_systems_with_group_sync_policy()
            for system in systems:
                self.process_system(system, check_only, resolve_only)
            log.info('GroupMembershipInconsistencies process finished')
