# -*- coding: utf-8 -*-


import logging

from django.utils.encoding import smart_str

from idm.core.management.base import IdmBaseCommand
from idm.core.models import System
from idm.users.models import User


def _check_workflow(role, log):
    """ Прогоняет workflow для роли role, проверяет изменились ли подтверждающие, и рассылает им письма.

    Возвращает True, если аппруверы изменились, и False если всё осталось как есть.
    """
    workflow_data = role.run_workflow()

    log.info('Getting approvers for user %(user)s role %(role)s' % dict(user=role.user, role=role))
    new_approvers = workflow_data.get('approvers')

    log.info('Compare old and new approvers for user %(user)s role %(role)s'
             % dict(user=role.user, role=role))
    old_approves = role.get_open_request_approves()

    def _get_new_approvers_set(approvers):
        """Возвращает множество c отсортированными кортежами подтверждающих"""
        approvers_set = set()
        if approvers:
            for approver in approvers:
                approvers_set.add(tuple(sorted([login for login in approver])))
        return approvers_set

    def _get_old_approvers_set(approves):
        """Возвращает множество c отсортированными кортежами по аппрувреквестам"""
        approvers_set = set()
        for approve in approves:
            approvers_set.add(tuple(sorted([a.approver.username for a in approve.requests.all()])))
        return approvers_set

    new_approvers_set = _get_new_approvers_set(new_approvers)
    old_approvers_set = _get_old_approvers_set(old_approves)

    if new_approvers_set == old_approvers_set:  # ничего не меняем
        log.info('Old and New approvers are equal for user %(user)s role %(role)s'
                 % dict(user=role.user, role=role))
        return False

    log.info('Close old role request')
    try:  # экшена может не быть, если это первый запрос роли и аппруверов не было
        action = role.requests.filter(is_done=False).latest('added').action
    except:
        action = None
    role.requests.all().update(is_done=True)

    if new_approvers:  # если роль требует подтверждения, создадим для него Approve
        # запрос роли остался прежним, поменялись только необходимые подтверждения,
        # значит новый экшн создавать нет нужды
        log.info('Add new role request')
        role_request = role.requests.create(action=action)

        log.info('Create new approvers and requests for user %(user)s role %(role)s'
                 % dict(user=role.user, role=role))
        for approver in new_approvers:
            approve = role_request.approves.create(role=role, requester=role.user)

            for login in approver:
                try:
                    user = User.objects.users().get(username=login)
                    approve.requests.create(approver=user)
                except User.DoesNotExist:
                    log.exception('Approver %(login)s not found' % login)
                    continue

    else:  # или просто отдаем пользователю роль, если подтверждений не требуется
        log.info('Approves are not nesessary for user %(user)s role %(role)s'
                 % dict(user=role.user, role=role))
        if role.state == 'requested':
            role.set_state('approved')
        else:
            role.set_state('granted')

    return True


class Command(IdmBaseCommand):
    help = 'Проверяет наличие изменений в регламенте подтверждения ролей и создает новые аппрувы по необходимости.'

    def add_arguments(self, parser):
        super().add_arguments(parser)

        parser.add_argument(
            '--system',
            type='system',
            help='Обрабатывать только конкретную систему.'
        ),
        parser.add_argument(
            '--force',
            action='store_true',
            dest='force',
            default=False,
            help='Не смотреть на дату изменения workflow.'
        )

    def idm_handle(self, *args, **options):
        try:
            if options.get('system'):
                systems = System.objects.filter(pk=options['system'].pk, is_broken=False)
            else:
                systems = System.objects.get_operational()

            for system in systems:
                log = logging.getLogger(smart_str(__name__ + '.' + system.slug))

                log.info('Trying to restart workflow')

                roles = system.roles.filter(state__in=('requested', 'rerequested', 'review_request'))
                if not options.get('force'):
                    workflow_change_date = system.get_workflow_changed_date()
                    log.info(
                        'Getting all roles requested before workflow was changed {date}'.format(
                            date=workflow_change_date
                        )
                    )
                    roles = roles.filter(updated__lte=workflow_change_date)
                else:
                    log.info('Getting all roles in requested state')

                changed = unchanged = failed = 0
                for role in roles:
                    try:
                        if _check_workflow(role, log):
                            changed += 1
                        else:
                            unchanged += 1
                    except RuntimeError:
                        log.exception('Error during _check_workflow.')
                        failed += 1

                log.info('changed={changed}, unchanged={unchanged}, failed={failed}'.format(**locals()))

        except Exception:
            log = logging.getLogger(__name__)
            log.exception('error during command execution')
            raise
