import datetime
import json
import logging

from constance import config
from django.conf import settings
from django.utils import timezone

from idm.core.management.base import IdmBaseCommand
from idm.core.models import System, Role
from idm.monitorings.metric import ReviewRolesThresholdExceededMetric
from idm.utils import calendar

log = logging.getLogger(__name__)


class Command(IdmBaseCommand):
    help = """Регулярный пересмотр ролей пользователей."""

    intranet_only = True

    @staticmethod
    def add_to_cache_for_monitoring(data):
        ReviewRolesThresholdExceededMetric.set(data)

    @staticmethod
    def get_threshold(system):
        data = json.loads(config.IDM_REVIEW_ROLES_PER_SYSTEM_THRESHOLD)
        return data.get(system.slug, config.IDM_REVIEW_ROLES_PER_SYSTEM_DEFAULT_THRESHOLD)

    def add_arguments(self, parser):
        super().add_arguments(parser)

        parser.add_argument(
            '--system',
            type='system',
            help='Обрабатывать только конкретную систему.'
        )
        parser.add_argument(
            '--days',
            type='int',
            help='Обрабатывать роли, истёкшие не позже, чем N дней назад.',
            default=0
        )
        parser.add_argument(
            '--force',
            action='store_true',
            dest='force',
            default=False,
            help='Не смотреть на дату изменения workflow.'
        )

    def idm_handle(self, *args, **options):
        days = options.get('days')
        since_date = timezone.now() - datetime.timedelta(days=days)

        try:
            systems = System.objects.get_operational().filter(has_review=True)

            if options.get('system'):
                systems = systems.filter(pk=options['system'].pk)
            else:
                systems = systems.filter(is_active=True)

            roles_to_review = {}
            systems_exceeded = {}

            for system in systems:
                system_roles_to_review = Role.objects.get_roles_for_review(system, since_date)
                count = system_roles_to_review.count()
                if system.metainfo.roles_failed_on_last_review and not count:
                    system.metainfo.roles_failed_on_last_review = None
                    system.metainfo.save(update_fields=['roles_failed_on_last_review'])
                threshold = self.get_threshold(system)
                if count > threshold:
                    systems_exceeded[system.slug] = count
                else:
                    roles_to_review[system] = system_roles_to_review

            self.add_to_cache_for_monitoring(systems_exceeded)

            expiration_date = calendar.get_next_workday(
                timezone.now() + timezone.timedelta(days=settings.REQUESTED_ROLE_TTL), datetime.time(15),
            )
            for system in roles_to_review:
                system_failed_roles = roles_to_review[system].review_roles(system, expiration_date)
                if system_failed_roles:
                    system.metainfo.roles_failed_on_last_review = system_failed_roles
                elif system.metainfo.roles_failed_on_last_review:
                    system.metainfo.roles_failed_on_last_review = None
                else:
                    continue
                system.metainfo.save(update_fields=['roles_failed_on_last_review'])

        except Exception:
            log.exception('error during command execution')
            raise
