# coding: utf-8
from __future__ import print_function

import logging

from idm.core.constants.system import IDM_SYSTEM_TREE
from idm.core.workflow.exceptions import RoleRequestError, RoleAlreadyExistsError
from idm.core.management.base import IdmBaseCommand
from idm.core.models import System, SystemRolePush, Role
from idm.core.workflow.common.subject import subjectify


log = logging.getLogger(__name__)


class Command(IdmBaseCommand):
    help = 'Push all roles after system creation'
    intranet_only = True

    def idm_handle(self, *args, **options):
        self_system = System.objects.get_idm_system()
        self_system.fetch_actual_workflow()
        system_on_node = self_system.nodes.get(slug_path=IDM_SYSTEM_TREE.SYSTEM_ON_PATH)
        for push in SystemRolePush.objects.select_related('user', 'group', 'system__creator'):
            user, group, system = push.user, push.group, push.system
            creator = system.creator
            system_node = system_on_node.children.active().get(slug=system.slug)
            node = system_node.children.get(slug='role').children.active().get(slug=push.role_slug)
            name, ident = ('user', user.username) if user else ('group', group.external_id)
            request_description = 'role %s for %s %s while creating system %s' % (
                node.name_en, name, ident, system.slug,
            )
            log.info('Trying to request %s', request_description)
            subject = subjectify(user or group)
            ok = False
            try:
                if subject.is_active():
                    Role.objects.request_role(
                        creator,
                        subject,
                        self_system,
                        'New system {}'.format(node.name_en),
                        node.data,
                        fields_data={'scope': '/'}
                    )
            except RoleAlreadyExistsError:
                ok = True
            except RoleRequestError:
                log.exception('Failed to request %s', request_description)
            except Exception:
                log.exception('Unknown error while requesting %s', request_description)
            else:
                ok = True

            if ok:
                push.delete()
