# -*- coding: utf-8 -*-
# Generated by Django 1.9.13 on 2019-07-30 10:38


from django.db import migrations, connection, DatabaseError, IntegrityError

from idm.utils.sql import generate_index_for_roles


def bulk_update(apps, is_returnable):
    Role = apps.get_model('core', 'Role')
    PAGE_SIZE = 10000
    RETURNABLE_STATES = {
        'requested', 'approved', 'granted', 'review_request',
        'need_request', 'rerequested', 'sent',
        'imported', 'onhold', 'created'
    }

    if is_returnable:
        query = Role.objects.filter(state__in=RETURNABLE_STATES)
    else:
        query = Role.objects.exclude(state__in=RETURNABLE_STATES)

    offset = 0

    while True:
        get_query = query.filter(pk__gt=offset).order_by('pk').values_list('pk',flat=True)[:PAGE_SIZE]
        pks = list(get_query)
        if len(pks) == 0:
            break

        old_offset = offset
        offset = pks[-1]

        update_query = query.filter(pk__gt=old_offset, pk__lte=offset)
        try:
            update_query.update(is_returnable=is_returnable)
        except DatabaseError as err:
            if not is_returnable:
                print('Somehow we could not set is_returnable=False for roles with %d<pk<=%d (%s)' % (
                    query[0].pk,
                    offset,
                    err
                ))
                raise err
            else:
                for role in update_query:
                    try:
                        role.is_returnable = True
                        role.save(update_fields=['is_returnable'])
                    except IntegrityError:
                        print('You should deprive role with pk=%d' % role.pk)


def set_is_returnable_field(apps):
    bulk_update(apps, is_returnable=False)
    bulk_update(apps, is_returnable=True)


def process_index(apps, schema_editor, direction):
    try:
        with connection.cursor() as cursor:
            for forwards, backwards in generate_index_for_roles(concurrently=True):
                sql = forwards if direction == 'forward' else backwards
                cursor.execute(sql)
    except DatabaseError as err:
        print('Could not complete migration: %s' % err)
        print('Dropping created indices...')
        if direction == 'forward':
            backward(apps, schema_editor)


def forward(apps, schema_editor):
    process_index(apps, schema_editor, direction='forward')
    try:
        set_is_returnable_field(apps)
    except Exception as err:
        print('Some error occured: %s' % err)
        print('Dropping index...')
        backward(apps, schema_editor)



def backward(apps, schema_editor):
    process_index(apps, schema_editor, direction='backward')


class Migration(migrations.Migration):
    initial = True

    dependencies = [
        ('core', '0002_foreign_keys'),
    ]

    operations = [
        migrations.RunSQL('''
            CREATE FUNCTION count_estimate(query text) RETURNS INTEGER AS
            $func$
            DECLARE
                rec   record;
                ROWS  INTEGER;
            BEGIN
                FOR rec IN EXECUTE 'EXPLAIN ' || query LOOP
                    ROWS := SUBSTRING(rec."QUERY PLAN" FROM ' rows=([[:digit:]]+)');
                    EXIT WHEN ROWS IS NOT NULL;
                END LOOP;
             
                RETURN ROWS;
            END
            $func$ LANGUAGE plpgsql;
            ''', 'DROP FUNCTION count_estimate(query text);'
        ),
        migrations.RunSQL('''
            CREATE UNIQUE INDEX idm_rolenode_slugpath_uniq_where
            ON upravlyator_rolenode (system_id, slug_path) WHERE state='active';
        ''', 'DROP INDEX idm_rolenode_slugpath_uniq_where'),
        migrations.RunSQL('''
            CREATE INDEX rn_closure_depth ON "upravlyator_rolenodeclosure" (parent_id, depth)
        ''', 'DROP INDEX rn_closure_depth'),
        migrations.RunSQL('''
            CREATE UNIQUE INDEX upravlyator_noderesponsibility_uniq
            ON upravlyator_noderesponsibility (node_id, user_id) WHERE is_active=TRUE
        ''', 'DROP INDEX CONCURRENTLY upravlyator_noderesponsibility_uniq'),
    ]
