import random
from hashlib import md5

from django.conf import settings
from django.db import models, transaction
from django.utils.translation import ugettext_lazy as _, override
from idm.notification.utils import send_notification

from idm.api.exceptions import BadRequest, Forbidden
from idm.core.constants.approverequest import APPROVEREQUEST_DECISION
from idm.core.constants.role import ROLE_STATE
from idm.core.constants.workflow import DEFAULT_PRIORITY
from idm.core.workflow.exceptions import InactiveSystemError, BrokenSystemError
from idm.core.querysets.approverequest import ApproveRequestManager
from idm.framework.fields import StrictForeignKey
from idm.framework.requester import requesterify


class ApproveRequest(models.Model):
    """Запрос на подтверждение роли конкретному адресату.

    Каждый запрос имеет свой уникальный хэш, который должен
    быть использован в ссылке для подтверждения.
    """
    DECISIONS = tuple(APPROVEREQUEST_DECISION.TRANSLATIONS.items())

    approve = StrictForeignKey('core.Approve', verbose_name=_('Подтверждение'), related_name='requests',
                                null=False, blank=False, on_delete=models.CASCADE)
    approver = StrictForeignKey('users.User', verbose_name=_('Подтверждающий'),
                                 related_name='approve_requests', null=False, blank=False, on_delete=models.CASCADE)
    decision = models.CharField(
        max_length=10, verbose_name=_('Решение'), choices=DECISIONS, blank=True,
        default=APPROVEREQUEST_DECISION.NOT_DECIDED
    )
    notify = models.NullBooleanField(_('Оповещать'), default=None)
    hash = models.CharField(_('Уникальный хэш'), default=None, unique=True, max_length=32)
    parent = StrictForeignKey(
        'self', null=True, blank=True, verbose_name=_('Ссылка на родительский ApproveRequest'), related_name='children',
        on_delete=models.CASCADE,
    )
    added = models.DateTimeField(auto_now_add=True, editable=False, null=True)
    updated = models.DateTimeField(auto_now=True, editable=False, null=True, db_index=True)
    # приоритет подтверждающего
    priority = models.PositiveSmallIntegerField(default=DEFAULT_PRIORITY)
    is_notification_sent = models.BooleanField(default=False, verbose_name=_('Посылали ли письмо для подтверждения'))

    objects = ApproveRequestManager()

    class Meta:
        verbose_name = _('Запрос на подтверждение')
        verbose_name_plural = _('Запросы на подтверждение')
        unique_together = (
            ('parent', 'approver', 'decision'),
            ('parent', 'approver'),
        )
        db_table = 'upravlyator_approverequest'
        ordering = ('id',)

    def __str__(self):
        return '%d, to=%s, notify=%s' % (self.pk, self.approver.username, self.notify)

    @property
    def approved(self):
        if self.decision in (APPROVEREQUEST_DECISION.NOT_DECIDED, APPROVEREQUEST_DECISION.IGNORE):
            return None
        else:
            return self.decision == APPROVEREQUEST_DECISION.APPROVE

    def save(self, *args, **kwargs):
        if self.hash is None:
            self.hash = md5(
                (self.approver.email + str(random.random()) + settings.SECRET_KEY).encode('utf-8')
            ).hexdigest()
        return super(ApproveRequest, self).save(*args, **kwargs)

    def send_email(self, comment):
        lang_ui = self.approver.lang_ui
        if self.decision != APPROVEREQUEST_DECISION.NOT_DECIDED:
            # подтверждение уже получено - ничего отправлять не будем
            return

        with override(lang_ui):
            role_request = self.approve.role_request
            requester = role_request.requester
            role = role_request.role

            subject_template = _('Подтверждение роли. %s.')
            templates = [
                'emails/approve_role_%s.html' % role.system.slug,
                'emails/approve_role_%s.txt' % role.system.slug,
                'emails/approve_role.html',
                'emails/approve_role.txt',
            ]

            subject = subject_template % role.system.get_name()
            context = {
                'requester': requester,
                'user': role.user,
                'group': role.group,
                'role': role,
                'comment': comment,
                'for_email': True,
            }
            send_notification(subject, templates, [self.approver], context,
                              headers={'Reply-To': requester.email} if requester and requester.email else None)

    def set_decided(self, requester, decision):
        assert decision in dict(self.DECISIONS).keys()
        if self.approver_id != requester.impersonated.id:
            raise Forbidden(_('Вы не можете совершать действия с чужим подтверждением роли'))
        role_request = self.approve.role_request

        system = role_request.role.system
        if system.is_broken:
            raise BrokenSystemError(_('Нельзя подтвердить/отклонить роль в сломанной системе'))
        elif not system.is_active:
            raise InactiveSystemError(_('Нельзя подтвердить/отклонить роль в неактивной системе'))

        role_request.role.lock(retry=True)
        self.decision = decision
        approve_requests_to_set_decided = (
            role_request.get_approve_requests().filter(
                approver=requester.impersonated, decision=APPROVEREQUEST_DECISION.NOT_DECIDED
            )
        )
        approve_requests_to_set_decided.update(decision=decision)
        role_request.update_approves()
        return True

    @transaction.atomic
    def set_approved(self, requester, comment=None, from_api=False):
        """
        Подтвердить запрос на роль
        """
        if self.decision != APPROVEREQUEST_DECISION.NOT_DECIDED:
            return
        requester = requesterify(requester)
        self.set_decided(requester, APPROVEREQUEST_DECISION.APPROVE)
        role = self.approve.role_request.role
        if role.state == 'granted':
            return  # роль уже подтверждена, больше ничего не делаем
        if role.is_active:
            role.set_state('granted', requester=requester, approve_request=self, comment=comment, from_api=from_api)
        else:
            role.set_state('approved', requester=requester, approve_request=self, comment=comment, from_api=from_api)

    @transaction.atomic
    def set_declined(self, requester, comment=None, from_api=False):
        """
        Отказать в запросе на роль
        """
        if self.decision != APPROVEREQUEST_DECISION.NOT_DECIDED:
            return
        requester = requesterify(requester)
        self.set_decided(requester, APPROVEREQUEST_DECISION.DECLINE)
        role = self.approve.role_request.role
        # если роль уже подтверждена, то отзываем
        if role.is_active:
            role.set_state(
                ROLE_STATE.DEPRIVING, requester=requester,
                approve_request=self, transition='deprive',
                comment=comment, from_api=from_api,
            )

        # если роль еще не подтверждена — отклоняем
        elif role.state == 'requested':
            role.set_state(
                'declined', requester=requester,
                approve_request=self, comment=comment,
                from_api=from_api,
            )

    @transaction.atomic
    def set_ignored(self, requester, comment=None, from_api=False):
        """
        Отморозиться
        """
        if self.decision != APPROVEREQUEST_DECISION.NOT_DECIDED:
            return
        requester = requesterify(requester)
        self.set_decided(requester, APPROVEREQUEST_DECISION.IGNORE)
        role = self.approve.role_request.role
        role_state = role.state
        role.deprive_or_decline(
            requester, comment=comment,
            approve_request=self, bypass_checks=True,
            from_api=from_api,
        )
        role.refresh_from_db()
        if role_state == role.state:
            ApproveRequest.objects.recalculate_main_priority(self.approve)

    def start_discussion(self, requester, **kwargs):
        role = self.approve.role_request.role
        if self.approver_id != requester.impersonated.id:
            raise Forbidden(message=_('Вы не можете совершать действия с чужим подтверждением роли'))
        if role.state not in ROLE_STATE.STATES_FOR_DISCUSSION:
            raise BadRequest(message=_('Эта роль не для обсуждения'))
        if role.parent_id is not None:
            raise BadRequest(message=_('Связанные роли обсуждать нельзя'))

        issue = self.approve.role_request.get_or_create_st_issue(requester.impersonated)
        issue_url = settings.IDM_ST_BASE_URL + issue
        return issue_url

    def as_approver(self):
        from idm.core.workflow.plain.approver import Approver
        return Approver(self.approver, priority=self.priority, notify=self.notify)
