import logging

from django.db import models
from django.dispatch import receiver
from django.utils.translation import ugettext_lazy as _

from idm.core import canonical, signals
from idm.core.querysets.noderesponsibility import NodeResponsibilityManager
from idm.framework.fields import StrictForeignKey
from idm.framework.mixins import RefreshFromDbWithoutRelatedMixin

log = logging.getLogger(__name__)


class NodeResponsibility(RefreshFromDbWithoutRelatedMixin, models.Model):
    created_at = models.DateTimeField(_('Дата создания'), auto_now_add=True, editable=False, null=True)
    updated_at = models.DateTimeField(_('Дата обновления'), auto_now=True, editable=False, null=True)
    removed_at = models.DateTimeField(_('Дата удаления'), editable=False, null=True)

    node = StrictForeignKey('core.RoleNode', null=False, blank=False, verbose_name=_('Узел дерева ролей'),
                             related_name='responsibilities', on_delete=models.CASCADE)
    is_active = models.BooleanField(_('Действующий ответственный'), default=True)
    user = StrictForeignKey('users.User', null=False, blank=False, verbose_name=_('Пользователь'),
                             related_name='noderesponsibilities', on_delete=models.CASCADE)
    notify = models.BooleanField(_('Оповещать при запросе ролей на узел'), default=False)
    objects = NodeResponsibilityManager()

    class Meta:
        verbose_name = _('Ответственный за узел')
        verbose_name_plural = _('Ответственные за узлы')
        db_table = 'upravlyator_noderesponsibility'

    def __str__(self):
        return '%(user)s (%(active)s, notify=%(notify)s)' % {
            'user': self.user.username,
            'active': 'on' if self.is_active else 'off',
            'notify': 'yes' if self.notify else 'no',
        }

    def as_canonical(self):
        return canonical.CanonicalResponsibility(
            user=self.user,
            notify=self.notify
        )

    def as_api(self):
        return {
            'username': self.user.username,
            'notify': self.notify,
        }


@receiver(signals.responsibilities_added)
def grant_internal_role_on_responsibility_addition(role_node, responsibilities, **kwargs):
    from idm.permissions.utils import add_perms_by_role
    for responsibility in responsibilities:
        add_perms_by_role('roles_manage', responsibility.user, role_node.system, role_node)


@receiver(signals.responsibilities_removed)
def deprive_internal_role_on_responsibility_removal(role_node, responsibilities, **kwargs):
    from idm.permissions.utils import remove_perms_by_role, RoleNotFound
    for responsibility in responsibilities:
        try:
            remove_perms_by_role('roles_manage', responsibility.user, role_node.system, role_node)
        except RoleNotFound:
            log.exception('No matching internal role for responsibility %d of node %s',
                          responsibility.pk, role_node.slug_path)
