from __future__ import annotations

import collections
import datetime
import functools
import logging
import sys
import time
import traceback
from typing import Iterable, Optional, List, Dict, Any, Collection

import constance
import waffle
from django.conf import settings
from django.db import models, transaction, OperationalError
from django.db.models import NOT_PROVIDED, Q
from django.dispatch import receiver
from django.utils import timezone
from django.utils.encoding import force_text
from django.utils.module_loading import import_string
from django.utils.translation import ugettext_lazy as _, ugettext_noop, override
from ylog.context import log_context

from idm.core import signals, canonical
from idm.core.conflicts import find_conflicts
from idm.core.constants.action import ACTION_DATA_KEYS, ACTION, MAX_ACTION_ERROR_LENGTH
from idm.core.constants.affiliation import AFFILIATION
from idm.core.constants.approverequest import APPROVEREQUEST_DECISION
from idm.core.constants.errors import ERROR_LEN
from idm.core.constants.groupmembership import GROUPMEMBERSHIP_STATE
from idm.core.constants.role import ROLE_STATE
from idm.core.constants.rolefield import FIELD_TYPE
from idm.core.constants.system import SYSTEM_GROUP_POLICY, SYSTEM_INCONSISTENCY_POLICY
from idm.core.constants.workflow import RUN_REASON, REQUEST_TYPE
from idm.core.exceptions import (
    RoleStateSwitchError,
    MembershipPassportLoginError,
    MultiplePassportLoginUsersError,
)
from idm.core.querysets.role import RoleManager
from idm.core.utils import is_public_role, humanize_role
from idm.core.workflow.common.subject import subjectify
from idm.core.workflow.exceptions import (
    RoleAlreadyExistsError,
    BrokenSystemError,
    Forbidden,
    NotifyReferenceRolesValidationError,
    SilentReferenceRolesValidationError,
    WorkflowError,
    NoMobilePhoneError,
)
from idm.core.workflow.plain.approver import combine_notifications, AnyApprover
from idm.core.workflow.shortcuts import workflow
from idm.framework.fields import NullJSONField as JSONField
from idm.framework.fields import StrictForeignKey
from idm.framework.mixins import RefreshFromDbWithoutRelatedMixin
from idm.framework.requester import requesterify, Requester
from idm.framework.task import DelayingError
from idm.framework.utils import add_to_instance_cache
from idm.monitorings.metric import OverlengthedRefRoleChainMetric
from idm.notification.utils import send_notification, report_problem
from idm.permissions import shortcuts as permissions_shortcuts
from idm.users.constants.user import USER_TYPES
from idm.users.models import User, GroupMembership
from idm.utils import min_with_none, calendar, events
from idm.utils.actions import format_error_traceback
from idm.utils.sql import wrap_unique_violation

log = logging.getLogger(__name__)


class Role(RefreshFromDbWithoutRelatedMixin, models.Model):
    node = StrictForeignKey('RoleNode', blank=False, null=False, verbose_name=_('Узел дерева ролей'),
                            related_name='roles', on_delete=models.CASCADE)
    system = StrictForeignKey(
        'core.System', related_name='roles', null=False, blank=False, verbose_name=_('Система'),
        on_delete=models.CASCADE,
    )
    added = models.DateTimeField(auto_now_add=True)
    updated = models.DateTimeField(auto_now_add=True, db_index=True)
    is_public = models.NullBooleanField(null=True, blank=True, default=None)
    is_active = models.BooleanField(default=False, db_index=True)
    is_returnable = models.BooleanField(default=True)
    state = models.TextField(
        verbose_name=_('Состояние'),
        choices=list(ROLE_STATE.STATE_CHOICES.items()),
        default='created',
        db_index=True,
    )

    user = StrictForeignKey(
        'users.User', null=True, blank=True, related_name='roles', verbose_name=_('Пользователь'),
        on_delete=models.CASCADE,
    )
    group = StrictForeignKey(
        'users.Group', null=True, blank=True, related_name='roles', verbose_name=_('Группа'), on_delete=models.CASCADE,
    )

    system_specific = JSONField(blank=True, null=True, db_index=True)
    expire_at = models.DateTimeField(blank=True, null=True, db_index=True)
    review_at = models.DateTimeField(blank=True, null=True, default=None)
    fields_data = JSONField(blank=True, null=True, db_index=True)
    request_fields = JSONField(blank=True, null=True)
    inconsistency = models.OneToOneField(
        'Inconsistency',
        on_delete=models.SET_NULL,
        verbose_name=_('Расхождение, по которому заведена эта роль'),
        related_name='role',
        null=True
    )
    last_request = models.OneToOneField(
        'RoleRequest',
        on_delete=models.SET_NULL,
        null=True,
        verbose_name=_('Последний запрос'),
        related_name='+',
    )
    parent = StrictForeignKey('self', null=True, blank=True, related_name='refs', on_delete=models.CASCADE)
    ad_groups = JSONField(blank=True, null=True)
    email_cc = JSONField(blank=True, null=True)
    ref_roles = JSONField(blank=True, null=True)
    ttl_date = models.DateTimeField(null=True)
    ttl_days = models.IntegerField(null=True)
    review_date = models.DateTimeField(null=True, default=None)
    review_days = models.IntegerField(null=True, default=None)
    send_sms = models.BooleanField(default=False)
    # не отправлять уведомлений при выдаче/отзыве роли
    no_email = models.BooleanField(default=False)
    granted_at = models.DateTimeField(blank=True, null=True)
    depriving_at = models.DateTimeField(blank=True, null=True)
    deprive_comment = models.TextField(blank=True, default='', verbose_name=_('Комментарий к отзыву роли'))
    depriver = StrictForeignKey(
        'users.User',
        null=True,
        blank=True,
        related_name='depriving_roles',
        verbose_name=_('Пользователь, отозвавший роль'),
        on_delete=models.CASCADE,
    )
    organization = StrictForeignKey(
        'users.Organization',
        null=True,
        blank=True,
        related_name='roles',
        verbose_name=_('Organization'),
        on_delete=models.CASCADE,
    )
    with_inheritance = models.BooleanField(default=True, verbose_name=_('Будет ли роль выдана вложенным департаментам'))
    with_external = models.BooleanField(default=True, verbose_name=_('Будет ли роль выдана внешним'))
    with_robots = models.BooleanField(default=True, verbose_name=_('Будет ли роль выдана роботам'))
    without_hold = models.BooleanField(default=False, verbose_name=_(
        'Будут ли персональные роли мгновенно отозваны при выходе из группы'))

    objects = RoleManager()

    class Meta:
        ordering = ('-updated', '-pk')
        index_together = (
            # запросы по пользователю
            ('is_public', 'system', 'node', 'user', 'group', 'updated'),
            ('is_public', 'is_active', 'state', 'system', 'node', 'user', 'group', 'updated'),
            # запросы по системе
            ('is_public', 'system', 'updated'),
            ('is_public', 'is_active', 'state', 'system', 'updated'),
            ('is_public', 'system', 'node', 'user', 'updated'),
            ('is_public', 'is_active', 'state', 'system', 'node', 'user', 'updated'),
            # запросы по группе
            ('is_public', 'group', 'updated'),
            ('is_public', 'is_active', 'state', 'group', 'updated'),
            # запросы а-ля вьюеры
            ('system', 'id'),
            ('is_active', 'system', 'id'),
            ('is_active', 'state', 'system', 'id'),
            # запросы для stat
            ('is_public', 'parent', 'state', 'system', 'updated'),
            ('is_public', 'parent', 'state', 'system'),
            # ускорение для request_applicable_personal_roles
            ('user', 'parent', 'is_active', 'state'),
        )
        db_table = 'upravlyator_role'

    def __repr__(self):
        if self.is_group_role():
            subject_repr = 'group=%s,%d' % (self.group.slug, self.group.external_id)
        else:
            subject_repr = 'user=%s' % self.user.username
        result = force_text('<Role: id=%s, system=%s, %s, role=%s, state=%s, fields_data=%s>' % (
            self.pk, self.system.slug, subject_repr, self.node.data, self.state, self.fields_data
        ))
        return result

    def __str__(self):
        result = _('Роль "%(path)s" %(subject)s в системе %(system)s') % {
            'path': self.node.value_path,
            'system': self.system.get_name(lang='ru'),
            'subject': self.get_subject().inflect('кого-чего')
        }
        return result

    def get_absolute_url(self):
        return '%s#role=%s' % (self.system.get_absolute_url(), self.id)

    def get_url(self):
        return '%s%s' % (settings.IDM_BASE_URL, self.get_absolute_url())

    def save(self, *args, **kwargs):
        # запрещаем обновлять по умолчанию все поля роли
        if self.pk and not kwargs.get('update_fields'):
            raise RuntimeError('Role object cannot be saved without `update_fields` argument')
        with wrap_unique_violation(self):
            return super(Role, self).save(*args, **kwargs)

    def get_subject(self):
        return subjectify(self)

    def get_personal_roles(self):
        if self.is_group_role():
            return Role.objects.filter(parent=self)
        else:
            raise ValueError('Cannot get personal roles of a user role')

    def is_user_role(self) -> bool:
        return self.user_id is not None

    def is_group_role(self):
        return self.group_id is not None

    def is_referrable_group_role(self):
        """ Роль является групповой, и у неё могут быть персональные дочерние роли """
        return self.is_group_role() and self.system.group_policy == 'unaware'

    def is_referrable(self):
        """У роли есть или потенциально должны/могут быть дочерние"""
        return self.is_referrable_group_role() or self.ref_roles

    def is_pushable(self):
        """Нужно ли оповещать систему о выдаче данной роли"""
        if self.system.is_group_aware():
            # TODO: в следующем релизе заменить это на result=True
            # В переходный период, пока персональные роли не удалены, они не должны стать pushable
            result = self.is_group_role() or self.is_user_role() and not self.is_personal_role()
        else:
            result = self.is_user_role()
        return result

    def is_personal_role(self):
        return self.is_user_role() and self.parent_id is not None and self.parent.is_group_role()

    def is_public_role(self):
        return is_public_role(self)

    def is_rerunnable(self):
        return self.is_returnable and (self.parent_id is None or self.parent.user_id is not None)

    def is_subscribable(self, ignore_state=False):
        if not ignore_state and not self.is_active:
            return False
        if self.system.slug in settings.IDM_SID67_EXCLUDED_SYSTEMS:
            return False
        for system_slug, slug_paths in settings.IDM_SID67_EXCLUDED_NODES.items():
            if self.system.slug == system_slug and self.node.slug_path in slug_paths:
                return False
        return True

    def set_raw_state(self, state: str, **kwargs):
        """Устанавливает новое состояние роли пользователя"""
        now = timezone.now()

        kwargs['state'] = state
        kwargs['is_returnable'] = state in ROLE_STATE.RETURNABLE_STATES
        kwargs['is_active'] = state in ROLE_STATE.ACTIVE_STATES
        kwargs.setdefault('updated', now)
        if state == ROLE_STATE.GRANTED:
            kwargs.setdefault('granted_at', now)

        if self.state not in ROLE_STATE.RETURNABLE_STATES and state in ROLE_STATE.RETURNABLE_STATES:
            Role.objects.check_role_uniqueness(self)

        for key, value in kwargs.items():
            setattr(self, key, value)

        self.save(update_fields=list(kwargs.keys()))

    def create_action(
        self,
        action: str,
        comment: str = None,
        error: str = '',
        requester: Optional[Requester] = NOT_PROVIDED,
    ) -> 'Action':
        impersonated = impersonator = None
        if requester is not NOT_PROVIDED and requester is not None:
            impersonated = requester.impersonated
            impersonator = requester.impersonator

        return self.actions.create(
            user_id=self.user_id,
            group_id=self.group_id,
            requester=impersonated,
            impersonator=impersonator,
            action=action,
            data={'comment': force_text(comment)},
            error=error[:ERROR_LEN],
            system=self.system,
        )

    def set_state(
            self,
            value: str,
            transition: str = None,
            from_any_state: bool = False,
            requester: Requester = None,
            action_data: Dict[str, Any] = None,
            comment: str = None,
            error: str = '',
            approve_request: "ApproveRequest" = None,
            transfer: "Transfer" = None,
            passport_login_created: bool = False,
            parent_action: "Action" = None,
            inconsistency: "Inconsistency" = None,
            force: bool = False,
            run_workflow: bool = None,
            ignore_parent_check: bool = False,
            with_push: bool = True,
            expiration_date: datetime.datetime = None,
            from_api: bool = False,
    ) -> "Action":
        """Устанавливает новое состояние роли пользователя value, requester - тот кто изменил состояние.
        При аппруве каждым из пользователей вызывается этот метод,
        а он уже определяет все ли проаппрувили и отправлять ли роль в систему"""
        # проверим, что система не сломана
        if self.system.is_broken and (self.state, value) not in ROLE_STATE.INNER_TRANSITIONS:
            raise BrokenSystemError('System %s is broken. Role state cannot be changed.' % self.system.slug)

        _ = ugettext_noop

        def create_action(action, robot_comment=None, _requester=NOT_PROVIDED):
            data = {}
            if action_data is not None:
                data = action_data.copy()

            if robot_comment or comment:
                joined_comment = ': '.join([force_text(part) for part in (robot_comment, comment) if part])
                data['comment'] = joined_comment

            impersonated = impersonator = None
            if _requester is not NOT_PROVIDED:
                if _requester is not None:
                    impersonated = _requester.impersonated
                    impersonator = _requester.impersonator
            elif requester is not None:
                impersonated = requester.impersonated
                impersonator = requester.impersonator

            action_ = self.actions.create(
                user_id=self.user_id,
                group_id=self.group_id,
                requester=impersonated,
                impersonator=impersonator,
                action=action,
                data=data,
                error=error[:ERROR_LEN],
                inconsistency=inconsistency,
                system_id=self.system_id,
                approverequest=approve_request,
                transfer=transfer,
                parent=parent_action,
            )

            return action_

        def deprive_role(action_, _requester, set_state_kwargs):
            if not self.is_force_deprive(action_, _requester):
                self.set_raw_state(
                    ROLE_STATE.DEPRIVING_VALIDATION,
                    depriving_at=timezone.now() + timezone.timedelta(minutes=settings.IDM_DEPRIVING_AFTER_MIN),
                    **set_state_kwargs
                )
                depriving_comment = 'Роль будет отозвана через 15 минут'
                action_.data['comment'] = '\n\n'.join(filter(None, (action_.data.get('comment'), depriving_comment)))
                action_.save(update_fields=['data'])
            else:
                self.set_raw_state(ROLE_STATE.DEPRIVING, **set_state_kwargs)
                if self.is_pushable():
                    self.system.remove_role_async(action_, with_push=with_push)
                else:
                    # если система не хочет знать о групповых ролях, отзываем роль, не трогая плагин
                    state_anystate_to_deprived(self)
                    self.send_email_with_results(action_)

        def to_requested_or_rerequested(self, action_, to='requested'):
            self.set_raw_state(to, expire_at=self.calc_expiration(ttl_date=self.ttl_date,
                                                                  ttl_days=settings.REQUESTED_ROLE_TTL))
            return action_

        def to_requested(self, action_):
            return to_requested_or_rerequested(self, action_, to='requested')

        def to_rerequested(self, action_):
            return to_requested_or_rerequested(self, action_, to='rerequested')

        def state_created_to_requested(self):
            action_ = create_action(ACTION.REQUEST)
            return to_requested(self, action_)

        def state_created_to_imported(self):
            action_ = create_action(ACTION.IMPORT)
            # выставляем дату, когда запрос роли должна протухнуть
            expire_at = timezone.now() + timezone.timedelta(days=settings.IDM_INCONSISTENCY_ROLE_EXPIRE_DAYS)
            self.set_raw_state('imported', expire_at=expire_at)

            # спрашиваем аппруверов, выдавать ли роль по неконсистентности
            log.info('Start workflow for created role %s', self.pk)
            # в качестве запрашивающего используем нашего робота
            robot = User.objects.get_idm_robot()
            workflow_context = self.apply_workflow(requester=robot, ignore_approvers=force, reason=RUN_REASON.IMPORT)
            self.report_conflicts(workflow_context)

            if force:
                # сразу выдаем такую роль, т.к. подтверждение не требуется в случае первоначальной синхронизации
                log.info('Force-granting role %d for inconsistency %d', self.pk, inconsistency.pk)
                self.set_state('granted', inconsistency=inconsistency, force=force)
            else:
                self.create_approve_requests(requesterify(robot), workflow_context, reason_action=action_)
            return action_

        def state_imported_to_granted(self):
            create_action(ACTION.APPROVE)
            if self.get_num_approves_left() == 0:
                self.requests.update(is_done=True)  # устанавливаем признак завершенности запроса
                if force:
                    message = _('Созданная при первоначальной синхронизации роль добавлена в систему.')
                else:
                    message = _('Созданная при разрешении расхождения роль повторно подтверждена '
                                'и осталась в системе.')
                to_granted(self, message)

        def state_requested_to_approved(self):
            action_ = create_action(ACTION.APPROVE)

            # роль добавляется только когда все проаппрувили
            if self.get_num_approves_left() == 0:
                # Если финальное слово за IDM, то роль в approved может висеть сколько угодно,
                # пока не достучится до системы
                # а если финальное слово за системой, роль может устареть
                expire_at = None
                if self.system.role_grant_policy == 'system':
                    expire_at = self.expire_at

                self.set_raw_state('approved', expire_at=expire_at)
                self.requests.update(is_done=True)  # устанавливаем признак завершенности запроса

                if self.is_pushable():
                    # роль добавлена обычным образом
                    self.system.add_role_async(action=action_, requester=requester)
                else:
                    # групповая роль, про которую система знать не хочет
                    grant_action = self.set_state('granted')
                    if self.last_request:
                        self.last_request.fetch_requester()
                    if self.parent and self.parent.last_request:
                        self.parent.last_request.fetch_requester()
                    self.send_email_with_results(grant_action, last_requester=requester)

                    role_request = self.last_request
                    if role_request is not None:
                        role_request.close_issue()

            return action_

        def state_granted_to_approved(self):
            if (self.parent_id is None and not ignore_parent_check) or not self.is_active or not self.is_pushable():
                raise RoleStateSwitchError(
                    'Cannot rewind role state back to approved for not referenced role')
            action_ = create_action(ACTION.APPROVE)
            self.set_raw_state('approved')
            distinct_exclude = None
            if inconsistency is not None and inconsistency.type == inconsistency.TYPE_OUR:
                distinct_exclude = ('system_specific',)
            self.system.add_role_async(action_, distinct_exclude=distinct_exclude)

        def state_granted_to_granted(self):
            if transfer is None:
                raise RoleStateSwitchError('Cannot keep role granted without specifying transfer')
            robot_comment = None
            if transfer.state == 'auto_rejected':
                robot_comment = 'Роль не перезапрошена, так как по данным заявки из Стаффа пересмотр не требуется'
            elif self.system.review_on_relocate_policy != 'review':
                robot_comment = 'Роль не перезапрошена, так как в системе отключен пересмотр при смене подразделения'
            action_ = create_action(ACTION.KEEP_GRANTED, robot_comment=robot_comment)
            return action_

        def to_onhold(self):
            from idm.core.models import Action
            # Холдим только групповые, групповые связанные с групповыми, и выданные по групповым
            if self.is_user_role() and not (self.parent_id and self.parent.is_group_role()):
                raise RoleStateSwitchError('Cannot hold role')

            if self.last_request and not self.last_request.is_done:
                self.last_request.set_done()  # Переводим в onhold, поэтому не должно быть никаких запросов

            # Для групповых ролей откладываем связанные и персональные, выданные по ним
            if self.is_group_role():
                parent_action = Action.objects.create(
                    action=ACTION.MASS_ACTION,
                    data={
                        'name': 'put_on_hold_group_referenced_roles',
                        'group_id': self.group.id,
                    }
                )
                self.refs.hold_or_decline_or_deprive(parent_action=parent_action)

            action_ = create_action(ACTION.HOLD)

            nonlocal expiration_date
            expire_at = expiration_date or (timezone.now() + timezone.timedelta(constance.config.IDM_ONHOLD_TTL_DAYS))
            if self.expire_at:
                expire_at = min(expire_at, self.expire_at)

            self.set_raw_state('onhold', expire_at=expire_at)
            return action_

        state_granted_to_onhold = to_onhold
        state_imported_to_onhold = to_onhold
        state_expiring_to_onhold = to_onhold
        state_need_request_to_onhold = to_onhold
        state_rerequested_to_onhold = to_onhold
        state_review_request_to_onhold = to_onhold

        def state_onhold_to_granted(self):
            to_granted(self, _('Роль восстановлена при восстановлении группы или '
                               'возвращении сотрудника в группу'))

        def state_depriving_validation_to_granted(self):
            """depriving_validation бывает только у персональных-выданных-по-групповой ролей"""
            to_granted(self, _('Роль восстановлена при восстановлении группы или '
                               'возвращении сотрудника в группу.'))

        def state_requested_to_declined(self):
            _requester = requester
            if (
                approve_request is not None
                and approve_request.decision == APPROVEREQUEST_DECISION.IGNORE
            ):
                action_ = create_action(ACTION.IGNORE)
                _requester = None
                if not self.last_request.has_declined_approves():
                    return action_

            action_ = create_action(ACTION.DECLINE, _requester=_requester)
            # устанавливаем статус в declined, толкаем плагин для отправки сообщения
            self.set_raw_state('declined', expire_at=None)
            self.requests.update(is_done=True)  # устанавливаем признак завершенности запроса

            last_request = getattr(getattr(approve_request, 'approve', None), 'role_request', None)
            self.send_email_with_results(action_, last_request=last_request)
            role_request = self.last_request
            if role_request is not None:
                role_request.close_issue()

            return action_

        def state_awaiting_to_declined(self):
            action_ = create_action(ACTION.DECLINE)
            self.set_raw_state('declined', expire_at=None)
            return action_

        def state_requested_to_expired(self):
            self.set_raw_state('expired', expire_at=None)
            self.requests.update(is_done=True)  # устанавливаем признак завершенности запроса
            action_ = create_action(ACTION.EXPIRE, _('Время ожидания подтверждений роли вышло.'))

            self.send_email_with_results(action_)
            return action_

        def state_sent_to_expired(self):
            return state_requested_to_expired(self)

        def _state_approved_to_failed_or_idm_error(self, state, action, comment):
            self.set_raw_state(state, expire_at=None)
            action_ = create_action(action, comment)
            try:
                self.send_email_with_results(action_, passport_login_created=passport_login_created)
            except Exception:
                log.exception('Cannot send error mail about role %d', self.pk)
            return action_

        def state_approved_to_failed(self):
            return _state_approved_to_failed_or_idm_error(
                self, ROLE_STATE.FAILED, ACTION.FAIL,
                _('Не удалось добавить роль в систему из-за ошибки в системе.'))

        def state_approved_to_idm_error(self):
            return _state_approved_to_failed_or_idm_error(
                self, ROLE_STATE.IDM_ERROR, ACTION.IDM_ERROR,
                _('Не удалось добавить роль в систему из-за ошибки в IDM.'))

        def state_approved_to_awaiting(self):
            self.set_raw_state('awaiting', expire_at=None)
            self.check_group_membership_system_relations_async()
            return create_action(ACTION.AWAIT, _('Ожидание необходимых условий.'))

        def state_approved_to_sent(self):
            assert self.system.role_grant_policy == 'system'
            self.set_raw_state('sent')
            return create_action(ACTION.SENT, _('Роль передана в систему.'))

        def state_approved_to_granted(self):
            assert self.system.role_grant_policy == 'idm'
            return to_granted(self)

        def state_awaiting_to_approved(self):
            action_ = create_action(ACTION.APPROVE, _('Все необходимые условия выполнены.'))

            expire_at = None
            if self.system.role_grant_policy == 'system':
                expire_at = self.expire_at

            self.set_raw_state('approved', expire_at=expire_at)

            if self.is_pushable():
                # роль добавлена обычным образом
                self.system.add_role_async(action=action_, requester=requester)
            else:
                # групповая роль, про которую система знать не хочет
                grant_action = self.set_state('granted')
                self.send_email_with_results(grant_action)
            return action_

        def state_sent_to_granted(self):
            assert self.system.role_grant_policy == 'system'
            return to_granted(self)

        def to_granted(self, comment_=None, action_data=None):
            if comment_ is None:
                comment_ = _('Роль подтверждена и добавлена в систему.')

            self.set_raw_state('granted',
                               expire_at=self.calc_expiration(self.ttl_date, self.ttl_days),
                               review_at=self.calc_expiration(self.review_date, self.review_days))
            action_ = create_action(ACTION.GRANT, comment_)
            if action_data:
                action_.data.update(action_data)
                action_.save(update_fields=('data',))

            self.system.request_refs_async(self, action_)
            self.check_group_membership_passport_logins_async()
            self.fetch_user()
            self.poke_awaiting_roles_async()
            self.check_group_membership_system_relations_async()
            self.deprive_other_roles_if_exclusive(action_)
            return action_

        def to_depriving(self, comment_=None, _action_data=None):
            _requester = requester
            if (
                approve_request is not None
                and approve_request.decision == APPROVEREQUEST_DECISION.IGNORE
            ):
                action_ = create_action(ACTION.IGNORE)
                _requester = None
                if not self.last_request.has_declined_approves():
                    return action_

            # expire для временных ролей
            if transition == 'expire':
                if self.parent_id and self.state != ROLE_STATE.ONHOLD:
                    raise RoleStateSwitchError('Referenced role cannot expire')
                if comment_ is None:
                    comment_ = _('Истек срок действия роли.')
                action_ = create_action(ACTION.EXPIRE, comment_, _requester=_requester)
            else:
                action_type = ACTION.REDEPRIVE if self.state == ROLE_STATE.DEPRIVING else ACTION.DEPRIVE
                action_ = create_action(action_type, comment_, _requester=_requester)

            if action_data is not None:
                force_deprive = action_data.get(ACTION_DATA_KEYS.FORCE_DEPRIVE)
                if force_deprive is not None:
                    if _action_data is None:
                        _action_data = {ACTION_DATA_KEYS.FORCE_DEPRIVE: force_deprive}
                    else:
                        _action_data[ACTION_DATA_KEYS.FORCE_DEPRIVE] = force_deprive

            if _action_data:
                action_.data.update(_action_data)
                action_.save(update_fields=('data',))

            self.requests.update(is_done=True)
            set_state_kwargs = {
                'expire_at': None,
                'review_at': None,
            }
            if self.state not in (ROLE_STATE.DEPRIVING, ROLE_STATE.DEPRIVING_VALIDATION):
                # Если роль переходит первый раз в статус depriving - выставим отзывающего и комментарий отызва роли
                set_state_kwargs['depriver'] = getattr(requester, 'impersonated', None)
                set_state_kwargs['deprive_comment'] = comment_ or comment or ''
            deprive_role(action_, requester, set_state_kwargs)

            return action_

        state_depriving_validation_to_depriving = to_depriving
        state_granted_to_depriving = to_depriving
        state_sent_to_depriving = to_depriving
        state_onhold_to_depriving = to_depriving

        def state_granted_to_expiring(self):
            if self.parent_id:
                raise RoleStateSwitchError('Referenced role does not expire')

            self.set_raw_state('expiring')
            action_ = create_action(ACTION.ASK_REREQUEST, _('Необходимо продлить роль'))
            return action_

        def state_granted_to_need_request(self):
            if self.parent_id:
                raise RoleStateSwitchError('Referenced role does not need rerequest')
            self.set_raw_state('need_request',
                               expire_at=(timezone.now() + timezone.timedelta(days=settings.REQUESTED_ROLE_TTL)))
            action_ = create_action(ACTION.ASK_REREQUEST, _('Необходимо перезапросить роль'))
            return action_

        def state_granted_to_review_request(self):
            """Прогоняет workflow для роли. Если нет ответственных, выдает роль, иначе - рассылает письма
            подтверждающим"""

            if self.parent_id:
                raise RoleStateSwitchError('Referenced role does not need review')

            nonlocal expiration_date
            expiration_date = expiration_date or calendar.get_next_workday(
                timezone.now() + timezone.timedelta(days=settings.REQUESTED_ROLE_TTL),
                datetime.time(15),
            )
            robot = User.objects.get_idm_robot()
            self.set_raw_state(
                'review_request',
                expire_at=expiration_date,
                review_date=None,
                review_days=None,
                review_at=None,
            )
            action_ = create_action(ACTION.REVIEW_REREQUEST)
            self.create_request(requesterify(robot), action_, reason=RUN_REASON.REREQUEST)
            action_.data['comment'] = _('Перезапрос роли в связи с регулярным пересмотром')
            action_.data['is_review'] = True
            action_.save()
            return action_

        def state_review_request_to_granted(self):
            action_ = create_action(ACTION.APPROVE)
            action_.data['is_review'] = True
            action_.save()
            if self.get_num_approves_left() == 0:
                self.requests.update(is_done=True)  # устанавливаем признак завершенности запроса
                to_granted(self, _('Перезапрошенная в связи с регулярным пересмотром роль повторно '
                                   'подтверждена и осталась в системе.'), {'is_review': True})
            return action_

        def state_review_request_to_depriving(self):
            assert transition in ['deprive', 'expire']
            if transition == 'expire':
                comment_ = _('Перезапрошенная в связи с регулярным пересмотром роль так и не '
                             'получила подтвержения в срок.')
            else:
                comment_ = _('Перезапрошенная в связи с регулярным пересмотром роль отозвана '
                             'и будет удалена из системы')
            return to_depriving(self, comment_, _action_data={'is_review': True})

        def state_need_request_or_expiring_to_depriving(self):
            assert transition in ['deprive', 'expire']
            if transition == 'expire':
                comment_ = _('Роль не перезапрошена в срок и будет удалена из системы.')
            else:
                comment_ = _('Неперезапрошенная роль отозвана и будет удалена из системы')
            return to_depriving(self, comment_)

        state_need_request_to_depriving = state_need_request_or_expiring_to_depriving
        state_expiring_to_depriving = state_need_request_or_expiring_to_depriving

        def back_to_requested_or_rerequested(self, reuse_approvers=False, to='requested'):
            action_ = create_action(ACTION.REREQUEST)
            to_requested_or_rerequested(self, action_, to=to)
            self.requests.update(is_done=True)
            self.create_request(requester, action_, reuse_approvers=reuse_approvers,
                                run_workflow=run_workflow)
            return action_

        state_declined_to_requested = back_to_requested_or_rerequested
        state_expired_to_requested = back_to_requested_or_rerequested
        state_failed_to_requested = back_to_requested_or_rerequested
        state_idm_error_to_requested = back_to_requested_or_rerequested
        state_deprived_to_requested = back_to_requested_or_rerequested

        def state_requested_to_requested(self):
            return back_to_requested_or_rerequested(self, reuse_approvers=True)

        def state_rerequested_to_rerequested(self):
            return back_to_requested_or_rerequested(self, reuse_approvers=True, to='rerequested')

        def state_need_request_to_rerequested(self):
            return back_to_requested_or_rerequested(self, to='rerequested')

        def state_expiring_to_rerequested(self):
            return back_to_requested_or_rerequested(self, to='rerequested')

        def state_granted_to_rerequested(self):
            return back_to_requested_or_rerequested(self, reuse_approvers=True, to='rerequested')

        def state_rerequested_to_depriving(self):
            assert transition in ['deprive', 'expire']
            if transition == 'expire':
                comment_ = _('Перезапрошенная роль так и не получила подтвержения в срок.')
            else:
                comment_ = _('Перезапрошенная роль отозвана и будет удалена из системы')
            return to_depriving(self, comment_)

        def state_imported_to_depriving(self):
            assert transition in ['deprive', 'expire']
            if transition == 'expire':
                comment_ = _('Созданная в связи с разрешением расхождения роль '
                             'так и не получила подтвержения в срок.')
            else:
                comment_ = _('Созданная в связи с разрешением расхождения роль '
                             'отозвана и будет удалена из системы')
            return to_depriving(self, comment_)

        def state_rerequested_to_granted(self):
            action_ = create_action(ACTION.APPROVE)
            if self.get_num_approves_left() == 0:
                self.requests.update(is_done=True)  # устанавливаем признак завершенности запроса
                to_granted(self, _('Роль повторно подтверждена и осталась в системе.'))
            return action_

        def state_depriving_to_depriving(self):
            return to_depriving(self, _('Отзываем неудаленную роль'))

        def state_depriving_to_deprived(self):
            action_comment = _('Роль удалена из системы.')
            return _to_deprived(self, action_comment)

        def state_anystate_to_deprived(self):
            self.requests.update(is_done=True)  # устанавливаем признак завершенности запроса
            return _to_deprived(self)

        def _to_deprived(self, comment=None):
            self.set_raw_state('deprived', expire_at=None)
            action_ = create_action(ACTION.REMOVE, robot_comment=comment)

            self.system.deprive_refs_async(self, action_, comment='Reference role is deprived')
            if self.user is not None and self.user.type == USER_TYPES.ORGANIZATION:
                from idm.core.tasks.roles import DepriveRolesWhenDetachingResource
                DepriveRolesWhenDetachingResource.apply_async(kwargs={'role_id': self.pk})
            return action_

        # ====v= ОСНОВНОЕ ТЕЛО ФУНКЦИИ =v====

        subject = subjectify(self)
        log.info('Attempt to change state from "%s" to "%s" for role %s of %s "%s"',
                 self.state, value, self.id, subject.get_ident(), subject.inflect('en'))
        func = locals().get(f'state_{self.state}_to_{value}')
        # нестандартный переход состояния роли для разрешения неконсистентностей
        # declined, expired -> depriving
        # granted -> deprived
        if func is None and from_any_state:
            func = locals().get(f'state_anystate_to_{value}')

        if func:
            old_state = self.state
            with transaction.atomic():
                # при запросах через апи, не ждем бесконечно лока
                self.lock(retry=True, nowait=from_api)
                result = func(self)
            signals.role_changed_state.send(sender=self, role=self, from_state=old_state, to_state=value,
                                            inconsistency=inconsistency)
            log.info('State of role %s successfully changed', self.id)
            return result
        else:
            create_action(ACTION.FAIL)
            available_states = set()
            for func in locals():
                if func.startswith('state_%s' % self.state) or (from_any_state and func.startswith('state_anystate')):
                    available_states.add(func.split('_to_')[1])
            if available_states:
                available_states = ', '.join(sorted(available_states)) + '.'
            else:
                available_states = 'none'

            raise RoleStateSwitchError(
                'Undefined state switch for role %(pk)s: %(from)s -> %(to)s. Available states: %(available)s' % {
                    'pk': self.pk,
                    'from': self.state,
                    'to': value,
                    'available': available_states
                })

    def _create_actions_from_workflow_data(self, role_request, workflow_context):
        def create_action(action, comment):
            self.actions.create(
                action=action,
                user_id=self.user_id,
                group_id=self.group_id,
                system_id=self.system_id,
                data={'comment': comment},
            )

        approvers_action_comment = f'Подтверждающие: {role_request.as_approvers_repr()}'
        email_cc = workflow_context.get('email_cc')
        if email_cc:
            approvers_action_comment += f'\n\nРассылки:'
            for state, emails in email_cc.items():
                approvers_action_comment += f'\n{state}: {",".join(x["email"] for x in emails)}'

        create_action(action=ACTION.APPLY_WORKFLOW, comment=approvers_action_comment)

        if workflow_context.get('workflow_comment'):
            create_action(action=ACTION.WORKFLOW_COMMENT, comment=workflow_context['workflow_comment'])

        if workflow_context.get('conflict_comment'):
            create_action(action=ACTION.CONFLICT_COMMENT, comment=workflow_context['conflict_comment'])

    def create_approve_requests(self, requester, workflow_context, reason_action, comment='', already_approved=None,
                                silent=False, from_api=False):
        """Эта функция запускает для роли workflow и рассылает письма тем сотрудникам,
            которые должны эту роль подтвердить.
        """
        approvers = workflow_context.get('approvers')
        notify_everyone = bool(workflow_context.get('notify_everyone'))
        granting_approve_request = None
        granter = None
        if already_approved is None:
            already_approved = set()

        if self.is_active:
            # Перезапросы активных ролей не могут быть тихими
            silent = False

        role_request = self.requests.create(
            role=self,
            requester=requester.impersonated,
            workflow=self.system.actual_workflow,
            notify_everyone=notify_everyone,
            reason=reason_action.action,
            comment=comment,
            is_silent=silent,
        )

        # Проставляем этот запрос как последний в роли
        self.last_request = role_request
        self.save(update_fields=('last_request',))

        reason_action.rolerequest = role_request
        reason_action.save()

        # чтобы не лазить в базу - сохраним аппрувы и их аппрувреквесты
        subject = self.get_subject()
        request_is_autoapproved = workflow_context.is_auto_approved(requester.impersonated, subject, already_approved)
        parent_approves = {}
        for or_group in approvers:
            approve = role_request.approves.create()
            # Не False, а None, чтобы правильно задать поле approved у approve request-a
            approve_is_autoapproved = None
            or_group_approvers = collections.OrderedDict()

            # Дедуплицируем аппруверов и узнаем, автоподтверждён ли approve_request
            for approver in or_group.approvers:
                if approver not in or_group_approvers:
                    or_group_approvers[approver] = {'notify': approver.notify}
                    if (approver.user == requester.impersonated or
                        subject == subjectify(approver.user) or
                        approver.user in already_approved
                    ):
                        or_group_approvers[approver]['auto'] = True
                        approve_is_autoapproved = True
                        if granter is None:
                            granter = approver.user
                else:
                    or_group_approvers[approver]['notify'] = combine_notifications(
                        or_group_approvers[approver], approver.notify
                    )

            for or_approver, approver_info in or_group_approvers.items():
                notify = approver_info['notify']
                autoapproved = approver_info.get('auto')
                parent = parent_approves.get(or_approver)
                approve_request = approve.requests.create(
                    approver=or_approver.user,
                    decision=APPROVEREQUEST_DECISION.APPROVE if autoapproved else APPROVEREQUEST_DECISION.NOT_DECIDED,
                    parent=parent,
                    notify=notify,
                    priority=or_approver.priority,
                )
                if parent is None:
                    parent_approves[or_approver] = approve_request
                if approve_is_autoapproved:
                    if granting_approve_request is None and or_approver.user == granter:
                        granting_approve_request = approve_request

        role_request.update_approves()

        self._create_actions_from_workflow_data(role_request, workflow_context)

        if request_is_autoapproved:
            role_request.is_done = True
            role_request.save()
            if self.is_active:
                # перезапрошенные роли нужно переводить в granted, а не в approved.
                target_state = 'granted'
            else:
                target_state = 'approved'

            granter_requester = Requester(impersonated=granter, impersonator=requester.impersonator)
            self.set_state(
                target_state, requester=granter_requester,
                approve_request=granting_approve_request,
                from_api=from_api,
            )
        else:
            role_request.notify_approvers(comment)
            # TODO: Подправить письма с учётом impersonator
            self.send_email_with_results(reason_action, context={'requester': requester.impersonated})

        return role_request

    def retry_failed(self, requester):
        requester = requesterify(requester)
        result = permissions_shortcuts.can_retry_failed_role(requester, self)

        if not result:
            raise Forbidden(_('Довыдача роли невозможна: %s') % result.message)

        if self.is_group_role():
            self.request_group_roles()
        else:
            self.parent.request_group_roles(user_ids=(self.user.id,))

    def rerequest(self, requester, **kwargs):
        requester = requesterify(requester)
        result = permissions_shortcuts.can_rerequest_role(requester, self)

        if not result:
            raise Forbidden(_('Перезапрос роли невозможен: %s') % result.message)
        if self.is_active:
            target_state = 'rerequested'
        else:
            target_state = 'requested'
        kwargs['run_workflow'] = not self.is_personal_role()
        if self.is_personal_role():
            # Для персональных ролей форсируем использование групповой формы
            cleaned_data = self.node.get_valid_fields_data(self.fields_data, group=self.parent.group)
        else:
            cleaned_data = self.node.get_valid_fields_data(self.fields_data, user=self.user, group=self.group)
        if self.fields_data != cleaned_data:
            self.fields_data = cleaned_data
            self.save(update_fields=('fields_data',))
        self.set_state(target_state, requester=requester, **kwargs)

    def create_request(
        self, requester, reason_action, run_workflow=True, reuse_approvers=False,
        comment='', reason=RUN_REASON.CREATE_APPROVE_REQUEST,
    ):
        from idm.core.models import RoleRequest

        subject = self.get_subject()
        Role.objects.check_role_request(
            requester, subject, self.system, self.node, self.fields_data, self.parent_id, self.is_public
        )
        Role.objects.check_role_uniqueness(self)
        role_request = None
        if run_workflow:
            workflow_context = self.apply_workflow(requester.impersonated, reason=reason)

            self.report_conflicts(workflow_context)

            already_approved = None
            if reuse_approvers:
                try:
                    last_request = self.get_last_request()
                except RoleRequest.DoesNotExist:
                    pass
                else:
                    already_approved = {
                        approve.approver for approve in
                        last_request.get_approve_requests(decision=APPROVEREQUEST_DECISION.APPROVE)
                    }

            role_request = self.create_approve_requests(requester, workflow_context, reason_action,
                                                        already_approved=already_approved, comment=comment)
        else:
            self.set_state('approved', requester=requester)
        return role_request

    def review(self, expiration_date: datetime.datetime = None):
        self.set_state('review_request', expiration_date=expiration_date)

    def ask_rerequest(self, transfer, parent_action=None):
        if self.system.review_on_relocate_policy != 'review':
            return self.keep_granted(transfer, parent_action)
        if self.user is not None:
            autoapproved_logins = [
                login.strip()
                for login
                in constance.config.IDM_TRANSFER_AUTOAPPROVED_LOGINS.split(',')
                if login.strip()
            ]
            if self.user.username in autoapproved_logins:
                return self.keep_granted(transfer, parent_action)

        try:
            self.set_state('need_request', transfer=transfer, parent_action=parent_action)
        except Exception:
            log.exception('Cannot change role id=%d state to need_request', self.pk)

    def keep_granted(self, transfer, parent_action=None):
        try:
            self.set_state('granted', transfer=transfer, parent_action=parent_action)
        except Exception:
            log.exception('Cannot keep role id=%d granted', self.pk)

    def deprive_or_decline(
        self,
        depriver: User,
        comment: str = None,
        bypass_checks: bool = False,
        parent_action: 'Action' = None,
        deprive_all: bool = False,
        approve_request: 'ApproveRequest' = None,
        to_replace: bool = False,
        with_push: bool = True,
        force_deprive: bool = False,
        from_api: bool = False,
    ):
        """Пытается отозвать роль от имени сотрудника <depriver>."""
        if depriver is None:
            with log_context(stack=''.join(traceback.format_stack())):
                log.warning('Depriver not passed for role %s', self.id)

        depriver = requesterify(depriver)
        if not bypass_checks:
            self.fetch_system()
            result = permissions_shortcuts.can_deprive_role(
                depriver,
                self,
                to_replace=to_replace,
                pass_check_state=force_deprive and not from_api,
            )
            if not result:
                raise Forbidden(result.message, data=result.data)

            # Эта проверка пока что нужна тут, для b2b
            # т.к. в permissions_shortcuts мы доверяем коннекту
            if not self.system.is_operational():
                raise Forbidden('System %s is broken. Role cannot be deprived.' % self.system.slug)

        target_state = transition = None
        self.refresh_from_db()
        if self.is_active or self.state == 'sent':
            target_state = 'depriving'
            transition = 'deprive'
        elif self.state in ('requested', 'awaiting'):
            target_state = 'declined'
            transition = 'decline'
        elif deprive_all and self.state in ('created', 'approved'):
            target_state = 'deprived'

        if target_state is not None:
            return self.set_state(
                target_state,
                transition=transition,
                requester=depriver,
                comment=comment,
                from_any_state=deprive_all,
                approve_request=approve_request,
                with_push=with_push,
                parent_action=parent_action,
                action_data={ACTION_DATA_KEYS.FORCE_DEPRIVE: force_deprive},
                from_api=from_api,
            )

    def inform_requester(self, action, last_requester=None):
        subjects = {
            'declined': _('Запрошенная вами роль в системе "%(system)s" отклонена'),
            'expired': _('Запрошенная вами роль в системе "%(system)s" не получила одобрения в срок'),
            'failed': _('Произошла ошибка при добавлении запрошенной вами в системе "%(system)s" роли'),
        }
        mail_subject_template = subjects[self.state]
        context = {
            'role': self,
            'system': self.system,
            'action': action,
        }
        requester = last_requester or self.get_last_requester_or_none()
        if not requester:  # если роль запрошена роботом, requester - None
            return
        if requester in self.get_subject().get_responsibles():
            # не оповещаем запросившего о том, что роль запрошена/протухла/отклонена, если он совпадает с владельцем,
            # так как владельцу мы отошлём другое письмо (ну или нет, если это запрещено через no_email)
            return
        templates = [
            'emails/requester/role_%(state)s_%(slug)s.html' % {'state': self.state, 'slug': self.system.slug},
            'emails/requester/role_%(state)s_%(slug)s.txt' % {'state': self.state, 'slug': self.system.slug},
            'emails/requester/role_%(state)s.html' % {'state': self.state},
            'emails/requester/role_%(state)s.txt' % {'state': self.state},
        ]
        try:
            with override(requester.lang_ui):
                mail_subject = mail_subject_template % {'system': self.system.get_name()}
                send_notification(mail_subject, templates, [requester], context)
        except Exception:
            log.exception('failed to send error report to user: %s', requester.username)

    def send_email_with_results(self, action, **kwargs):
        """Отправляет письмо с окончательным решением о запросе роли владельцу роли и, в некоторых случаях,
        запросившему роль
        action — ключевое действие, повлиявшее на решение. Например на отказ в запросе или отзыв роли.
        context - словарь данных, полученных от системы при добавлении роли, но которые мы не хотим хранить
                  (например, пароль)
        """
        self.fetch_node()
        if self.node.is_public is False or self.is_public is False:
            return

        assert (self.state in (ROLE_STATE.REQUESTED, ROLE_STATE.IMPORTED, ROLE_STATE.SENT, ROLE_STATE.GRANTED,
                               ROLE_STATE.REVIEW_REQUEST, ROLE_STATE.REREQUESTED, ROLE_STATE.ONHOLD,
                               ROLE_STATE.DECLINED, ROLE_STATE.EXPIRED, ROLE_STATE.DEPRIVED, ROLE_STATE.FAILED))

        subjects = {
            ROLE_STATE.REQUESTED: _('Роль в системе "%(system)s" требует подтверждения.'),
            ROLE_STATE.SENT: _('%(system)s. Роль передана в систему'),
            ROLE_STATE.GRANTED: _('%(system)s. Новая роль'),
            ROLE_STATE.DECLINED: _('%(system)s. Заявка на роль отклонена'),
            ROLE_STATE.DEPRIVED: _('%(system)s. Роль отозвана'),
            ROLE_STATE.IMPORTED: _('В системе %(system)s заведена роль в обход IDM.'),
            ROLE_STATE.FAILED: _('Ошибка при добавлении роли в систему "%(system)s"'),

            # cc-only письма
            ROLE_STATE.EXPIRED: _('%(system)s. Роль не получила одобрения в срок'),
        }

        self.fetch_user()
        self.fetch_group()
        context = kwargs.get('context') or {}
        context.update({
            'action': action,
            'role': self,
            'system': self.system,
            'user': self.user,
            'group': self.group,
        })

        if self.is_personal_role() and self.state != ROLE_STATE.ONHOLD and not self.email_cc:
            return

        # Тихие запросы не шлют писем, но только про запрос роли, про отзыв – шлём
        last_request = kwargs.get('last_request') or self.get_last_request_or_none()
        if last_request is not None and last_request.is_silent and self.state in ROLE_STATE.SILENCABLE_STATES:
            return

        # оповестим запросившего о проблемах с ролью
        if self.state in (ROLE_STATE.DECLINED, ROLE_STATE.EXPIRED, ROLE_STATE.FAILED):
            last_requester = kwargs.get('last_requester')
            if not last_requester and last_request:
                last_request.fetch_requester()
                last_requester = last_request.requester

            self.inform_requester(action, last_requester=last_requester)

        cc_only = False  # отсылать ли письмо только на соответствующую рассылку

        # пока что не оповещаем никого (кроме запросившего) о перезапрошенных ролях
        if self.state in (ROLE_STATE.REREQUESTED, ROLE_STATE.REVIEW_REQUEST):
            # TODO: сделать шаблоны и протестировать эти варианты
            return

        if self.state == ROLE_STATE.EXPIRED:
            # о протухших ролях оповещаем только на рассылку (и запросившему)
            cc_only = True
        elif self.state == ROLE_STATE.SENT and not kwargs.get('passport_login_created'):
            # про sent роли мы не оповещаем владельца – он ничего не может сделать с этим знанием
            cc_only = True
        elif self.no_email:
            # флаг no_email прямо запрещает отсылку почты владельцу роли – но только ему, на рассылку
            # письмо мы всё равно должны отправить
            cc_only = True

        subject = subjectify(self)
        mail_subject_template = subjects[self.state]

        templates = [
            'emails/role_%(state)s_%(slug)s.html' % {'state': self.state, 'slug': self.system.slug},
            'emails/role_%(state)s_%(slug)s.txt' % {'state': self.state, 'slug': self.system.slug},
            'emails/role_%(state)s.html' % {'state': self.state},
            'emails/role_%(state)s.txt' % {'state': self.state},
        ]
        templates_maillist = [
                                 'emails/role_%(state)s_%(slug)s_maillist.html' % {'state': self.state,
                                                                                   'slug': self.system.slug},
                                 'emails/role_%(state)s_%(slug)s_maillist.txt' % {'state': self.state,
                                                                                  'slug': self.system.slug},
                                 'emails/role_%(state)s_maillist.html' % {'state': self.state},
                                 'emails/role_%(state)s_maillist.txt' % {'state': self.state},
                             ] + templates

        if not cc_only:
            self = Role.objects.select_related('node', 'parent', 'parent__node', 'user').get(pk=self.pk)
            for responsible in subject.get_responsibles():
                email_context = context.copy()
                email_context.update({
                    'user': responsible,
                    'passport_login': self.fields_data.get('passport-login') if self.fields_data else None,
                    'passport_login_created': kwargs.get('passport_login_created', False),
                })
                with override(responsible.lang_ui):
                    self.fetch_system()
                    mail_subject = mail_subject_template % {'system': self.system.get_name()}
                    send_notification(mail_subject, templates, [responsible], email_context)

        self.send_email_to_maillist(mail_subject_template, templates_maillist, context)

    def send_email_to_maillist(self, mail_subject_template, templates_maillist, context):
        email_cc = (self.email_cc or {}).get(self.state)
        if not email_cc:
            return

        email_context = context.copy()
        languages = {'ru': [], 'en': []}
        for cc_item in email_cc:
            languages[cc_item['lang']].append(cc_item['email'])
        for language, emails in languages.items():
            with override(language):
                mail_subject = mail_subject_template % {'system': self.system.get_name()}
                send_notification(mail_subject, templates_maillist, emails, email_context)

    def should_have_login(self, is_required: Optional[bool] = None, ignore_group_role: bool = True) -> bool:
        if not self.is_user_role() and ignore_group_role:
            return False  # у группы паспортного логина быть не может

        self.fetch_node()
        return self.node.role_should_have_login(is_required=is_required)

    def get_passport_login(self):
        from idm.users.models import GroupMembership

        if not self.should_have_login():
            return None

        if self.is_personal_role():
            membership = (
                GroupMembership.objects
                    .filter(
                    user_id=self.user_id,
                    group_id=self.parent.group_id,
                    state=GROUPMEMBERSHIP_STATE.ACTIVE,
                )
                    .select_related('passport_login')
            ).first()
            if not membership:
                return None
            if not membership.passport_login:
                return None
            return membership.passport_login.login
        else:
            if isinstance(self.fields_data, dict):
                return self.fields_data.get('passport-login')
            else:
                return None

    def add_passport_login_to_membership(self, passport_login):
        from idm.users.models import GroupMembership

        if not self.is_personal_role():
            return
        membership = GroupMembership.objects.filter(user_id=self.user_id, group_id=self.parent.group_id).first()
        if not membership:
            return
        membership.fetch_passport_login()
        if membership.passport_login == passport_login:
            return
        if membership.passport_login != passport_login and membership.passport_login:
            raise MembershipPassportLoginError(
                'Passport login for membership %s already exists',
                membership.pk,
            )
        membership.passport_login = passport_login
        membership.save(update_fields=['passport_login'])

    def get_last_request(self):
        """Последний запрос роли."""
        from idm.core.models import RoleRequest
        if self.last_request_id is None:
            raise RoleRequest.DoesNotExist('RoleRequest matching query does not exist.')
        return self.last_request

    def get_last_request_or_none(self):
        """Последний запрос роли, если есть, иначе None"""
        return self.last_request

    def get_last_requester(self):
        """Инициатор последнего запроса роли"""
        from idm.core.models import RoleRequest
        try:
            requester = self.get_last_request().requester
        except RoleRequest.DoesNotExist:
            requester = None
        return requester

    def get_last_requester_or_none(self):
        """Последний инициатор запроса роли или None, если такого не нашлось"""
        from idm.core.models import RoleRequest
        try:
            requester = self.get_last_request().requester
        except RoleRequest.DoesNotExist:
            requester = None
        return requester

    def get_last_approves(self):
        """Аппрувы для последнего запроса роли"""
        from idm.core.models import Approve, RoleRequest
        try:
            last_request = self.get_last_request()
        except RoleRequest.DoesNotExist:
            return Approve.objects.none()
        else:
            return last_request.approves.order_by('pk')

    def get_open_request(self):
        """Открытый запрос роли, если есть. Если нет - RoleRequest.DoesNotExist"""
        from idm.core.models import RoleRequest
        if self.last_request_id and not self.last_request.is_done:
            return self.last_request
        else:
            raise RoleRequest.DoesNotExist('RoleRequest matching query does not exist.')

    def get_open_request_approves(self):
        """Все аппрувы открытого запроса роли"""
        return self.get_open_request().approves.order_by('pk')

    def get_undecided_approves(self):
        """Аппрувы, по которым не было принято решения, для открытого запроса роли"""
        return self.get_open_request().get_undecided_approves()

    def get_num_approves_left(self):
        """Количество подтверждений, которые осталось получить.
        Учитываются только аппрувы, которые необходимо получить по последнему запросу роли."""
        from idm.core.models import RoleRequest
        try:
            count = self.get_open_request().get_num_approves_left()
        except RoleRequest.DoesNotExist:
            count = 0
        return count

    def get_ancestors(self, include_self=False):
        ancestors = []
        current = self
        while current is not None:
            ancestors.append(current)
            if current.parent_id is None:
                current = None
            else:
                current.fetch_parent()
                current = current.parent
        ancestors.reverse()

        if not include_self:
            ancestors.pop()

        return ancestors

    def humanize(self, lang: str = None, format: str = 'full') -> str:
        return humanize_role(self, lang, format)

    def email_humanize(self, lang: str = None, format: str = 'full') -> str:
        from idm.core.mutation import RoleFieldMetrikaCounterMutation
        humanized = humanize_role(self, lang, format)
        if self.fields_data:
            fields_for_email = self.fields_data.copy()
            passport_login = fields_for_email.pop('passport-login', None)

            if RoleFieldMetrikaCounterMutation.is_applicable(self):
                counter_name = RoleFieldMetrikaCounterMutation.mutate(fields_for_email)
                if counter_name:
                    humanized += f' ({counter_name})'

            if fields_for_email:
                humanized += _('\nДанные полей') + f': {fields_for_email}'
            if passport_login:
                humanized += _('\nПаспортный логин') + f': {passport_login}'
        return humanized

    @property
    def ad_system(self):
        if not hasattr(self, '_ad_system'):
            from idm.core.models import System
            self._ad_system = System.objects.get(slug='ad_system')
        return self._ad_system

    def get_ad_ref_role(self, group_name: str) -> dict:
        from idm.core.models import RoleNode
        role_node = RoleNode.objects.get_node_by_unique_id(system=self.ad_system, unique_id=f'{group_name}::member')
        if role_node is None:
            raise WorkflowError(f'Группы {group_name} нет в AD')
        return {
            'system': 'ad_system',
            'role_data': role_node.data
        }

    def set_ad_groups(self, groups: List[str]) -> None:
        self.ad_groups = []
        if groups:
            self.ref_roles = self.ref_roles or []
            self.ref_roles.extend(map(self.get_ad_ref_role, groups))

    def _lock(self, nowait=False):
        return list(Role.objects.filter(pk=self.pk).select_for_update(nowait=nowait))

    def lock(self, for_task=False, retry=False, nowait=False):
        assert not (for_task is True and retry is True), 'lock() could be used with for_task or retry, but not both'
        try_count = 5
        if for_task:
            try:
                self._lock(nowait=nowait)
            except OperationalError:
                if nowait:
                    raise
                raise DelayingError('Could not lock the role %d right now, retrying' % self.pk)
        elif retry:
            for i in range(try_count):
                try:
                    self._lock(nowait=nowait)
                except OperationalError:
                    time.sleep(0.1 * (2 << i))  # backoff policy, up to 3200 ms
                    if i == try_count - 1:
                        raise
                else:
                    break
        else:
            self._lock()

        if self.pk is not None:
            self.refresh_from_db()

    def is_unique(self, ignore: Iterable[str] = None, among_states: Iterable[str] = None, passport_login_to_use=None):
        if among_states is None:
            among_states = ROLE_STATE.RETURNABLE_STATES
        qs = self.get_alike(ignore=ignore, among_states=among_states, passport_login_to_use=passport_login_to_use)
        return not qs.exists()

    def get_alike(self, ignore: Iterable[str] = None, among_states: Iterable[str] = None, passport_login_to_use=None):
        if among_states is None:
            among_states = ROLE_STATE.RETURNABLE_STATES
        ignorable = ('parent', 'fields_data', 'system_specific')

        # При создании персональной роли мы ещё не записали паспортный логин в fields_data
        # (это делается в таске при апруве)
        # но проверять роли на равенство нужно с учётом его наличия там
        fields_data = self.fields_data
        if passport_login_to_use:
            if fields_data is None:
                fields_data = {'passport-login': passport_login_to_use.login}
            elif isinstance(fields_data, dict) and not fields_data.get('passport-login'):
                fields_data = fields_data.copy()
                fields_data['passport-login'] = passport_login_to_use.login

        params = {
            'system': Q(system_id=self.system_id),
            'user': Q(user_id=self.user_id),
            'group': Q(group_id=self.group_id),
            'node': Q(node_id=self.node_id),
            'parent': Q(parent_id=self.parent_id),
            'fields_data': Q(fields_data=fields_data),
            'system_specific': Q(system_specific=self.system_specific),
        }

        if ignore is not None:
            for ignored_key in ignore:
                if ignored_key not in ignorable:
                    raise ValueError('Key %s cannot be ignored' % ignored_key)
            params = {key: value for key, value in params.items() if key not in ignore}
        qs = Role.objects.filter(*list(params.values()))
        qs = qs.filter(state__in=among_states)
        if self.pk:
            qs = qs.exclude(pk=self.pk)
        return qs

    def request_group_roles(self, user_ids: Collection[int] = None):
        """Запрашивает только роли по групповым"""
        if not self.is_referrable_group_role():
            return

        query = Q(
            state__in=GROUPMEMBERSHIP_STATE.ACTIVE_STATES,
            user__is_active=True,
        )
        if user_ids is not None:
            query &= Q(user_id__in=user_ids)
        if not self.with_robots:
            query &= Q(user__is_robot=False)
        if not self.with_inheritance:
            query &= Q(is_direct=True)
        if not self.with_external:
            query &= Q(user__affiliation__in=AFFILIATION.INTERNAL_VALUES) | Q(user__is_robot=True)

        self.fetch_node()
        self.fetch_system()
        for membership in self.group.memberships.filter(query).select_related('user'):
            self.create_group_member_role(membership.user)

    def request_ref_roles(self, parent_action: 'Action' = None):
        """Запрашивает только связанные роли"""

        # значение '' нужно для инициализации, при первом запуске запишем туда пустое множество
        # если кеш не ответит, то cache.get возвращает None, поэтому тут используется or
        cached_overlengthed_ref_roles = OverlengthedRefRoleChainMetric.get() or []
        overlengthed_ref_roles = set(cached_overlengthed_ref_roles)

        chain_length_error = len(self.get_ancestors(include_self=True)) >= settings.IDM_MAX_REF_ROLES_CHAIN_LENGTH
        if chain_length_error:
            overlengthed_ref_roles.add(self.pk)
            log.warning('Maximum length of ref roles chain exceeded ({length}). System: {slug}, role: {pk}'.format(
                length=settings.IDM_MAX_REF_ROLES_CHAIN_LENGTH,
                slug=self.system.slug,
                pk=self.pk
            ))
        if cached_overlengthed_ref_roles != overlengthed_ref_roles:
            OverlengthedRefRoleChainMetric.set(overlengthed_ref_roles)
        if chain_length_error:
            return

        ref_roles = []
        for ref in self.ref_roles or []:
            as_canonical = canonical.CanonicalRef.from_dict(ref)
            try:
                as_canonical.validate()
            except NotifyReferenceRolesValidationError as e:
                report_problem(
                    'Попытка запросить невалидную связанную роль',
                    ['emails/service/invalid_ref_role.txt'],
                    {
                        'system': self.system.name,
                        'node': self.node.get_name(),
                        'ref_role': ref,
                        'error': force_text(e)
                    }
                )

                # игнорируем некорректные описания связанных ролей.
                # exception здесь потому, что на такие ошибки нужно реагировать: исправлять воркфлоу систем
                log.exception('Wrong ref_roles in system %s role %s', self.system.slug, self.pk)
            except SilentReferenceRolesValidationError:
                log.info('Cannot find node %s for ref_role in system %s', self.node.get_name(), self.system.slug)
            else:
                ref_roles.append(as_canonical)

        active_refs = (
            self.refs.select_related('system', 'node').
                filter(state__in=ROLE_STATE.RETURNABLE_STATES).
                of_role_type(self)
        )
        refs_to_deprive = []
        for active_ref in active_refs:
            as_ref = canonical.CanonicalRef.from_model(active_ref)

            if as_ref in ref_roles:
                ref_roles.remove(as_ref)
            else:
                refs_to_deprive.append(active_ref)

        request_errors = []
        idm_robot = User.objects.get_idm_robot()
        for ref_role in ref_roles:
            fields = ref_role.fields_data
            if fields:
                fields = dict(fields)

            try:
                Role.objects.request_role(
                    requester=idm_robot,
                    subject=self.get_subject(),
                    system=ref_role.system,
                    comment='Reference role request',
                    data=ref_role.node,
                    fields_data=fields,
                    parent=self,
                    is_public=ref_role.is_public,
                    # Запрашивать связные роли в другой организации пока запрещено
                    organization_id=self.organization_id,
                    parent_action=parent_action,
                    with_external=self.with_external,
                    with_robots=self.with_robots,
                    with_inheritance=self.with_inheritance,
                    without_hold=self.without_hold,
                )
            except Exception:
                # запросить связанную роль не удалось
                request_errors.append(sys.exc_info())
                log.warning(
                    'Reference role request for role %d failed: system %s, node: %s, fields: %s, is_public: %s',
                    self.pk, ref_role.system.slug, ref_role.node.data, fields, ref_role.is_public, exc_info=True,
                )

        if request_errors:
            self.create_action(
                action=ACTION.REF_ROLE_ERROR,
                comment=_('Не удалось запросить связанные роли'),
                error='--------\n'.join([format_error_traceback(*exc_info) for exc_info in request_errors])[:ERROR_LEN]
            )

        for ref in refs_to_deprive:
            try:
                ref.deprive_or_decline(
                    depriver=idm_robot,
                    parent_action=parent_action,
                    bypass_checks=True,
                    comment='Reference role is deprived',
                )
            except Exception:
                log.warning('Cannot deprive ref %d of role %d that should no longer be active', ref.pk, self.pk)

    def request_refs(self):
        """Запрашивает связанные роли и персональные роли, если роль групповая"""

        self.request_group_roles()
        self.request_ref_roles()

    def deprive_refs(self, bypass_checks=False, comment=None, user=None):
        """Отзывает связанные роли"""
        qs = self.refs.filter(
            state__in=ROLE_STATE.ACTIVE_STATES | ROLE_STATE.NOT_YET_ACTIVE_STATES,
            system__is_active=True,
            system__is_broken=False,
        )
        if user:
            qs = qs.filter(user=user)

        idm_robot = User.objects.get_idm_robot()
        for ref in qs.select_related('system').iterator():
            try:
                ref.deprive_or_decline(idm_robot, bypass_checks=bypass_checks, comment=comment)
            except Exception:
                log.warning('Reference role depriving for role `%s` failed: %s', self.id, ref.pk, exc_info=True)

    def check_group_membership_passport_logins_async(self):
        if not (self.is_group_role() and
                self.system.group_policy == SYSTEM_GROUP_POLICY.AWARE_OF_MEMBERSHIPS_WITH_LOGINS and
                self.should_have_login(is_required=True, ignore_group_role=False)):
            return

        from idm.core.tasks import CheckGroupMembershipPassportLogin

        memberships = self.group.get_descendant_memberships().filter(passport_login=None)
        for membership in memberships:
            CheckGroupMembershipPassportLogin.apply_async(
                kwargs={'membership_id': membership.pk, 'role_id': self.pk}
            )

    def should_be_put_on_hold(self):
        return self.state == 'granted' and (
            self.user_id is None or
            (
                self.user.is_active and
                not self.user.is_frozen
            )
        ) and not self.without_hold

    def put_on_hold(self, comment=None, parent_action=None, expiration_date=None):
        try:
            self.set_state('onhold', comment=comment, parent_action=parent_action, expiration_date=expiration_date)
        except Exception:
            log.exception('Cannot put on hold a role %s', self.pk)

    def run_workflow(self, requester=None, ignore_approvers=False, reason=None,
                     execution_method=None, request_type=REQUEST_TYPE.REQUEST):
        """
        Прогоняет workflow для запроса или отзыва роли {role}.
        """
        if requester is None and self.user_id is not None:
            requester = self.user

        subject = self.get_subject()
        self.fetch_parent()
        if self.last_request:
            self.last_request.fetch_requester()
        params = {
            'code': subject.get_workflow_code(self.system),
            'role_data': self.node.data,
            'node': self.node,
            'system': self.system,
            'requester': requester,
            'original_requester': self.last_request.requester if self.last_request else None,
            'request_type': request_type,
            'fields_data': self.fields_data,
            'system_specific': self.system_specific,
            'scope': self.node.parent.value_path,
            'subject': subject,
            'parent': self.parent,
            'ignore_approvers': ignore_approvers,
            'reason': reason,
            'execution_method': execution_method,
            'request_fields': self.request_fields,
            'ttl_days': self.ttl_days,
            'inheritance_settings': {
                'with_inheritance': self.with_inheritance,
                'with_robots': self.with_robots,
                'with_external': self.with_external,
                'without_hold': self.without_hold,
            }
        }
        result = workflow(**params)
        return result

    def apply_workflow(
        self,
        requester: Requester,
        save: bool = True,
        ignore_approvers: bool = False,
        reason: str = None,
        request_type: str = REQUEST_TYPE.REQUEST,
    ) -> 'Role':
        """
        Прогнать объект роли через workflow соответствующей системы и занести необходимую информацию
        в объект роли, проверить необходимые результаты выполнения workflow.
        Возвращает словарь с результатами выполнения workflow.

        :param requester: User, запрашивающий
        :param save: bool, сохранять ли роль после прогона workflow
        :param ignore_approvers: bool, игнорировать ли отсутствие/равенство None переменной approvers в контексте
        :param request_type: вызывается ли воркфлоу для процесса запроса или отзыва
        :return: dict, результат прогона workflow
        :raises WorkflowError
        """
        workflow_data = self.run_workflow(requester=requester, ignore_approvers=ignore_approvers,
                                          reason=reason, request_type=request_type)

        approvers = workflow_data.get('approvers')
        if not ignore_approvers:
            if approvers is None:
                # здесь approvers is None, значит для роли нет правил в workflow
                log.warning('Undefined behaviour in system %s for role request: role %s, requester %s',
                            self.system, self, requester)
                raise WorkflowError(
                    'Не найдено правило, согласно которому должна быть '
                    'подтверждена роль {role}, запрос сделал {requester}'.format(
                        role=self,
                        requester=requester
                    )
                )

        if workflow_data.get('send_sms'):
            if self.is_group_role() or self.user.actual_mobile_phone:
                self.send_sms = True
            else:
                message = _('Согласно правилу workflow необходимо послать пользователю %(user)s sms, '
                            'однако у него не указан на стаффе мобильный телефон')
                message = message % {
                    'user': self.user.username
                }
                raise NoMobilePhoneError(message)

        if (self.is_group_role() and
                self.group.get_descendant_members().count() >= constance.config.SUPERVISION_ROLE_GROUP_SIZE):
            workflow_data['approvers'].append(
                AnyApprover([
                    user for user in
                    User.objects.filter(username__in=constance.config.SUPERVISION_ROLE_USERS.split(','))
                ])
            )

        self.email_cc = workflow_data.get('email_cc')
        self.ttl_days = min_with_none(self.ttl_days, workflow_data.get('ttl_days'))
        self.review_days = min_with_none(self.review_days, workflow_data.get('review_days'))
        self.remove_redundant_ttl()
        self.no_email = workflow_data.get('no_email', False)
        self.ref_roles = workflow_data.get('ref_roles')
        self.set_ad_groups(workflow_data.get('ad_groups'))

        if self.id and save:
            self.save(update_fields=('send_sms', 'ad_groups', 'email_cc', 'no_email', 'ref_roles',
                                     'ttl_days', 'ttl_date', 'review_days', 'review_date'))

        return workflow_data

    def rerun_workflow(self, robot, parent_action=None):
        assert not self.is_personal_role(), 'Running workflow for personal roles is illegal'

        result = True
        error_message = ''
        current = {}
        fieldnames = ['ad_groups', 'email_cc', 'no_email', 'ref_roles',
                      'ttl_date', 'ttl_days', 'expire_at',
                      'review_date', 'review_days', 'review_at']
        for fieldname in fieldnames:
            current[fieldname] = getattr(self, fieldname)

        diff = {}
        try:
            self.apply_workflow(requester=robot, save=False, ignore_approvers=True, reason=RUN_REASON.RERUN)
            if self.granted_at:
                self.expire_at = min_with_none(
                    self.expire_at,
                    self.calc_expiration(self.ttl_date, self.ttl_days, relative_to=self.granted_at)
                )
                self.review_at = min_with_none(
                    self.review_at,
                    self.calc_expiration(self.review_date, self.review_days, relative_to=self.granted_at)
                )
            for fieldname in fieldnames:
                new = getattr(self, fieldname)
                prev = current[fieldname]
                if new != prev:
                    diff[fieldname] = [prev, new]
            if diff:
                self.save(update_fields=list(diff.keys()))
            data = {
                'diff': diff
            }
            if self.is_active:
                grant_action = self.actions.filter(action='grant').order_by('-added').first()
                self.system.request_refs_async(self, grant_action)
        except Exception as e:
            result = False
            error_message = force_text(e)
            data = {
                'error': error_message
            }

        self.actions.create(
            action='rerun_workflow',
            user=self.user,
            group=self.group,
            requester=robot,
            data=data,
            error=error_message[:MAX_ACTION_ERROR_LENGTH],
            system=self.system,
            parent=parent_action,
        )
        return result

    def may_be_asked_to_rerequest(self):
        rerequest_is_enabled = type(self).objects.rerequest_is_enabled()
        return (
            rerequest_is_enabled and
            self.parent is None and
            self.state == 'granted' and
            self.system.is_operational() and
            self.system.review_on_relocate_policy == 'review'
        )

    def expire_alike(self):
        """Перевести такие же запрошенные роли пользователя/группы в expired. Вызывается при разрешении
        неконсистентности"""
        # TODO: Здесь есть одна проблема: из неконсистентностей мы знаем только system_specific,
        # а из роли только fields_data, потому что она запрошена, но ещё не выдана.
        # Соответственно, здесь нельзя точно установить соответствие между запрошенными ролями и существующими
        # неконсистентностями. Сейчас это делается через копирование в fields_data подмножества system_specific,
        # но в будущем нужно решить это лучше.
        alike_roles = self.get_alike(ignore=('parent', 'system_specific'), among_states={'requested'})
        comment = _('Отзыв запроса роли в связи с выдачей аналогичной роли при разрешении расхождения')
        for role in alike_roles:
            add_to_instance_cache(role, 'system', self.system)
            role.set_state('expired', comment=comment)

    def has_expired_ttl(self):
        return self.ttl_date is not None and timezone.now() >= self.ttl_date

    def remove_redundant_ttl(self):
        if self.ttl_days and self.ttl_date:
            if self.ttl_date > timezone.now() + timezone.timedelta(days=self.ttl_days):
                self.ttl_date = None
            else:
                self.ttl_days = None
        if self.review_date and self.review_days:
            if self.review_date > timezone.now() + timezone.timedelta(days=self.review_days):
                self.review_date = None
            else:
                self.review_days = None

    def add_passport_login(self, passport_login, approve):
        from idm.core.models import UserPassportLogin
        login = UserPassportLogin.objects.add(passport_login, self)
        if self.fields_data is None:
            self.fields_data = {}
        self.fields_data['passport-login'] = passport_login
        self.save(update_fields=['fields_data'])
        if approve and login.is_fully_registered:
            self.set_state('approved')
        return login

    def create_group_member_role(self, user, replace_login=False):
        """Создаём для пользователя user пользовательскую роль по групповой.
        При этом мы копируем поля:
        - send_sms - потому что рассылка sms направлена на пользователя, оповещает его, что ему выдана роль
        - ad_groups - потому что в группы AD помещаются пользователи, а не группы
        - with_inheritance, with_robots, with_external, without_hold - чтобы удобнее было отзывать такие роли
        - email_cc - в случае, если pass_to_personal=True, то есть явно указано передавать поле в персональные роли
        И *не* копируем поля:
        - ref_roles - связанные роли связываются с групповой ролью и должны быть также групповыми
        - ttl_days - нет смысла копировать поле ttl_days, потому что пользовательские роли связаны с групповой,
        у которой ttl_days уже прописан. При истечении срока действия групповой роли пользовательские роли будут
        отозваны автоматически.
        - review_days - аналогично ttl_days
        Поле no_email выставляем в True чтобы не слать почту о выдаче пользовательской роли самому пользователю
        """
        subject = subjectify(user)
        if not subject.is_requestable_for():
            return

        try:
            with transaction.atomic():
                role = Role.objects.create_role(
                    subject=subject,
                    system=self.system,
                    node=self.node,
                    fields_data=self.fields_data,
                    parent=self,
                    save=False,
                    organization_id=self.organization_id,
                )
                login = None
                if role.should_have_login():
                    membership = self.group.memberships.filter(user=user).first()
                    if membership:  # членство может быть не непосредственным, такие мы игнорим
                        membership.fetch_passport_login()
                        login = membership.passport_login
                    elif user.passport_logins.count() == 1:  # но если членство не непосредственное,
                        login = user.passport_logins.get()  # мы всё равно можем попытаться найти логин
                ignored_states = {ROLE_STATE.ONHOLD} if (login and replace_login) else set()

                if role.is_unique(ignore=['system_specific'],
                                  among_states=ROLE_STATE.REF_REREQUESTABLE_STATES - ignored_states,
                                  passport_login_to_use=login):
                    role.ad_groups = self.ad_groups
                    role.send_sms = self.send_sms
                    role.no_email = True
                    role.with_inheritance = self.with_inheritance
                    role.with_robots = self.with_robots
                    role.with_external = self.with_external
                    role.without_hold = self.without_hold
                    # Копирование полей email_cc где pass_to_personal = True
                    if self.email_cc:
                        role.email_cc = {}
                        for action in self.email_cc:
                            copy_items = []
                            for cc_item in self.email_cc[action]:
                                if cc_item.get('pass_to_personal'):
                                    copy_items.append(cc_item)
                            if len(copy_items) > 0:
                                role.email_cc[action] = copy_items
                    role.save()
                    role.set_state(ROLE_STATE.REQUESTED, comment='Reference role request')
                    role.set_state(ROLE_STATE.APPROVED)
                    return role
                else:
                    existing_roles = role.get_alike(ignore=['system_specific'],
                                                    among_states=ROLE_STATE.RETURNABLE_STATES - ignored_states,
                                                    passport_login_to_use=login)
                    chosen_role = existing_roles.select_related('parent', 'parent__group').order_by('-pk').first()
                    if chosen_role is None:
                        inactive_roles = role.get_alike(ignore=['system_specific'],
                                                        among_states=ROLE_STATE.REF_REREQUESTABLE_STATES - ignored_states,
                                                        passport_login_to_use=login)
                        chosen_role = inactive_roles.select_related('parent', 'parent__group').order_by('-pk').first()
                    if chosen_role is None:
                        # что-то пошло не так, то ли роли нет, то ли есть
                        return

                    add_to_instance_cache(chosen_role, 'system', role.system)
                    add_to_instance_cache(chosen_role, 'node', role.node)
                    add_to_instance_cache(chosen_role, 'group', role.group)
                    add_to_instance_cache(chosen_role, 'user', role.user)
                    changed_fields = []
                    for key in ('ad_groups', 'send_sms', 'no_email'):
                        group_value = getattr(self, key)
                        if getattr(chosen_role, key) != group_value:
                            changed_fields.append(key)
                            setattr(chosen_role, key, group_value)
                    if changed_fields:
                        chosen_role.save(update_fields=changed_fields)

                    if chosen_role.state in (ROLE_STATE.ONHOLD, ROLE_STATE.DEPRIVING_VALIDATION):
                        chosen_role.set_state(ROLE_STATE.GRANTED, comment='User has returned to some group')
                    elif chosen_role.state == ROLE_STATE.AWAITING:
                        if login and login.is_fully_registered:
                            chosen_role.set_state(ROLE_STATE.APPROVED, comment='Все условия выполнены')
                    elif chosen_role.state in ROLE_STATE.INACTIVE_STATES:
                        chosen_role.rerequest(requester=None, comment='Reference role re-request')
                    return chosen_role
        except Exception:
            log.warning('Cannot create personal role for user %s for group role %d', user.username, self.pk, exc_info=1)

    def report_conflicts(self, workflow_context):
        """
            Уведомляет о ролях, конфликтующих с указанной
        """
        if self.is_personal_role():
            return

        if not self.is_public_role():
            return

        if waffle.switch_is_active('find_conflicts_in_workflow'):
            conflicts = workflow_context.get('conflicts', {})
        else:
            conflict_rules = workflow_context.get('conflicts')
            if not conflict_rules:
                return

            conflicts = self.find_conflicts(conflict_rules)

        for recipient, conflict_set in conflicts.items():
            owner = self.get_subject()
            email_subject = _('Обнаружены конфликты при запросе роли для %(owner)s (%(ident)s)') % {
                'owner': owner.get_name(),
                'ident': owner.get_readable_ident()
            }
            send_notification(
                email_subject,
                'emails/role_conflict.txt',
                [recipient],
                {
                    'role': self,
                    'conflicts': list(conflict_set),
                }
            )

    def find_conflicts(self, rules: List[List[str]]) -> dict:
        self.fetch_node()
        return find_conflicts(subjectify(self), self.node, rules)

    def check_fields_data(self):
        """Проверка данных в fields_data"""
        # пока проверяем только паспортный логин, возможно позже добавится еще что-то
        from idm.core.models import UserPassportLogin

        if not isinstance(self.fields_data, dict):
            return
        field_slugs = [field.slug for field in self.node.get_fields() if field.type == FIELD_TYPE.PASSPORT_LOGIN]
        for slug in field_slugs:
            login = self.fields_data.get(slug)
            if not login:
                continue
            passport_login = UserPassportLogin.objects.filter(login=login).first()
            if not passport_login:
                continue
            if passport_login.user_id and passport_login.user_id != self.user_id:
                self.fetch_user()
                passport_login.fetch_user()
                raise MultiplePassportLoginUsersError(
                    'Role %s for user %s on passport_login %s can not be issued, passport_login belongs to user %s'
                    % (self.node.value_path, self.user.username, passport_login.login, passport_login.user.username)
                )

    def need_retry_push_to_system(self, system_inconsistency_policy):
        return (
            (self.parent_id is not None or system_inconsistency_policy == SYSTEM_INCONSISTENCY_POLICY.TRUST_IDM)
            and self.state in ROLE_STATE.ACTIVE_RETURNABLE_STATES
            and self.is_pushable()
        )

    def get_main_approvers_for_all_groups(self):
        return self.get_open_request().get_main_approvers_for_all_groups()

    def get_additional_approvers_for_all_groups(self):
        return self.get_open_request().get_additional_approvers_for_all_groups()

    def get_emailed_approvers_for_all_groups(self):
        return self.get_open_request().get_emailed_approvers_for_all_groups()

    def check_awaiting_conditions(self, passport_login_id, comment=None):
        comments = [comment] if comment else []
        target_state = ROLE_STATE.GRANTED
        validators = [import_string(condition) for condition in settings.CHECK_AWAITING_CONDITION_VALIDATORS]
        for validator in validators:
            is_valid, comment = validator(self).check(passport_login_id=passport_login_id)
            if not is_valid:
                target_state = ROLE_STATE.AWAITING
                comments.append(comment)
        return target_state, comments

    def check_group_membership_system_relations_async(self):
        from idm.core.tasks.group_memberships import SyncAndPushMemberships

        self.fetch_system()
        if self.group_id is None or self.system.group_policy not in SYSTEM_GROUP_POLICY.AWARE_OF_MEMBERSHIPS:
            return

        SyncAndPushMemberships.apply_async(
            kwargs={
                'system_id': self.system_id,
                'group_id': self.group_id,
            },
            countdown=settings.IDM_PLUGIN_TASK_COUNTDOWN,
        )

    def deprive_other_roles_if_exclusive(self, grant_action):
        if not self.node.is_exclusive or not self.is_user_role() or self.parent_id:
            return

        qs = (
            Role.objects
                .active()
                .filter(
                node_id=self.node_id,
                group=None,
                parent=None,
            )
                .exclude(pk=self.pk)
        )

        if self.fields_data and 'passport-login' in self.fields_data:
            fields = self.fields_data.copy()
            fields.pop('passport-login')
            query = {f'fields_data__{field}': value for field, value in fields.items()}
            qs = qs.filter(**query)
        else:
            qs = qs.filter(fields_data=self.fields_data)

        if waffle.switch_is_active('deprive_other_roles_if_exclusive'):
            qs.deprive_or_decline(
                User.objects.get_idm_robot(),
                bypass_checks=True,
                comment=_('Отзыв роли из-за выдачи новой роли %s на эксклюзивный узел' % self.get_url()),
                parent_action=grant_action,
            )
        else:
            log.info(
                'Trying to deprive %s because new role %s was added. '
                'But switch deprive_other_roles_if_exclusive is not active',
                list(qs.values_list('pk', flat=True)), self.pk,
            )

    def poke_awaiting_roles_async(self):
        if self.user is None or self.user.type != USER_TYPES.ORGANIZATION:
            return
        resource_id = self.fields_data.get('resource_id') if self.fields_data else None
        if resource_id is None:
            return

        from idm.core.tasks.roles import PokeAwaitingRoles

        fields = {'resource_id': resource_id}
        resource_type = self.fields_data.get('resource_type')
        if resource_type is not None:
            fields['resource_type'] = resource_type

        PokeAwaitingRoles.apply_async(
            kwargs={
                'system_id': self.system_id,
                'organization_id': self.organization_id,
                'fields': fields,
            }
        )

    def restore(self, comment):
        if self.state == ROLE_STATE.REQUESTED:
            with transaction.atomic():
                self.deprive_or_decline(
                    User.objects.get_idm_robot(),
                    bypass_checks=True,
                    comment=_('Отклонение в связи с восстановлением'),
                    force_deprive=True,
                )
                self.set_raw_state(ROLE_STATE.APPROVED)

        elif self.state == ROLE_STATE.DEPRIVED:
            try:
                self.set_raw_state(ROLE_STATE.APPROVED)
            except RoleAlreadyExistsError:
                # все RETURNABLE_STATES кроме REQUESTED нас устраивают
                alike_roles = self.get_alike(among_states=[ROLE_STATE.REQUESTED])
                if alike_roles.exists():
                    alike_roles.deprive_or_decline(
                        User.objects.get_idm_robot(),
                        bypass_checks=True,
                        comment=_('Отклонение роли в связи с восстановлением %s' % self.get_url()),
                        parent_action=None,
                    )
                    return self.restore(comment)
                else:
                    return

        elif self.state == ROLE_STATE.DEPRIVING:
            self.set_raw_state(ROLE_STATE.GRANTED)

        else:
            raise ValueError('Wrong state %s' % self.state)

        self.actions.create(
            user=self.user,
            group=self.group,
            requester=User.objects.get_idm_robot(),
            action=ACTION.RESTORE,
            system=self.system,
            data={'comment': comment},
        )

    def is_force_deprive(self, action, requester):
        """Метод определяет отозвать роль сразу или оставить в статусе depriving на IDM_DEPRIVING_AFTER_MIN минут"""
        # ToDo: добавить условий на отзыв роли с форсом
        # Если явно передали флаг force_deprive
        if action.data and action.data.get(ACTION_DATA_KEYS.FORCE_DEPRIVE):
            return True
        # Роль уже прошла валидацию, так что можно попробовать отозвать ее еще раз
        if self.state == ROLE_STATE.DEPRIVING:
            return True
        # Если пользователь уволен
        if self.user and self.user.is_active is False:
            return True
        # Если отзывающий - владелец роли, но только для персональной роли
        if self.user and (requester is not None and requester.impersonated == self.user):
            return True
        # Если родительская роль отозвана
        if self.parent is not None and self.parent.state == ROLE_STATE.DEPRIVED:
            return True
        # Если флаг на отложеный отзыв не включен
        if not waffle.switch_is_active('idm.deprive_not_immediately'):
            return True
        return False

    @staticmethod
    def calc_expiration(ttl_date: timezone.datetime,
                        ttl_days: int,
                        relative_to: timezone.datetime = None) -> timezone.datetime:
        """
        Вычислить дату, когда истечет первый из ttl_date (абсолютный) и ttl_days (относительный)
        :param relative_to: относительно чего считать ttl_days
        """
        if relative_to is None:
            relative_to = timezone.now()
        if ttl_days:
            expiration_date = relative_to + timezone.timedelta(days=ttl_days)
            if ttl_date and ttl_date < expiration_date:
                return ttl_date
            return expiration_date
        else:
            return ttl_date

    def get_personal_granted_at(self, username: Optional[str] = None) -> Optional[datetime.datetime]:
        if self.state != ROLE_STATE.GRANTED:
            return
        if self.is_user_role() or self.is_personal_role():
            return self.granted_at
        if self.is_group_role():
            if not username:
                return self.granted_at

            is_direct_filter = Q(is_direct__isnull=True) | Q(is_direct=False)
            if self.with_inheritance is False:
                is_direct_filter = Q(is_direct=True)

            membership = GroupMembership.objects.filter(
                is_direct_filter,
                group=self.group,
                user__username=username,
                state__in=GROUPMEMBERSHIP_STATE.ACTIVE_STATES,
            ).order_by('date_joined').first()
            if membership:
                return max(self.granted_at, membership.date_joined)


@receiver(signals.role_changed_state)
def export_role_to_tirole(role: Role, from_state: str, to_state: str, **_):
    if (
        from_state != to_state
        and to_state in {ROLE_STATE.GRANTED, ROLE_STATE.DEPRIVED}
        and role.group_id is None
        and role.system.export_to_tirole
    ):
        transaction.on_commit(functools.partial(
            events.add_event,
            event_type=events.EventType.YT_EXPORT_REQUIRED,
            system_id=role.system_id,
            role_id=role.id,
        ))
