from django.db import models
from django.utils.translation import ugettext_lazy as _

from idm.core import canonical
from idm.core.querysets.nodealias import RoleAliasManager
from idm.framework.fields import StrictForeignKey
from idm.framework.mixins import LocalizedModel


class RoleAlias(models.Model, LocalizedModel):
    DEFAULT_ALIAS = 'default'
    FIREWALL_ALIAS = 'firewall'
    ALIAS_TYPES = (
        (DEFAULT_ALIAS, _('Обычный синоним')),
        (FIREWALL_ALIAS, _('Правило firewall')),
    )
    created_at = models.DateTimeField(_('Дата создания'), auto_now_add=True, editable=False, null=True)
    updated_at = models.DateTimeField(_('Дата обновления'), auto_now=True, editable=False, null=True)
    removed_at = models.DateTimeField(_('Дата удаления'), editable=False, null=True)

    node = StrictForeignKey('core.RoleNode', null=False, blank=False, verbose_name=_('Узел дерева ролей'),
                             related_name='aliases', on_delete=models.CASCADE)
    type = models.CharField(_('Тип синонима'), max_length=50, db_index=True, null=True)
    name = models.CharField(_('Описание'), max_length=255, default='', blank=True)
    name_en = models.CharField(_('Описание (по-английски)'), max_length=255, default='', blank=True)
    is_active = models.BooleanField(_('Действующий синоним'), default=True)

    objects = RoleAliasManager()

    class Meta:
        verbose_name = _('Синоним роли')
        verbose_name_plural = _('Синонимы роли')
        db_table = 'upravlyator_rolealias'

    def __str__(self):
        return '%s: %s' % (self.type, self.name)

    def as_canonical(self):
        return canonical.CanonicalAlias(
            name=self.name,
            name_en=self.name_en,
            type=self.type
        )

    @classmethod
    def from_canonical(cls, canonical_obj):
        return cls(
            type=canonical_obj.type,
            name=canonical_obj.name,
            name_en=canonical_obj.name_en,
        )

    def natural_key(self):
        return self.node.natural_key(), self.name_en

    def get_name(self, lang=None):
        return self.get_localized_field('name', lang)

    def as_api(self):
        result = {
            'type': self.type,
            'name': {
                'en': self.name_en,
                'ru': self.name,
            }
        }
        return result
