import collections
import json
import logging
from operator import itemgetter
from typing import Optional, Union, List

import attr
import waffle
from django import forms
from django.conf import settings
from django.db import models, transaction
from django.db.models import NOT_PROVIDED, Q
from django.db.models.functions import Least
from django.dispatch import receiver
from django.utils import timezone
from django.utils.translation import ugettext_lazy as _, ugettext
from django_pgaas import atomic_retry
from intrasearch_fetcher.exceptions import IntrasearchFetcherException

from idm.api.exceptions import InternalAPIError
from idm.closuretree.models import closure_model_save
from idm.core import signals, canonical
from idm.core.constants.instrasearch import INTRASEARCH_METHOD, INTRASEARCH_STATE
from idm.core.constants.node_relocation import RELOCATION_STATE
from idm.core.constants.rolenode import ROLENODE_STATE
from idm.core.fetchers import RoleNodeFetcher
from idm.core.models import Action
from idm.core.querysets.node import RoleNodeManager
from idm.core.queues import RoleNodeQueue
from idm.core.utils import get_localized_node_name, get_localized_node_description, humanize_node
from idm.core.workflow.exceptions import DataValidationError, RoleAlreadyExistsError
from idm.framework.fields import StrictForeignKey, NullJSONField as JSONField
from idm.framework.mixins import LocalizedModel, RefreshFromDbWithoutRelatedMixin
from idm.framework.utils import add_to_instance_cache
from idm.nodes.tracking import ExternalTrackingNode
from idm.sync import intrasearch

log = logging.getLogger(__name__)


class RoleNode(RefreshFromDbWithoutRelatedMixin, LocalizedModel, ExternalTrackingNode):
    created_at = models.DateTimeField(_('Дата создания'), editable=False, null=True, auto_now_add=True)
    updated_at = models.DateTimeField(_('Дата обновления'), editable=False, null=True, auto_now=True, db_index=True)
    depriving_at = models.DateTimeField(_('Дата удаления'), editable=False, null=True, db_index=True)

    system = StrictForeignKey(
        'core.System', verbose_name=_('Система'), related_name='nodes', null=False, blank=False,
        on_delete=models.CASCADE,
    )

    name = models.CharField(_('Имя роли'), max_length=1000, default='', db_index=True)
    name_en = models.CharField(_('Имя роли (на английском)'), max_length=1000, default='', db_index=True)
    data = JSONField(verbose_name=_('Данные роли'), blank=True, null=True, db_index=True)
    fullname = JSONField(verbose_name=_('Полные данные об имени роли'), blank=True, null=True)
    description = models.TextField(_('Описание'), default='', blank=True)
    description_en = models.TextField(_('Описание (на английском)'), default='', blank=True)
    is_public = models.BooleanField(_('Публичная роль'), default=True, db_index=True)
    is_auto_updated = models.BooleanField(_('Это поддерево автоматически обновляется'), default=False, db_index=True)
    is_key = models.BooleanField(_('Этот узел является ключом при выборе роли'), default=True, db_index=True)
    is_exclusive = models.BooleanField(_('На этот узел можно запросить только одну роль на человека'), default=False,
                                       db_index=True)
    nodeset = StrictForeignKey('core.RoleNodeSet', verbose_name=_('Группа ролей'), null=True, blank=True,
                               related_name='nodes', on_delete=models.CASCADE)
    unique_id = models.CharField(_('Уникальный идентификатор узла'), max_length=255, blank=True, default='',
                                 db_index=True)
    review_required = models.BooleanField(_('Требуется ревью'), null=True)
    comment_required = models.BooleanField(_('Требуется комментарий при запросе роли'), default=False)

    # Полный путь по всем slug-ам
    slug_path = models.TextField(_('Путь по слагам ролей'), default='', blank=True, db_index=True)
    # Путь по слагам узлов, у которых is_key=False
    value_path = models.TextField(_('Путь по слагам value-полей'), default='', blank=True)
    relocation_state = models.CharField(
        verbose_name=_('Состояние релокации'),
        choices=RELOCATION_STATE.NAMES,
        default=None,
        null=True,
        max_length=15,
        db_index=True,
        blank=True,
    )
    isearch_state = models.CharField(
        verbose_name=_('Состояние пуша в intrasearch'),
        choices=INTRASEARCH_STATE.NAMES,
        default=None,
        null=True,
        max_length=5,
        db_index=True,
        blank=True,
    )
    need_isearch_push_method = models.CharField(
        verbose_name=_('Надо допушить в intrasearch'),
        choices=INTRASEARCH_METHOD.NAMES,
        default=None,
        null=True,
        max_length=10,
        db_index=True,
        blank=True,
    )

    slug_path_ok_after_push_at = models.DateTimeField(
        null=True, blank=True, verbose_name=_('Время, когда можно запросить роль после пуша'), db_index=True,
    )
    slug_path_ok_after_move_at = models.DateTimeField(
        null=True, blank=True, verbose_name=_('Время, когда можно запросить роль после перемещения'), db_index=True
    )
    suggest_ok_after_push_at = models.DateTimeField(
        null=True, blank=True, verbose_name=_('Время, когда можно найти узел в саджесте после пуша'), db_index=True,
    )
    suggest_ok_after_move_at = models.DateTimeField(
        null=True, blank=True, verbose_name=_('Время, когда можно найти узел в саджесте после перемещения'),
        db_index=True
    )
    pushed_at = models.DateTimeField(null=True, blank=True, verbose_name=_('Время создания через пуш'), db_index=True)
    moved_at = models.DateTimeField(null=True, blank=True, verbose_name=_('Время перемещения через пуш'), db_index=True)
    deprive_emailed_at = models.DateTimeField(
        null=True, blank=True, verbose_name=_('Время последней отправки письма про удаление')
    )

    objects = RoleNodeManager()
    fetcher = RoleNodeFetcher()
    queue_class = RoleNodeQueue
    EXTERNAL_ID_FIELD = 'unique_id'
    REQUIRE_EXTERNAL_ID = False
    SUPPORTS_RESTORE = True

    class Meta:
        verbose_name = _('Узел дерева ролей')
        verbose_name_plural = _('Узлы дерева ролей')
        indexes = [
            models.Index(
                fields=('depriving_at', 'system'),
                name='depriving_at_idx',
                condition=Q(state='depriving'),
            ),
            # idm/core/migrations/0067_rolenode_text_search.py GIN Index
        ]
        index_together = (
            ('state', 'level', 'is_public'),
            ('system', 'level', 'id'),
            ('is_key', 'state', 'is_public'),
            ('is_key', 'state', 'is_public', 'updated_at'),
            ('is_key', 'level', 'state'),
            ('level', 'slug', 'state'),
            # саджест
            ('state', 'level', 'is_public', 'name'),
            ('state', 'level', 'is_public', 'name_en'),
            ('state', 'slug_path'),
            # IDM-5016: Ускорить get_owners
            ('slug', 'system', 'state'),
        )
        db_table = 'upravlyator_rolenode'

    def __str__(self):
        return self.name

    @property
    def self_path(self):
        return '/' + self.system.slug + self.slug_path

    @property
    def parent_path(self):
        # При изменении учесть, что используемые здесь поля джоинятся в v1 RoleNodeResource
        if self.level == 0:
            return None
        else:
            parent_path = self.parent.slug_path
            if not self.is_key:
                parent_path = self.parent.slug_path[:-(len(self.parent.slug) + 1)]

        self.fetch_system()
        return '/' + self.system.slug + parent_path

    def rebuild_closure_tree(self):
        self._closure_change_init()
        if self.parent and self.level != self.parent.level + 1:
            self.save(update_fields=['level'])
        else:
            closure_model_save(sender=self.__class__, instance=self, created=False)

    def natural_key(self):
        return self.slug_path

    def is_pushable(self):
        return not self.is_key

    def send_intrasearch_push_for_descendants(self, method, action_id=None):
        from idm.core.tasks.intrasearch_push import PushItemChange
        if not settings.IDM_SEND_INTRASEARCH_PUSHES:
            return
        node_ids = list(
            self.get_descendants(include_self=True)
                .filter(is_key=False)
                .values_list('id', flat=True)
        )
        transaction.on_commit(lambda: PushItemChange.delay(
            step='bulk_init',
            action_id=action_id,
            node_ids=node_ids,
            method=method,
        ))

    def send_intrasearch_push(self, method, action_id=None, block=False):
        from idm.core.tasks.intrasearch_push import PushItemChange
        if not settings.IDM_SEND_INTRASEARCH_PUSHES or not self.is_pushable():
            return
        task = PushItemChange if block else PushItemChange.delay
        transaction.on_commit(lambda: task(
            action_id=action_id,
            node_id=self.pk,
            method=method,
        ))

    def as_intrasearch_push(self, method):
        from idm.api.v1.rolenode import RoleNodeResource

        if INTRASEARCH_METHOD.should_serialize(method):
            resource = RoleNodeResource()
            bundle = resource.build_bundle(obj=self, request=None)
            bundle = resource.full_dehydrate(bundle)
            bundle = resource.alter_detail_data_to_serialize(None, bundle)
            data = resource.serialize(None, bundle, 'application/json')
        else:
            data = json.dumps({
                'system': {
                    'slug': self.system.slug
                },
                'id': self.id,
                'slug_path': self.slug_path,
                'updated_at': self.updated_at.isoformat(),
            })
        return data

    def as_api(self):
        result = {
            'slug': self.slug,
            'slug_path': self.slug_path,
            'name': {
                'ru': self.name,
                'en': self.name_en,
            },
            'help': {
                'ru': self.description,
                'en': self.description_en,
            },
            'visibility': self.is_public,
        }
        if self.nodeset_id is not None:
            result['set'] = self.nodeset.set_id
        result['fields'] = [field.as_api() for field in self.fields.all() if field.is_active]
        result['aliases'] = [alias.as_api() for alias in self.aliases.all() if alias.is_active]
        result['responsibilities'] = [responsibility.as_api() for responsibility in self.responsibilities.all()
                                      if responsibility.is_active]
        return result

    def as_canonical(self):
        fields = [field.as_canonical() for field in self.fields.all() if field.is_active]
        aliases = [alias.as_canonical() for alias in self.aliases.all() if alias.is_active]
        responsibilities = [responsibility.as_canonical() for responsibility in self.responsibilities.all()
                            if responsibility.is_active]

        self.fetch_nodeset()
        canonical_node = canonical.CanonicalNode(
            slug=self.slug,
            name=self.name,
            name_en=self.name_en,
            description=self.description,
            description_en=self.description_en,
            is_public=self.is_public,
            is_exclusive=self.is_exclusive,
            unique_id=self.unique_id,
            review_required=self.review_required,
            comment_required=self.comment_required or False,
            set=self.nodeset.set_id if self.nodeset else '',
            fields={field.as_key(): field for field in fields},
            aliases={alias.as_key(): alias for alias in aliases},
            responsibilities={responsibility.as_key(): responsibility for responsibility in responsibilities},

            hash=self.hash,
        )
        return canonical_node

    @classmethod
    def from_canonical(cls, canonical_node, system):
        from idm.core.models import RoleNodeSet

        nodeset = None
        if canonical_node.set:
            nodeset = RoleNodeSet.objects.restore_or_create(
                system,
                canonical_node.set,
                canonical_node.name,
                canonical_node.name_en
            )
        node = cls(
            slug=canonical_node.slug,
            name=canonical_node.name,
            name_en=canonical_node.name_en,
            description=canonical_node.description,
            description_en=canonical_node.description_en,
            is_public=canonical_node.is_public,
            is_exclusive=canonical_node.is_exclusive,
            unique_id=canonical_node.unique_id,
            review_required=canonical_node.review_required,
            comment_required=canonical_node.comment_required,
            nodeset=nodeset,
            hash=canonical_node.hash,

            system=system,

        )
        return node

    def gen_slug_path(self):
        return f'{self.parent.slug_path}{self.slug}{self.SEP}'

    def calc_fields(self):
        changed = False
        new_fields = {
            'slug_path': '/',
            'data': {},
            'value_path': '/',
            'fullname': [],
        }

        if self.parent is not None:
            new_fields['slug_path'] = self.gen_slug_path()
            new_fields['data'] = self.parent.data.copy()
            if self.is_key:
                new_fields['value_path'] = self.parent.value_path
            else:
                new_fields['data'][self.parent.slug] = self.slug
                new_fields['value_path'] = f'{self.parent.value_path}{self.slug}{self.SEP}'
            new_fields['fullname'] = self.parent.fullname[:]
            new_fields['fullname'].append({
                'name': self.name,
                'name_en': self.name_en,
                'description': self.description,
                'description_en': self.description_en
            })

        for field, value in new_fields.items():
            if getattr(self, field) != value:
                setattr(self, field, value)
                changed = True

        return changed

    def save(self, force_insert=False, force_update=False, using=None, update_fields=None, recalc_fields=False):
        if update_fields is None or recalc_fields:
            if self.calc_fields() and update_fields is not None:
                update_fields.extend(['slug_path', 'data', 'value_path', 'fullname'])

        return super(RoleNode, self).save(force_insert=force_insert, force_update=force_update, using=using,
                                          update_fields=update_fields)

    def find_matching_child(self, canonical, nodes):
        match = None
        if canonical.unique_id:
            for node in nodes:
                if node.unique_id == canonical.unique_id:
                    match = node
                    break
        if match:
            return match
        for node in nodes:
            if node.slug == canonical.slug:
                match = node
                break
        return match

    def get_queue_instance(self, **kwargs):
        queue = self.queue_class(system=kwargs.get('system'))
        return queue

    def compare(self, data, in_auto_mode=False, **kwargs):
        in_auto_mode = self.is_auto_updated or in_auto_mode or kwargs.get('force_update')
        queue = super(RoleNode, self).compare(data, in_auto_mode=in_auto_mode, **kwargs)
        return queue

    def reset_hash(self):
        self.hash = ''
        self.save(update_fields=('hash',))

    def move(self, new_parent, requester):
        action = self.actions.create(
            action='role_node_moved',
            system=self.system,
            requester=requester.impersonated,
            impersonator=requester.impersonator,
            role_node=self,
            data={
                'from': self.parent.slug_path,
                'to': new_parent.slug_path,
            }
        )

        # Если пуши на добавление нод после перемещения узла отработают раньше (в пайплайне),
        # мы рискуем удалить актуальные ноды из индекса
        self.send_intrasearch_push_for_descendants(INTRASEARCH_METHOD.REMOVE, action.pk)

        self.parent.reset_hash()
        self.parent = new_parent
        self.relocation_state = RELOCATION_STATE.SUPERDIRTY
        # Не перестраиваем дерево здесь, т.к. это произойдёт в пайплайне для superdirty
        if self._closure_change_check():
            delattr(self, "_closure_old_parent")
        self.save(update_fields=['relocation_state', 'parent', 'level'])

    def set_relocation_state(self, state):
        assert state in RELOCATION_STATE.STATUSES
        self.relocation_state = state
        self.save(update_fields=['relocation_state'])

    def get_valid_fields_data(self, fields_data=None, user=None, group=None):
        """Проверяем данные в полях на валидность, удаляем лишние поля"""

        if fields_data is None:
            fields_data = {}
        validation_form = self.build_form(fields_data, user, group)
        if not validation_form.is_valid():
            raise DataValidationError(_('Некоторые поля заполнены некорректно'), validation_form.errors)

        # Джанговые формы возвращают в cleaned_data пустые строки или другие нулевые значения,
        # если в data нет ключа, а поле необязательное. Чтобы не переписывать половину логики,
        # мы убираем такие поля из cleaned_data.
        # Также, если поле необязательное и blank_allowed есть в опциях поля,
        # то его clean() возвращает NOT_PROVIDED - такие поля мы тоже удаляем из результирующего dict-а
        cleaned_data = {
            slug: value
            for slug, value in list(validation_form.cleaned_data.items())
            if slug in fields_data and fields_data[slug] is not None and value is not NOT_PROVIDED
        }
        if not cleaned_data:
            cleaned_data = None  # Возвращаем всегда None, а не словарь, для единообразия
        return cleaned_data

    def add_fields(self, fields: List[canonical.CanonicalField], requester: 'User', sync_key: Action):
        from idm.core.models import RoleField

        to_create = []
        to_update = []
        update_fields = ('is_required', 'name', 'name_en', 'options', 'dependencies')
        db_fields_by_key = {(db_field.type, db_field.slug): db_field for db_field in self.fields.all()}
        for field in fields:
            if (db_field := db_fields_by_key.get((field.type, field.slug))) is not None:
                db_field.is_required = field.is_required
                db_field.name = field.name
                db_field.name_en = field.name_en
                db_field.options = field.options
                db_field.dependencies = field.dependencies
                to_update.append(db_field)
            else:
                db_field = RoleField.from_canonical(field)
                db_field.node = self
                to_create.append(db_field)

        if to_create:
            RoleField.objects.bulk_create(to_create, batch_size=settings.BULK_BATCH_SIZE)
        if to_update:
            RoleField.objects.bulk_update(to_create, fields=update_fields, batch_size=settings.BULK_BATCH_SIZE)
        if db_fields := to_create + to_update:
            Action.objects.bulk_create([
                Action(
                    role_field=db_field,
                    action='role_node_field_created',
                    role_node=self,
                    system_id=self.system_id,
                    requester=requester,
                    parent=sync_key,
                    data={'field_data': attr.asdict(db_field.as_canonical())}
                )
                for db_field in db_fields
            ], batch_size=settings.BULK_BATCH_SIZE)
            signals.fields_added.send(sender=self, fields=db_fields, role_node=self, requester=requester)

    def change_fields(self, modifications, requester, sync_key):
        from idm.core.models import Action, RoleField

        modifications_by_key = {modification.modified_value.as_key(): modification for modification in modifications}
        db_fields_by_key = {
            (db_field.type, db_field.slug): db_field
            for db_field in self.fields.filter(is_active=True).all()
        }
        action_data = {}
        to_update = []
        update_fields = set()
        for key, db_field in db_fields_by_key.items():
            diff = {}
            key = db_field.as_canonical().as_key()

            if (modification := modifications_by_key.get(key)) is None:
                continue

            for item in modification.flat:
                diff[item.name] = [item.old, item.new]
                update_fields.add(item.name)
                setattr(db_field, item.name, item.new)

            action_data[key] = {
                'diff': diff,
                'field_data': modification.new_value.as_dict(),
            }
            to_update.append(db_field)

        if to_update and update_fields:
            RoleField.objects.bulk_update(to_update, fields=update_fields, batch_size=settings.BULK_BATCH_SIZE)
            Action.objects.bulk_create([
                Action(
                    role_field=db_field,
                    action='role_node_field_changed',
                    role_node=self,
                    system=self.system,
                    requester=requester,
                    parent=sync_key,
                    data=action_data[(db_field.type, db_field.slug)]
                )
                for db_field in to_update
            ],
                batch_size=settings.BULK_BATCH_SIZE,
            )
            signals.fields_changed.send(sender=self, fields=to_update, role_node=self, requester=requester)

    def remove_fields(
        self,
        fields: Union[str, canonical.CanonicalField],
        requester: 'User', sync_key: 'Action',
        action_data: dict = None
    ):
        from idm.core.models import RoleField
        now = timezone.now()

        action_data = action_data or {}
        actions = []
        db_fields_by_key = {(db_field.type, db_field.slug): db_field for db_field in self.fields.filter(is_active=True)}
        if fields == '*':
            to_update = db_fields_by_key.values()
        else:
            to_update = []
            for field in fields:
                if (db_field := db_fields_by_key.get(field.as_key())) is not None:
                    to_update.append(db_field)

        for db_field in to_update:
            db_field.is_active = False
            db_field.removed_at = now
        if to_update:
            RoleField.objects.bulk_update(to_update, fields=('is_active',), batch_size=settings.BULK_BATCH_SIZE)
            Action.objects.bulk_create([
                Action(
                    action='role_node_field_removed',
                    system=self.system,
                    requester=requester,
                    data=action_data,
                    role_field=db_field,
                    role_node=self,
                    parent=sync_key,
                )
                for db_field in to_update
            ], batch_size=settings.BULK_BATCH_SIZE)
            signals.fields_removed.send(sender=self, fields=to_update, role_node=self, requester=requester)

    def add_responsibilities(
        self,
        responsibilities: List[canonical.CanonicalResponsibility],
        requester: 'User',
        sync_key: Action,
    ):
        from idm.core.models import RoleNodeResponsibilityAction, NodeResponsibility

        to_create = []
        to_update = []
        update_fields = ('notify', 'removed_at', 'is_active')
        db_responsibilities_by_key = {
            db_responsibility.user.username: db_responsibility
            for db_responsibility in self.responsibilities.select_related('user').all()
        }
        for responsibility in responsibilities:
            if (db_responsibility := db_responsibilities_by_key.get(responsibility.as_key())) is not None:
                db_responsibility.notify = responsibility.notify
                db_responsibility.removed_at = None
                db_responsibility.is_active = True
                to_update.append(db_responsibility)
            else:
                to_create.append(
                    NodeResponsibility(
                        user=responsibility.user,
                        node=self,
                        notify=responsibility.notify,
                        is_active=True,
                        removed_at=None,
                    )
                )

        if to_create:
            NodeResponsibility.objects.bulk_create(to_create, batch_size=settings.BULK_BATCH_SIZE)
        if to_update:
            NodeResponsibility.objects.bulk_update(to_update, fields=update_fields, batch_size=settings.BULK_BATCH_SIZE)
        if db_responsibilities := to_create + to_update:
            RoleNodeResponsibilityAction.objects.bulk_create([
                RoleNodeResponsibilityAction(
                    action='role_node_responsibility_created',
                    role_node=self,
                    node_responsibility=db_responsibility,
                    system_id=self.system_id,
                    requester=requester,
                    parent=sync_key,
                )
                for db_responsibility in db_responsibilities
            ], batch_size=settings.BULK_BATCH_SIZE)

            signals.responsibilities_added.send(
                sender=self,
                responsibilities=db_responsibilities,
                role_node=self,
                requester=requester
            )

    def change_responsibilities(self, modifications: 'DiffSet', requester: 'User', sync_key: 'Action'):
        from idm.core.models import RoleNodeResponsibilityAction, NodeResponsibility

        modifications_by_key = {modification.modified_value.as_key(): modification for modification in modifications}
        db_responsibilities_by_key = {
            db_responsibility.user.username: db_responsibility
            for db_responsibility in (
                self.responsibilities
                    .select_related('user')
                    .filter(is_active=True).all()
            )
        }

        action_data = {}
        to_update = []
        update_fields = set()
        for key, db_responsibility in db_responsibilities_by_key.items():
            diff = {}

            if (modification := modifications_by_key.get(db_responsibility.user.username)) is None:
                continue

            for item in modification.flat:
                diff[item.name] = [item.old, item.new]
                update_fields.add(item.name)
                setattr(db_responsibility, item.name, item.new)

            action_data[key] = {'diff': diff}
            to_update.append(db_responsibility)
        if to_update and update_fields:
            NodeResponsibility.objects.bulk_update(to_update, fields=update_fields, batch_size=settings.BULK_BATCH_SIZE)

            RoleNodeResponsibilityAction.objects.bulk_create([
                RoleNodeResponsibilityAction(
                    action='role_node_responsibility_changed',
                    node_responsibility=db_responsibility,
                    role_node=self,
                    system=self.system,
                    requester=requester,
                    parent=sync_key,
                    data=action_data[db_responsibility.user.username]
                )
                for db_responsibility in to_update
            ], batch_size=settings.BULK_BATCH_SIZE)

            signals.responsibilities_changed.send(
                sender=self,
                responsibilities=to_update,
                role_node=self,
                requester=requester,
            )

    def remove_responsibilities(
        self,
        responsibilities: Union[str, List[canonical.CanonicalResponsibility]],
        requester: 'User',
        sync_key: 'Action',
        action_data: dict = None,
    ):
        from idm.core.models import RoleNodeResponsibilityAction, NodeResponsibility

        now = timezone.now()
        action_data = action_data or {}
        db_responsibilities_by_key = {
            db_responsibility.user.username: db_responsibility
            for db_responsibility in self.responsibilities.select_related('user').filter(is_active=True)
        }
        if responsibilities == '*':
            to_update = db_responsibilities_by_key.values()
        else:
            to_update = []
            for responsibility in responsibilities:
                if (db_responsibility := db_responsibilities_by_key[responsibility.as_key()]) is not None:
                    to_update.append(db_responsibility)

        for db_responsibility in to_update:
            db_responsibility.removed_at = now
            db_responsibility.is_active = False

        if to_update:
            NodeResponsibility.objects.bulk_update(
                to_update,
                fields=('is_active', 'removed_at'),
                batch_size=settings.BULK_BATCH_SIZE,
            )
            RoleNodeResponsibilityAction.objects.bulk_create([
                RoleNodeResponsibilityAction(
                    action='role_node_responsibility_removed',
                    system=self.system,
                    requester=requester,
                    data=action_data,
                    node_responsibility=db_responsibility,
                    role_node=self,
                    parent=sync_key,
                )
                for db_responsibility in to_update
            ], batch_size=settings.BULK_BATCH_SIZE)
            signals.responsibilities_removed.send(sender=self, responsibilities=to_update,
                                                  role_node=self, requester=requester)

    def add_aliases(self, aliases: List[canonical.CanonicalAlias], requester: 'User', sync_key: Action):
        from idm.core.models import RoleAlias

        action_data = {}
        to_create = []
        to_update = []
        update_fields = ('removed_at', 'is_active')
        db_aliases_by_key = {
            (db_alias.type, db_alias.name, db_alias.name_en): db_alias
            for db_alias in self.aliases.all()
        }
        for alias in aliases:
            if (db_alias := db_aliases_by_key.get(alias.as_key())) is not None:
                db_alias.removed_at = None
                db_alias.is_active = True
                to_update.append(db_alias)
            else:
                to_create.append(
                    RoleAlias(
                        type=alias.type,
                        name=alias.name,
                        name_en=alias.name_en,
                        node=self,
                        is_active=True,
                        removed_at=None
                    )
                )
            action_data[alias.as_key()] = {'alias_data': attr.asdict(alias)}

        if to_create:
            RoleAlias.objects.bulk_create(to_create, batch_size=settings.BULK_BATCH_SIZE)
        if to_update:
            RoleAlias.objects.bulk_update(to_update, fields=update_fields, batch_size=settings.BULK_BATCH_SIZE)
        if db_aliases := to_create + to_update:
            Action.objects.bulk_create([
                Action(
                    role_alias=db_alias,
                    action='role_node_alias_created',
                    role_node=self,
                    system_id=self.system_id,
                    requester=requester,
                    parent=sync_key,
                    data=action_data[(db_alias.type, db_alias.name, db_alias.name_en)]
                )
                for db_alias in db_aliases
            ], batch_size=settings.BULK_BATCH_SIZE
            )

            signals.aliases_added.send(sender=self, aliases=db_aliases, requester=requester)

    def remove_aliases(
        self,
        aliases: Union[str, List[canonical.CanonicalAlias]],
        requester: 'User',
        sync_key: Action,
        action_data: dict = None,
    ):
        from idm.core.models import Action, RoleAlias

        now = timezone.now()
        action_data = action_data or {}
        db_aliases_by_key = {
            (db_alias.type, db_alias.name, db_alias.name_en): db_alias
            for db_alias in self.aliases.filter(is_active=True).all()
        }
        if aliases == '*':
            to_update = db_aliases_by_key.values()
        else:
            to_update = []
            for alias in aliases:
                if (db_alias := db_aliases_by_key.get(alias.as_key())) is not None:
                    to_update.append(db_alias)

        for db_alias in to_update:
            db_alias.is_active = False
            db_alias.removed_at = now

        if to_update:
            RoleAlias.objects.bulk_update(
                to_update,
                fields=('is_active', 'removed_at'),
                batch_size=settings.BULK_BATCH_SIZE,
            )
            Action.objects.bulk_create([
                Action(
                    action='role_node_alias_removed',
                    requester=requester,
                    data=action_data,
                    role_alias=db_alias,
                    role_node=self,
                    system=self.system,
                    parent=sync_key,
                )
                for db_alias in to_update
            ], batch_size=settings.BULK_BATCH_SIZE)
            signals.aliases_removed.send(sender=self, aliases=to_update, role_node=self, requester=requester)

    def mark_depriving(self, requester=None, immediately=False):
        expiration = timezone.now()
        if not immediately:
            expiration += timezone.timedelta(settings.IDM_DEPRIVING_NODE_TTL)
        descendants = self.get_descendants(include_self=True)
        active_descendants_ids = set(descendants.active().values_list('id', flat=True))
        active_descendants = (
            RoleNode.objects
                .filter(id__in=active_descendants_ids)
                .select_related('system')
        )
        active_descendants.update(
            state=ROLENODE_STATE.DEPRIVING,
            hash='',
            depriving_at=Least(expiration, 'depriving_at'),
        )
        active_descendants.filter(is_key=False).update(need_isearch_push_method=INTRASEARCH_METHOD.REMOVE)
        self.actions.create(
            action='role_node_marked_depriving',
            system_id=self.system_id,
            role_node=self,
            requester=requester.impersonated if requester else None,
            impersonator=requester.impersonator if requester else None,
        )
        # Сбросим хеш предку, если он активен, так как при синке неактивные узлы не учитываются
        RoleNode.objects.active().filter(id=self.parent_id).update(hash='')

        # FIXME: пока отдаём только те ноды, которые перевели в depriving,
        #  чтобы можно было их сразу отозвать.
        #  Возможно, здесь имеет смысл отдавать всех потомков в state=depriving,
        #  но тогда нужен нормальный индекс в БД на это место
        return active_descendants

    def mark_active(self):
        self.state = ROLENODE_STATE.ACTIVE
        self.depriving_at = None
        if self.is_pushable():
            self.need_isearch_push_method = INTRASEARCH_METHOD.ADD
        self.save(update_fields=['state', 'depriving_at', 'need_isearch_push_method'])
        ancestors = self.get_ancestors(include_self=True)
        ancestors.update(hash='')

    def as_split_path(self):
        # первый и последний элементы пустые, так как путь начинается и оканчивается на /
        return self.value_path.split('/')[1:-1]

    @property
    def is_value(self):
        return not self.is_key

    def is_requestable(self):
        return self.is_value and not self.is_root_node() and not self.get_grandchildren().exists()

    def get_aliases(self, type=None):
        aliases = self.aliases.filter(is_active=True)
        if type is not None:
            aliases = aliases.filter(type=type)
        return aliases

    def get_responsibilities(self):
        return self.responsibilities.filter(is_active=True, user__is_active=True).order_by('user__username')

    def get_fields(self, fields_data=NOT_PROVIDED):
        from idm.core.models import RoleField

        ancestors = self.get_ancestors(include_self=True)
        role_fields = (
            RoleField.objects
                .filter(node__in=ancestors, is_active=True)
                .select_related('node')
                .order_by('node__level')
        )

        fields_dict = {}
        for role_field in role_fields:
            if role_field.type == 'undo' and role_field.slug in fields_dict:
                del fields_dict[role_field.slug]
            else:
                if role_field.slug in fields_dict:
                    raise ValueError('There are more than one field with slug %s, and they clash', role_field.slug)
                fields_dict[role_field.slug] = role_field

        fields = [item[1] for item in sorted(fields_dict.items(), key=itemgetter(0))]
        if fields_data is NOT_PROVIDED:
            return fields

        fields = self.check_dependencies(fields, fields_data)
        return fields

    def check_dependencies(self, fields, fields_data):
        dependencies = [(field.slug, field.get_dependent_fieldnames()) for field in fields]
        fields_dict = {field.slug: field for field in fields}
        sorted_deps = []
        unsorted_deps = collections.deque(dependencies)
        while unsorted_deps:
            slug, deps = unsorted_deps.popleft()
            if all((dep in sorted_deps for dep in deps)):
                sorted_deps.append(slug)
            else:
                unsorted_deps.append((slug, deps))
        fields_list = []
        for slug in sorted_deps:
            field = fields_dict[slug]
            if field.check_dependencies(fields_data):
                fields_list.append(field)
        return fields_list

    def get_descendants(self, include_self=False, depth=None, exact_depth=None):
        # Пришлось перенести сюда код из базового метода, потому что в данном случае Django
        # делает по джойну на каждый вызов filter()
        refname = self._closure_childref()

        params = {'%s__parent' % refname: self.pk}
        if depth is not None:
            params['%s__depth__lte' % refname] = depth
        elif exact_depth is not None:
            params['%s__depth' % refname] = exact_depth
        descendants = self._toplevel().objects.filter(**params)
        if not include_self:
            descendants = descendants.exclude(pk=self.pk)
        super_qs = descendants.order_by('%s__depth' % self._closure_childref())
        qs = super_qs.filter(system_id=self.system_id)
        return qs

    def get_name(self, lang=None):
        return get_localized_node_name(self, lang)

    def get_description(self, lang=None):
        return get_localized_node_description(self, lang)

    def get_grandchildren(self):
        grandchildren = type(self).objects.filter(parent__parent=self, level=self.level + 2)
        return grandchildren.filter(state__in=self.ACTIVE_STATES)

    def get_public_grandchildren(self):
        return self.get_grandchildren().public()

    def search_grandchildren(self, search_term):
        use_proxied = waffle.switch_is_active('idm.use_intrasearch_for_roles')

        if use_proxied:
            fetcher = intrasearch.IntrasearchFetcher('idm_rolenodes')

            filters = {
                's_parent_path': self.self_path,
            }

            try:
                nodes = fetcher.search(
                    text=search_term,
                    filters=filters,
                    per_page=settings.IDM_INTRASEARCH_ROLE_SEARCH_PER_PAGE,
                )
            except IntrasearchFetcherException as err:
                raise InternalAPIError(*err.args)

            slug_paths = [node['fields']['slug_path'] for node in nodes]

            grandchildren = RoleNode.objects.active().filter(slug_path__in=slug_paths)
        else:
            grandchildren = self.get_public_grandchildren().text_search(search_term, with_aliases=True)

        return list(grandchildren.values_list('id', flat=True))

    def build_path(self, for_parent=Ellipsis):
        return self.unique_id

    def humanize(self, lang=None, format='full'):
        return humanize_node(self, lang, format)

    def build_form(self, fields_data, user, group):
        from idm.core.models import RoleField

        assert bool(user) != bool(group)  # aka XOR
        ignored_types = RoleField.IGNORED_USER_FIELD_TYPES if user else RoleField.IGNORED_GROUP_FIELD_TYPES
        node_fields = self.get_fields(fields_data)
        form_class = type('ValidationForm', (forms.Form,), {
            field.slug: field.as_formfield()
            for field in node_fields
            if field.type not in ignored_types
        })
        return form_class(data=fields_data)

    def role_should_have_login(self, is_required: Optional[bool] = None) -> bool:
        if is_required is None:
            conditions = (False, True)
        else:
            conditions = (is_required,)

        node_fields = [field.slug for field in self.get_fields() if field.is_required in conditions]

        # мы считаем, что паспортный логин может лежать только в поле со слагом passport-login
        # другие поля типа passportlogin мы игнорируем
        return 'passport-login' in node_fields

    def get_short_name(self, lang=None):
        """
        Возвращает описание роли в коротком формате
        """
        return self.humanize(lang=lang, format='short')

    def deprive(self, requester, sync_key=None, from_api=False):
        from idm.core.tasks import RoleNodeRemoved

        if self.state != ROLENODE_STATE.DEPRIVING:
            return False

        action = None
        previous_actions = self.actions.filter(action='role_node_deleted')
        if previous_actions.exists():
            action = previous_actions.order_by('-added', '-pk').first()

        if action is None:
            action_data = {}
            try:
                moved_to_node = self.moved_to
            except RoleNode.DoesNotExist:
                pass
            else:
                action_data['moved_to'] = {
                    'id': moved_to_node.pk,
                    'path': moved_to_node.slug_path
                }

            action_data['from_api'] = from_api

            action = self.actions.create(
                action='role_node_deleted',
                system=self.system,
                data=action_data,
                parent=sync_key,
                role_node=self,
                requester=requester.impersonated,
                impersonator=requester.impersonator,
            )
        RoleNodeRemoved.apply_async(
            kwargs={'action_id': action.pk,
                    'rolenode_id': self.pk,
                    'rolenode_path': self.slug_path,
                    'system_slug': self.system.slug,
                    },
            countdown=settings.IDM_NODE_REMOVAL_TASK_COUNTDOWN
        )

        return True

    def deprive_deleted_roles(self):
        """
        Отзывает выданые роли, отклоняет запрошенные и отказывает в выдаче подтвержденным ролям,
        которые удалены из системы (depriving_nodes).
        """
        roles = self.roles.returnable().select_related('parent')
        for role in roles.iterator():
            add_to_instance_cache(role, 'system', self.system)
            role.deprive_or_decline(depriver=None, bypass_checks=True, force_deprive=True,
                                    comment=ugettext('Роль больше не поддерживается системой'))

    # non-db side-effect is the last in the function
    @atomic_retry
    def deprive_roles_async(self, requester=None, from_api=False, reason=''):
        from idm.core.models import Action
        from idm.core.tasks import RoleNodeRolesDeprived

        action = Action.objects.create(
            action='role_node_roles_deprived',
            requester=requester,
            role_node=self,
            data={
                'reason': reason,
                'from_api': from_api,
            }
        )
        RoleNodeRolesDeprived.apply_async(
            kwargs={'action_id': action.id},
            countdown=settings.IDM_PLUGIN_TASK_COUNTDOWN,
        )


@receiver(signals.role_node_added)
def move_role_on_node_movement(role_node, moved_from, **kwargs):
    """Перемещаем роли вместе с узлом, но только если и новый узел, и старый узел – листовые.
    В противном случае оставляем их у старого узла, и когда он перейдёт в deprived, привязанные к
    нему роли отзовутся."""
    if moved_from and moved_from.is_requestable() and role_node.is_requestable():
        log.info('Moving all roles with the node %d (%s)', role_node.pk, role_node.slug_path)
        roles = moved_from.roles.select_related('parent')
        for role in roles:
            role.node = role_node
            try:
                role.save(update_fields=['node'])
            except RoleAlreadyExistsError:
                log.exception('Could not move role %d to node %d from node %d:'
                              'a matching role in returnable or created state already exists',
                              role.pk, role_node.pk, moved_from.pk)


@receiver(signals.role_node_added)
def update_pushed_at_and_moved_at(role_node, moved_from, action, **kwargs):
    if action.data['from_api']:
        if moved_from:
            role_node.get_descendants(include_self=True).update(moved_at=timezone.now())
        else:
            role_node.pushed_at = timezone.now()
            role_node.save(update_fields=['pushed_at'])


@receiver(signals.role_node_modified_before)
def push_delete_if_slug_has_changed(role_node, diff, **kwargs):
    changed_fields = {field.name for field in diff.flat}
    if 'slug' in changed_fields:
        # пока нет действия, приходится полагаться на удачу
        # TODO: обновиться до on_commit
        role_node.send_intrasearch_push(INTRASEARCH_METHOD.REMOVE, None)


@receiver(signals.role_node_added)
def push_add_rolenode_to_intrasearch(role_node, moved_from, action, **kwargs):
    if role_node.is_public:
        role_node.send_intrasearch_push(INTRASEARCH_METHOD.ADD, action.pk)


@receiver(signals.role_node_modified)
def push_modify_rolenode_to_intrasearch(role_node, diff, action, **kwargs):
    changed_fields = {field.name for field in diff.flat}

    if 'is_public' in changed_fields:
        if role_node.is_public:
            role_node.need_isearch_push_method = INTRASEARCH_METHOD.ADD
            role_node.send_intrasearch_push(INTRASEARCH_METHOD.ADD, action.pk)
        else:
            role_node.need_isearch_push_method = INTRASEARCH_METHOD.REMOVE
            role_node.send_intrasearch_push(INTRASEARCH_METHOD.REMOVE, action.pk)
        role_node.save(update_fields=['need_isearch_push_method'])
    elif 'slug' not in changed_fields:
        # не шлём апдейтов при изменении слага (потому что пришлём позже сразу пачку)
        role_node.send_intrasearch_push(INTRASEARCH_METHOD.MODIFY, action.pk)


@receiver(signals.role_node_deprived)
def push_deprive_rolenode_to_intrasearch(role_node, moved_to, action, **kwargs):
    role_node.send_intrasearch_push(INTRASEARCH_METHOD.REMOVE, action.pk)
