from django.db import models
from django.utils.translation import ugettext_lazy as _

from idm.core.querysets.nodeset import RoleNodeSetManager
from idm.framework.fields import StrictForeignKey
from idm.framework.mixins import LocalizedModel


class RoleNodeSet(LocalizedModel, models.Model):
    """Группа узлов дерева ролей"""

    set_id = models.CharField(_('Идентификатор группы'), max_length=255, default='')
    system = StrictForeignKey('core.System', verbose_name=_('Система'), null=False, blank=False, on_delete=models.CASCADE)
    is_active = models.BooleanField(_('Флаг активности'), default=True)
    name = models.CharField(_('Имя'), default='', max_length=255)
    name_en = models.CharField(_('Имя по-английски'), default='', max_length=255)

    objects = RoleNodeSetManager()

    class Meta:
        unique_together = (
            ('set_id', 'system', 'is_active'),
        )
        index_together = (
            ('system', 'is_active'),
        )
        verbose_name = _('Набор узлов')
        verbose_name_plural = _('Наборы узлов')
        db_table = 'upravlyator_rolenodeset'

    def __str__(self):
        return 'Набор узлов %(set_id)s в системе "%(system)s", %(activity)s' % {
            'set_id': self.set_id,
            'system': self.system.get_name(lang='ru'),
            'activity': _('Активен') if self.is_active else _('Неактивен')
        }

    def get_name(self, lang=None):
        return self.get_localized_field('name', lang)
