import itertools
import logging

from django.conf import settings
from django.db import models
from django.db.models import NOT_PROVIDED
from django.template.loader import render_to_string
from django.utils.translation import ugettext_lazy as _, override
from ids.exceptions import BackendError

from idm.api.exceptions import InternalAPIError
from idm.core.constants.action import ACTION
from idm.core.constants.approverequest import APPROVEREQUEST_DECISION
from idm.framework.fields import StrictForeignKey
from idm.notification.utils import get_issues_repository
from idm.users.models import User

log = logging.getLogger(__name__)


class RoleRequest(models.Model):
    """Запрос Роли. Определяет аппрувы для Роли и общий статус завершания подтверждения"""
    REQUEST_REASONS = (
        ('unknown', _('Неизвестна')),
        ('request', _('Запрос роли')),
        ('rerequest', _('Перезапрос роли')),
        ('review_rerequest', _('Регулярный пересмотр')),
        ('import', _('Разрешение расхождения')),
    )

    role = StrictForeignKey(
        'core.Role', related_name='requests', verbose_name=_('роль'), null=False, on_delete=models.CASCADE,
    )
    requester = StrictForeignKey(
        'users.User',
        null=True, blank=False,
        related_name='role_requests',
        verbose_name=_('Запросивший'),
        on_delete=models.CASCADE,
    )
    workflow = StrictForeignKey('core.Workflow', related_name='workflow_requests', verbose_name=_('Workflow'),
                                 null=True, blank=True, default=None, on_delete=models.SET_NULL)
    is_done = models.BooleanField(verbose_name=_('Завершенность'), default=False)
    added = models.DateTimeField(auto_now_add=True, verbose_name=_('Дата добавления'))
    notify_everyone = models.BooleanField(default=False, verbose_name=_('Оповестить всех подтверждающих'))
    is_silent = models.BooleanField(default=False, verbose_name=_('Не оповещать владельца роли об этом запросе'))
    reason = models.CharField(choices=REQUEST_REASONS, default='unknown', max_length=20,
                              verbose_name=_('Причина запроса'))
    st_issue = models.TextField(blank=True, default='', verbose_name=_('Ключ тикета для обсуждения'))
    comment = models.TextField(blank=True, default='', verbose_name=_('Комментарий'))
    review_at = models.DateTimeField(null=True, default=None, verbose_name=_('Дата ближайшего пересмотра'))
    review_days = models.IntegerField(blank=True, null=True, default=None,
                                      verbose_name=_('Количество дней до ближайшего пересмотра'))

    class Meta:
        verbose_name = _('Запрос роли')
        verbose_name_plural = _('Запросы роли')
        db_table = 'upravlyator_rolerequest'

    def get_or_create_st_issue(self, actor):
        if self.st_issue:
            return self.st_issue

        requester = self.requester
        role = self.role
        subject = role.get_subject()
        if subject.is_user:
            subject.user.fetch_department_group()
        subject_responsibles = list(subject.get_responsibles(with_heads=True))
        approvers = list(self.get_approvers())
        responsibles_for_system = list(role.system.get_responsibles())
        access = (
                [requester]
                + subject_responsibles
                + approvers
                + responsibles_for_system
        )
        access = [user.username for user in access]

        assignee = requester.username
        followers = [assignee, actor.username]
        summary = self.make_summary_for_st_ticket()
        description = self.make_description_for_st_ticket()

        issues_repo = get_issues_repository()
        try:
            issue = issues_repo.create(
                queue=settings.IDM_ST_QUEUE_FOR_DISCUSSION,
                access=access,
                assignee=assignee,
                followers=followers,
                summary=summary,
                description=description,
            )
        except BackendError as tracker_error:
            if tracker_error.response is not None:
                message = tracker_error.response.content
            else:
                message = 'Tracker returned unknown response'
            log.exception('Error during creating issue for request %s', self.id)
            raise InternalAPIError(message=message)

        self.st_issue = issue.key
        self.save(update_fields=['st_issue'])
        role.actions.create(
            action=ACTION.START_DISCUSSION,
            data={'comment': settings.IDM_ST_BASE_URL + issue.key},
            requester=actor,
        )
        return self.st_issue

    def make_description_for_st_ticket(self):
        requester = self.requester
        role = self.role
        with override('ru'):
            template_name = 'emails/approve_role.txt'
            context = {
                'requester': requester,
                'user': role.user,
                'group': role.group,
                'role': role,
                'comment': self.comment,
                'for_email': False,
            }
            description = render_to_string(template_name, context).strip()
        return description

    def make_summary_for_st_ticket(self):
        role = self.role
        subject_template = _('Подтверждение роли. %s.')
        summary = subject_template % role.system.get_name()
        return summary

    def get_reason(self):
        reason = self.reason
        if self.reason == 'import':
            reason = 'inconsistency'
        return reason

    def set_done(self):
        self.is_done = True
        self.save()

    def has_declined_approves(self):
        return self.approves.filter(approved=False).exists()

    def update_approves(self):
        from idm.core.models import Approve

        self.approves.filter(requests__decision=APPROVEREQUEST_DECISION.DECLINE).decline()
        self.approves.exclude(approved=False).filter(requests__decision=APPROVEREQUEST_DECISION.APPROVE).approve()

        approves_to_decline_pks = []
        for approve in self.approves.filter(approved=None):
            if not approve.requests.exclude(decision=APPROVEREQUEST_DECISION.IGNORE).exists():
                approves_to_decline_pks.append(approve.pk)

        Approve.objects.filter(pk__in=approves_to_decline_pks).decline()

    def __str__(self):
        return 'role=%r, is_done=%s, added=%s' % (self.role, self.is_done, self.added)

    def get_num_approves_left(self):
        """Количество подтверждений, которые осталось получить. """
        return self.get_undecided_approves().count()

    def get_undecided_approves(self):
        return self.approves.prefetch_related('requests__approver').filter(approved=None)

    def get_approvers(self):
        from idm.core.models import ApproveRequest

        approve_requests_for_role = ApproveRequest.objects.filter(approve__role_request=self)
        approvers_id = approve_requests_for_role.values_list('approver', flat=True)
        approvers = User.objects.filter(id__in=approvers_id)
        return approvers

    def get_approve_requests(self, decision=NOT_PROVIDED):
        from idm.core.models import ApproveRequest

        qs = (
            ApproveRequest.objects.
            filter(approve__role_request=self).
            select_related(
                'approve__role_request__requester',
                'approve__role_request__role',
                'approve__role_request__role__system',
                'approve__role_request__role__user',
                'approve__role_request__role__group',
                'approve__role_request__role__node',
                'approver',
            ).
            order_by('approve', 'priority', 'pk')
        )
        if decision is not NOT_PROVIDED:
            qs = qs.filter(decision=decision)
        return qs

    def as_approvers_repr(self):
        from idm.core.workflow.plain.approver import AnyApprover
        approve_requests = self.get_approve_requests()
        approvers = []
        for approve, requests in itertools.groupby(approve_requests, lambda request: request.approve_id):
            requests = list(requests)
            if len(requests) == 1:
                approver = requests[0].as_approver()
            else:
                approver = AnyApprover([request.as_approver() for request in requests])
            approvers.append(approver)
        return '%r' % approvers

    def notify_approvers(self, comment):
        """Оповещаем подтверждающих о необходимости подтвердить роль. Каждому из подтверждающих отправляем
        письмо со со ссылкой для подтверждения роли.
        notify_everyone означает, что нужно оповестить всех подтверждающих.
        """
        from idm.core.models import ApproveRequest

        # Оповещаем подтверждающих только в том случае, если это обычный, первоначальный запрос роли,
        # и этот запрос не автоподтверждён. Для всех остальных случаев раз в сутки формируется дайджест.
        if self.is_done or self.role.is_active or self.role.is_public is False or self.role.node.is_public is False:
            return

        approve_requests = self.get_approve_requests()

        if self.notify_everyone:
            approvers_to_notify = approve_requests
        else:
            approvers_to_notify = ApproveRequest.objects.get_approvers_to_notify(
                approve_requests,
                exclude_active_roles=True,  # на самом деле это не нужно, потому что выше мы уже отсекаем такое
            )

        ApproveRequest.objects.notify_approvers(approvers_to_notify, comment)

    def close_issue(self):
        issue_key = self.st_issue
        if not issue_key:
            return
        issues_repo = get_issues_repository()
        try:
            issue = issues_repo.get_one({'id': issue_key})
            issue.transitions['close'].execute(resolution='fixed')
        except Exception:
            log.exception('Error during closing issue for request %s', self.id)

    def get_main_approvers_for_all_groups(self):
        return list(filter(None, [approve.get_main_approvers() for approve in self.get_undecided_approves()]))

    def get_additional_approvers_for_all_groups(self):
        return list(filter(None, [approve.get_additional_approvers() for approve in self.get_undecided_approves()]))

    def get_emailed_approvers_for_all_groups(self):
        return list(filter(None, itertools.chain(*[approve.get_emailed_approvers()
                                                   for approve in self.get_undecided_approves()])))
