from django.core.exceptions import PermissionDenied
from django.db import models, transaction
from django.db.models import NOT_PROVIDED
from django.utils import timezone
from django.utils.translation import ugettext_lazy as _
from idm.sync.staff import transfers

from idm.core.exceptions import TransferStateSwitchError
from idm.core.querysets.transfer import TransferManager
from idm.framework.fields import StrictForeignKey, NullJSONField as JSONField
from idm.framework.mixins import RefreshFromDbWithoutRelatedMixin
from idm.utils.i18n import get_lang_key


class Transfer(RefreshFromDbWithoutRelatedMixin, models.Model):
    TYPE_USER = 'user'
    TYPE_GROUP = 'group'
    TYPE_USER_GROUP = 'user_group'
    TYPE_CHOICES = (
        (TYPE_USER, _('Пользователь перемещён')),
        (TYPE_GROUP, _('Группа перемещена')),
        (TYPE_USER_GROUP, _('Группа пользователя перемещена')),
    )
    STATE_CHOICES = (
        ('created', _('Не обработана')),
        ('undecided', _('Решение ещё не принято')),
        ('accepted', _('Решение: необходим пересмотр ролей')),
        ('rejected', _('Решение: пересмотр ролей не требуется')),
        ('auto_accepted', _('Автоматическое решеине: пересмотр ролей необходим')),
        ('auto_rejected', _('Автоматическое решение: пересмотр ролей не требуется')),
        ('expired', _('Решение не было вынесено в срок, роли не были пересмотрены')),
    )
    TYPES = dict(TYPE_CHOICES).keys()
    USER_TYPES = {TYPE_USER, TYPE_USER_GROUP}
    GROUP_TYPES = {TYPE_GROUP}
    parent = StrictForeignKey(
        'self',
        null=True, blank=True,
        related_name='children',
        verbose_name=_('Родительское перемещение'),
        on_delete=models.CASCADE
    )
    is_new = models.BooleanField(default=False, verbose_name=_('Флаг, означающий, что эти перемещения - нового типа. '
                                                               'До релиза автопересмотров False, после - True'))
    created_at = models.DateTimeField(_('Дата создания'), editable=False, null=True, auto_now_add=True)
    resolved_at = models.DateTimeField(_('Дата принятия решения'), null=True, editable=False)
    type = models.CharField(max_length=30, choices=TYPE_CHOICES, default='user', db_index=True,
                            verbose_name=_('Тип перемещения'))
    state = models.CharField(max_length=30, choices=STATE_CHOICES, default='undecided', db_index=True,
                             verbose_name=_('Состояние'))
    user = StrictForeignKey('users.User', related_name='transfers', null=True, blank=False,
                             verbose_name=_('Сотрудник'), on_delete=models.CASCADE)
    group = StrictForeignKey('users.Group', related_name='transfers', null=True, blank=False,
                              verbose_name=_('Группа'), on_delete=models.CASCADE)
    source = StrictForeignKey('users.Group', related_name='transfers_from', null=True, blank=False,
                               verbose_name=_('Группа-источник'), on_delete=models.CASCADE)
    target = StrictForeignKey('users.Group', related_name='transfers_to', null=True, blank=False,
                               verbose_name=_('Группа-назначение'), on_delete=models.CASCADE)
    source_path = models.TextField(default='', blank=True, verbose_name=_('Источник: путь по слагам групп'))
    source_name = JSONField(verbose_name=_('Источник: путь по названиям групп'), blank=True, null=True)
    target_path = models.TextField(default='', blank=True, verbose_name=_('Назначение: путь по слагам групп'))
    target_name = JSONField(verbose_name=_('Назначение: путь по названиям групп'), blank=True, null=True)

    objects = TransferManager()

    class Meta:
        verbose_name = _('Перемещение')
        verbose_name_plural = _('Перемещения')

    def __str__(self):
        result = '%s[%s]: ' % (self.get_type_display(), self.state)
        if self.type == 'group':
            owner = self.group.get_name()
        else:
            owner = self.user.get_full_name()
        result += '%s: %s -> %s' % (owner, self.get_source(), self.get_target())
        return result

    def save(self, *args, **kwargs):
        if not self.pk:
            self.source_path = '/%s/' % '/'.join(item['slug'] for item in self.source_name)
            self.target_path = '/%s/' % '/'.join(item['slug'] for item in self.target_name)
            self.is_new = True
        return super(Transfer, self).save(*args, **kwargs)

    @property
    def owner(self):
        if self.type in self.USER_TYPES:
            result = self.user
        else:
            result = self.group
        return result

    def get_name(self, fieldname, lang=None):
        if lang is None:
            lang = get_lang_key()
        if lang == 'ru':
            key = 'name'
        else:
            key = 'name_en'
        return '/%s/' % '/'.join(item[key] for item in getattr(self, fieldname))

    def get_source(self):
        return self.get_name('source_name')

    def get_target(self):
        return self.get_name('target_name')

    def as_comment(self, kept_granted):
        postfixes_keep = {
            self.TYPE_USER: '. Пользователь был перемещён из "{source}" в "{target}"',
            self.TYPE_USER_GROUP: '. Группа пользователя была перемещена из "{source}" в "{target}"',
            self.TYPE_GROUP: '. Группа была перемещена из "{source}" в "{target}"',
        }

        postfixes_rerequest = {
            self.TYPE_USER: ' в связи со сменой подразделения с "{source}" на "{target}"',
            self.TYPE_USER_GROUP: ' в связи с перемещением группы пользователя из "{source}" в "{target}"',
            self.TYPE_GROUP: ' в связи с перемещением группы из "{source}" в "{target}"',
        }

        if kept_granted:
            postfix = postfixes_keep[self.type]
        else:
            postfix = postfixes_rerequest[self.type]

        comment = postfix.format(source=self.get_source(), target=self.get_target())
        return comment

    def get_decision(self, from_date=None, to_date=None, last_sync_date=NOT_PROVIDED):
        assert self.type in (Transfer.TYPE_GROUP, Transfer.TYPE_USER)
        return transfers.get_transfer_decision(
            self,
            from_date=from_date,
            to_date=to_date,
            last_sync_date=last_sync_date,
        )

    def get_alike(self):
        alike = type(self).objects.exclude(pk=self.pk)
        if self.type in self.USER_TYPES:
            alike = alike.filter(user_id=self.user_id, type__in=self.USER_TYPES)
        else:
            alike = alike.filter(group_id=self.group_id, type__in=self.GROUP_TYPES)
        return alike

    def accept(self, requester=None, bypass_checks=False, state='accepted', decision=None):
        assert state in ('accepted', 'auto_accepted')
        self.decide(
            action='transfer_accepted',
            state=state,
            method_name='ask_rerequest',
            requester=requester,
            bypass_checks=bypass_checks,
            decision=decision,
        )

    def reject(self, requester=None, bypass_checks=False, state='rejected', decision=None):
        assert state in ('rejected', 'auto_rejected')
        self.decide(
            action='transfer_rejected',
            state=state,
            method_name='keep_granted',
            requester=requester,
            bypass_checks=bypass_checks,
            decision=decision,
        )

    def expire(self):
        self.actions.create(
            action='transfer_expired',
            user=self.user,
            group=self.group,
        )
        self.state = 'expired'
        self.resolved_at = timezone.now()
        self.save(update_fields=('state', 'resolved_at'))

    @transaction.atomic
    def decide(self, action, state, method_name, requester=None, bypass_checks=False, decision=None,
               parent_transfer=None):
        if not bypass_checks:
            if not requester.has_perm('core.resolve_transfer'):
                raise PermissionDenied(_('Вы не можете принимать решения по данному перемещению'))
        self.refresh_from_db()  # перемещение уже могло перейти в другой статус
        if self.state != 'undecided':
            raise TransferStateSwitchError(_('По данному перемещению уже было вынесено решение'))

        data = None
        if decision is not None:
            data = decision.as_data()
        parent_action = self.actions.create(
            action=action,
            requester=requester,
            user=self.user,
            group=self.group,
            data=data,
        )
        if parent_transfer is not None:
            self.parent = parent_transfer
        self.state = state
        self.resolved_at = timezone.now()
        self.save(update_fields=('state', 'resolved_at', 'parent'))
        self.get_alike().undecided().select_related('group', 'user').expire()

        for child_transfer in self.children.undecided().select_related('user', 'group'):
            child_transfer.decide(
                action=action,
                state=state,
                method_name=method_name,
                requester=requester,
                bypass_checks=bypass_checks,
            )
        if self.is_new and self.type == 'group' and parent_transfer is None:
            related_transfers = type(self).objects.related_for(self).select_related('user', 'group')
            for related_transfer in related_transfers:
                related_transfer.decide(
                    action=action,
                    state=state,
                    method_name=method_name,
                    requester=requester,
                    bypass_checks=bypass_checks,
                    decision=decision,
                    parent_transfer=self,
                )

        roles = self.owner.roles.askable_to_rerequest(self)
        method = getattr(roles, method_name)
        method(self, parent_action=parent_action)
