import logging

from django.conf import settings
from django.db import models
from django.utils.translation import ugettext_lazy as _

from idm.core.constants.passport_login import PASSPORT_LOGIN_STATE, CREATED_FLAG_REASON
from idm.core.querysets.passportlogin import UserPassportLoginManager
from idm.framework.fields import StrictForeignKey

log = logging.getLogger(__name__)


class UserPassportLogin(models.Model):
    """Паспортный логин пользователя"""
    user = StrictForeignKey(
        'users.User',
        related_name='passport_logins',
        verbose_name=_('Сотрудник'),
        blank=True,
        null=True,
        on_delete=models.SET_NULL,
    )
    login = models.CharField(max_length=50, unique=True, null=True)
    added = models.DateTimeField(auto_now_add=True, verbose_name=_('Дата добавления'))
    roles = models.ManyToManyField(
        'core.Role',
        blank=True,
        related_name='passport_logins',
        verbose_name=_('Роли, использующие данный паспортный логин'),
    )
    state = models.CharField(
        max_length=20,
        verbose_name=_('Состояние'),
        choices=PASSPORT_LOGIN_STATE.CHOICES,
        default=PASSPORT_LOGIN_STATE.CREATED,
        db_index=True
    )
    created_by_idm = models.BooleanField(default=False)
    created_flag_reason = models.CharField(max_length=10, choices=CREATED_FLAG_REASON.CHOICES, null=True)
    is_fully_registered = models.BooleanField(default=False)

    objects = UserPassportLoginManager()

    class Meta:
        ordering = ('login',)
        db_table = 'upravlyator_userpassportlogin'

    def __str__(self):
        return self.login

    def set_state(self, state):
        assert state in dict(PASSPORT_LOGIN_STATE.CHOICES).keys()
        self.state = state
        self.save()

    def save(self, *args, **kwargs):
        self.login = self.login.lower().strip()
        return super(UserPassportLogin, self).save(*args, **kwargs)

    def subscribe(self):
        from idm.core.tasks import SubscribePassportLogin
        SubscribePassportLogin.max_retries = 1
        log.info('Trying to subscribe passport login "%s" for user "%s" with sid 67', self.login, self.user_id)
        return SubscribePassportLogin.delay(passport_login=self.login)

    def unsubscribe(self):
        from idm.core.tasks import UnsubscribePassportLogin
        UnsubscribePassportLogin.max_retries = 1
        log.info('Trying to unsubscribe passport login "%s" for user "%s" with sid 67', self.login, self.user)
        return UnsubscribePassportLogin.delay(passport_login=self.login)

    def is_subscribable(self, new_role=None):
        from idm.core.models import RoleNode, System

        if new_role and new_role.is_subscribable(ignore_state=True):
            return True

        bad_systems = System.objects.filter(slug__in=settings.IDM_SID67_EXCLUDED_SYSTEMS).values_list('pk', flat=True)
        possible_nodes = RoleNode.objects.exclude(
            system__pk__in=bad_systems,
        ).filter(
            roles__passport_logins=self,
            roles__is_active=True,
        )
        non_subscribable_node_pks = possible_nodes.non_subscribable_pks()
        subscribable_nodes = set(possible_nodes.values_list('pk', flat=True)) - set(non_subscribable_node_pks)
        return bool(subscribable_nodes)
