import abc
from typing import Dict, Any, Union, Optional

from tastypie.bundle import Bundle

from idm.core.models import Role
from idm.core.models.appmetrica import AppMetrica
from idm.core.models.metrikacounter import MetrikaCounter


class BaseMutation(abc.ABC):
    @classmethod
    def is_applicable(cls, obj) -> bool:
        raise NotImplementedError

    @classmethod
    def mutate(cls, value: Dict[str, Any]):
        raise NotImplementedError


class RoleFieldMetrikaCounterMutation(BaseMutation):
    METRIKA_SYSTEM_SLUG = 'metrika'

    @classmethod
    def is_applicable(cls, obj: Union[Role, Bundle, Dict[str, Any]]) -> bool:
        if isinstance(obj, Bundle):
            if obj.data is isinstance(obj.data, dict):
                obj = obj.data
            elif isinstance(obj.obj, Role):
                obj = obj.obj
        if isinstance(obj, Role):
            obj.fetch_system()  # noqa
            return obj.system.slug == cls.METRIKA_SYSTEM_SLUG and 'counter_id' in obj.fields_data

        if isinstance(obj, dict):
            if obj.get('system', {}).get('slug') == cls.METRIKA_SYSTEM_SLUG:
                field_data = obj.get('fields_data', {})
                if isinstance(field_data, dict) and 'counter_id' in field_data:
                    return True
        return False

    @classmethod
    def mutate(cls, value: Dict[str, Any]) -> Optional[str]:
        if counter_name := MetrikaCounter.get_counter_name(value['counter_id']):
            value['counter_name'] = counter_name
            return counter_name


class RoleFieldAppMetricaMutation(BaseMutation):
    APP_METRICA_SYSTEM_SLUG = 'appmetrica'

    @classmethod
    def is_applicable(cls, obj: Union[Role, Bundle, Dict[str, Any]]) -> bool:
        if isinstance(obj, Bundle):
            if obj.data is isinstance(obj.data, dict):
                obj = obj.data
            elif isinstance(obj.obj, Role):
                obj = obj.obj
        if isinstance(obj, Role):
            obj.fetch_system()  # noqa
            return obj.system.slug == cls.APP_METRICA_SYSTEM_SLUG and 'application_id' in obj.fields_data

        if isinstance(obj, dict):
            if obj.get('system', {}).get('slug') == cls.APP_METRICA_SYSTEM_SLUG:
                fields_data = obj.get('fields_data', {})
                if isinstance(fields_data, dict) and 'application_id' in fields_data:
                    return True
        return False

    @classmethod
    def mutate(cls, value: Dict[str, Any]) -> Optional[str]:
        if application_name := AppMetrica.get_app_name(value['application_id']):
            value['application_name'] = application_name
            return application_name
