import functools
import logging
import sys
from typing import Callable, Optional, TYPE_CHECKING

from django.utils import timezone

from idm.core.constants.action import ACTION
from idm.core.constants.action_tiny import ACTION_TINY, STATUS
from idm.core.constants.instrasearch import INTRASEARCH_METHOD
from idm.core.constants.node_relocation import RELOCATION_STATE
from idm.framework.utils import add_to_instance_cache
from idm.utils.lock import lock

if TYPE_CHECKING:
    from idm.core.models import Action, RoleNode

log = logging.getLogger(__name__)


class NodePipeline(object):
    def __init__(self, system):
        self.system = system

    @staticmethod
    def state_greater(state_1: str, state_2: str) -> bool:
        a = RELOCATION_STATE.STATUSES
        return a.index(state_1) > a.index(state_2)

    @staticmethod
    def get_next_state(state: str) -> str:
        return RELOCATION_STATE.STATUSES[RELOCATION_STATE.STATUSES.index(state) - 1]

    @staticmethod
    def rebuild_closuretree(node: 'RoleNode', parent: Optional['RoleNode']):
        add_to_instance_cache(node, 'parent', parent)
        node.rebuild_closure_tree()

    @staticmethod
    def calc_fields(node: 'RoleNode', parent: Optional['RoleNode']):
        add_to_instance_cache(node, 'parent', parent)

        changed = node.calc_fields()
        if changed:
            node.save(update_fields=('slug_path', 'data', 'value_path', 'fullname'))

        if node.is_public:
            node.send_intrasearch_push(INTRASEARCH_METHOD.ADD)

    @staticmethod
    def rerun_workflow(node: 'RoleNode', parent: Optional['RoleNode'], parent_action: 'Action'):
        node.roles.rerunnable().rerun_workflow(parent_action=parent_action)

    def apply_f_for_active_descendants(self,
                                       node: 'RoleNode',
                                       initial_state: str,
                                       f: Callable[['RoleNode', 'RoleNode'], None]
                                       ):
        """
        Применить функцию f для всех потомков узла node, у которых relocation_state <= initial_state

        @param node: Узел, для потомков, которого будет применяться функция f
        @param initial_state: relocation_state, для обработки которого вызывается данная функция
        @param f: Функция, которая принимает узел и его предка
        @return:
        """
        for child in node.children.active():
            if self.state_greater(child.relocation_state, initial_state):
                continue

            if child.relocation_state != RELOCATION_STATE.GOOD:
                child.set_relocation_state(RELOCATION_STATE.GOOD)

            f(child, node)
            self.apply_f_for_active_descendants(child, initial_state, f)

    def pipeline_step(self, nodes, f):
        called_from = sys._getframe(1).f_code.co_name
        relocation_state = next((x for x in self.PIPELINE if x[1] == called_from))[0]
        changed = False
        next_state = self.get_next_state(relocation_state)
        for node in nodes.select_related('parent'):
            node.refresh_from_db()
            if node.relocation_state != relocation_state:
                continue
            changed = True
            f(node, node.parent)
            self.apply_f_for_active_descendants(node, relocation_state, f)
            node.set_relocation_state(next_state)

        return changed

    def rebuild_closuretree_for_descendants(self, nodes):
        return self.pipeline_step(nodes, self.rebuild_closuretree)

    def rebuild_fields_for_descendants(self, nodes):
        return self.pipeline_step(nodes, self.calc_fields)

    def rerun_workflow_for_descendants(self, nodes):
        from idm.core.models import Action

        if not nodes:
            return

        parent_action = Action.objects.create(
            action=ACTION.MASS_ACTION,
            data={'name': 'rerun nodes workflow for pipeline'}
        )
        rerun_workflow = functools.partial(self.rerun_workflow, parent_action=parent_action)
        return self.pipeline_step(nodes, rerun_workflow)

    PIPELINE = (
        (RELOCATION_STATE.SUPERDIRTY, 'rebuild_closuretree_for_descendants'),
        (RELOCATION_STATE.CT_COMPUTED, 'rebuild_fields_for_descendants'),
        (RELOCATION_STATE.FIELDS_COMPUTED, 'rerun_workflow_for_descendants'),
    )

    def locked_run(self, block: bool = False):
        with lock('idm_lock_node_pipeline_{}'.format(self.system.slug), block=block) as locked:
            if not locked:
                return False
            self.run()
            return True

    def run(self):
        from idm.core.models import ActionTiny
        log.info('Running pipeline for system %s', self.system.slug)
        self.system.metainfo.last_recalc_pipeline_start = timezone.now()
        self.system.metainfo.save(update_fields=['last_recalc_pipeline_start'])
        start = timezone.now()
        changed = False
        action_status = STATUS.OK
        traceback = ''
        for status, method_name in self.PIPELINE:
            nodes = self.system.nodes.active().filter(relocation_state=status).order_by('level')
            try:
                changed = getattr(self, method_name)(nodes) or changed
            except Exception as e:
                traceback = str(e)
                action_status = STATUS.ERROR
                log.exception('NodePipeline error. System %s', self.system.slug)

        if changed:
            ActionTiny.objects.create(
                system=self.system,
                action=ACTION_TINY.PIPELINE_FINISHED,
                status=action_status,
                traceback=traceback,
                start=start,
                finish=timezone.now()
            )

        self.system.metainfo.last_recalc_pipeline_finish = timezone.now()
        self.system.metainfo.save(update_fields=['last_recalc_pipeline_finish'])
