# -*- coding: utf-8 -*-


import constance
import logging

from idm.core.models import InternalRole, InternalRoleUserObjectPermission, NodeResponsibility
from idm.core.plugins.generic import Plugin as GenericPlugin
from idm.users.constants.user import USER_TYPES
from idm.utils import coroutine
from idm.permissions.hooks import Hooks

log = logging.getLogger(__name__)


class Plugin(GenericPlugin):
    """Это плагин для системы IDM (self), отличается от обычного generic-плагина тем,
    что идет в обход http
    """

    def __init__(self, system):
        super(Plugin, self).__init__(system)

        self.hooks = Hooks()

    @coroutine
    def get_roles(self, **kwargs):
        streams = [InternalCommonRoleStream(), InternalSystemsRoleStream()]
        return self._get_roles(streams)

    def _get_roles(self, streams):
        _ = yield
        limit = constance.config.IDM_INTERNAL_ROLES_PAGE_SIZE
        for stream in streams:
            queryset = stream.get_queryset()
            values_list = stream.values_list(queryset)
            row_with_min_pk = values_list.first()
            if row_with_min_pk is None:
                continue
            since = row_with_min_pk[0]
            while True:
                roles = set()
                rows = values_list.filter(pk__gte=since)[:limit]
                if not rows:
                    break
                for row in rows:
                    role = row[1:]
                    if role in roles:
                        continue
                    roles.add(role)
                    yield stream.row_as_dict(row)
                since = row[0] + 1
        return

    def info(self):
        return self.hooks.info()

    def add_role(self, username, role_data, fields_data, **kwargs):
        subject_type = kwargs.get('subject_type') or USER_TYPES.USER
        return self.hooks.add_role(login=username, role=role_data, fields=fields_data, subject_type=subject_type)

    def remove_role(self, username, role_data, system_specific, is_fired=None, **kwargs):
        subject_type = kwargs.get('subject_type') or USER_TYPES.USER
        return self.hooks.remove_role(
            login=username,
            role=role_data,
            data=system_specific,
            is_fired=is_fired,
            subject_type=subject_type,
        )

class InternalCommonRoleStream(object):
    """Внутренние общие роли"""
    def get_queryset(self):
        qs = (
            InternalRole
            .objects
            .filter(user_object_permissions__isnull=False, node__isnull=True)
            .order_by('pk')
            .distinct()
        )
        return qs

    def values_list(self, queryset):
        """Первым значением каждого элемента должен быть pk."""
        return queryset.values_list(
            'pk',
            'user_object_permissions__user__username',
            'user_object_permissions__user__type',
            'role',
        )

    def row_as_dict(self, row):
        pk, username, subject_type, role = row

        return {
            'login': username,
            'subject_type': subject_type,
            'path': '/group/common/role/%s/' % role,
        }


class InternalSystemsRoleStream(object):
    """Внутренние системные роли"""
    def __init__(self):
        internal_nodes = set(
            InternalRole
            .objects
            .values_list('node_id', flat=True)
        )

        nodes_to_exclude = set(
            NodeResponsibility
            .objects
            .filter(node__in=internal_nodes, is_active=True)
            .values_list('node_id', flat=True)
        )

        self.nodes = internal_nodes - nodes_to_exclude

    def get_queryset(self):
        return (
            InternalRoleUserObjectPermission
            .objects
            .filter(content_object__node_id__in=self.nodes)
            .order_by('pk')
        )

    def values_list(self, queryset):
        return queryset.values_list(
            'pk',
            'user__username',
            'user__type',
            'content_object__role',
            'content_object__node__system__slug',
            'content_object__node__value_path',
        )

    def row_as_dict(self, row):
        pk, username, subject_type, role, system_slug, value_path = row
        return {
            'login': username,
            'subject_type': subject_type,
            'path': '/group/system/system_on/%s/role/%s/' % (system_slug, role),
            'fields': {
                'scope': value_path,
            }
        }
