# coding: utf-8


import itertools
import logging
from collections import defaultdict

import constance
from django.db import models
from django.db.models import Q
from django.utils.translation import override, ugettext_lazy as _

from idm.core.constants.approverequest import APPROVEREQUEST_DECISION
from idm.core.constants.workflow import DEFAULT_PRIORITY
from idm.core.querysets.base import BasePermittedRolesQuerySet
from idm.notification.utils import send_notification

log = logging.getLogger(__name__)


class ApproveRequestQuerySet(BasePermittedRolesQuerySet):
    prefix = 'approve__role_request__role__'

    def select_related_for_set_decided(self):
        return self.select_related(
            'approve__role_request__role__system',
            'approve__role_request__role__parent',
            'approve__role_request__role__node',
        )


class ApproveRequestManager(models.Manager.from_queryset(ApproveRequestQuerySet)):
    def send_pending_approval_reminders(self):
        """Функция отсылает письма с дайджестом о ролях, которые необходимо подтвердить
           Дайджесты уходят людям с основным приоритетом, которые присутсвуют и у них notify != False
        """

        from idm.core.models import Approve
        requests_by_user = defaultdict(lambda: defaultdict(int))

        human_states = {
            'requested': _('Простых запросов'),
            'rerequested': _('Повторных запросов'),
            'review_request': _('Регулярных пересмотров'),
            'imported': _('Запросов по расхождениям'),
        }

        pending_approves = (
            Approve.objects.
                filter(approved=None, role_request__is_done=False).
                select_related('role_request__role__inconsistency')
        )

        # lang -> email -> state -> set of role pks
        reminders_cc = defaultdict(lambda: defaultdict(lambda: defaultdict(set)))

        for approve in pending_approves:
            role_request = approve.role_request
            role = role_request.role
            role_state = role.state
            if role.email_cc and isinstance(role.email_cc, dict):
                recipients = role.email_cc.get('reminders', [])
                for recipient_dict in recipients:
                    email = recipient_dict['email']
                    lang = recipient_dict['lang']
                    reminders_cc[lang][email][role_state].add(role.pk)

            requests = list(approve.requests.filter(
                decision=APPROVEREQUEST_DECISION.NOT_DECIDED, parent=None).select_related('approver').order_by('priority', 'pk'))
            # TODO https://st.yandex-team.ru/IDM-6835
            if len(requests) == 0:
                continue
            # для аппруверов с правилом ИЛИ - письмо людям с основным приоритетом,
            # но если включена опция notify_everyone, то всем
            if role_request.notify_everyone:
                for request in requests:
                    requests_by_user[request.approver][role_state] += 1
            else:
                someone_found = False
                for request in requests:
                    # апруверы с основным приоритетом
                    if request.priority == approve.main_priority:
                        requests_by_user[request.approver][role_state] += 1
                        someone_found = True

                # если не нашли апруверов с основным приоритетом, то шлём всем с минимальным приоритетом
                if someone_found is False:
                    # такого случая быть не должно
                    log.warning('Approvers with main priority were not found for approve %d during sending digest',
                                approve.id)
                    new_available_main_priority = min([request.priority for request in requests])

                    for request in requests:
                        if request.priority == new_available_main_priority:
                            requests_by_user[request.approver][role_state] += 1

        template = 'emails/pending_requests.txt'
        approvers = sorted(requests_by_user.keys(), key=lambda user: user.username)
        for approver in approvers:
            requests = requests_by_user[approver]
            lang_ui = approver.lang_ui

            with override(lang_ui):
                requests = {human_states.get(key): value for key, value in requests.items()}
                context = {
                    'requests': requests,
                    'num_requests': sum(requests.values()),
                    'approver': approver,
                }
                send_notification(_('Подтверждение ролей.'), template, [approver], context)

        for lang, mapping in reminders_cc.items():
            with override(lang):
                for email, states_dict in mapping.items():
                    requests = {human_states.get(key): len(value) for key, value in list(states_dict.items())}
                    context = {
                        'requests': requests,
                        'num_requests': sum(requests.values()),
                    }
                    send_notification(_('Подтверждение ролей.'), template, [email], context)

    def get_approvers_by_main_priority(self, approve_requests):
        """
            Считать основной приоритет с учетом отсутствии в группе.
            Возвращает приоритет и аппруверов с таким приоритетом. Игнорирует аппруверов с notify=False
        """
        approvers_with_main_priority = []
        new_available_main_priority = DEFAULT_PRIORITY

        for approve_request in approve_requests:
            if not approve_request.approver.is_absent and approve_request.notify is not False:
                # если приоритет не равен основному, то выйти из цикла
                if (new_available_main_priority != DEFAULT_PRIORITY and
                        approve_request.priority != new_available_main_priority):
                    break
                new_available_main_priority = approve_request.priority
                approvers_with_main_priority.append(approve_request)

        # если все в группе отсутствуют, то отправляем уведомление людям с минимальным приоритетом без notify = False
        if not approvers_with_main_priority:
            approvers_with_notify_true = [approver for approver in approve_requests if approver.notify is not False]
            if approvers_with_notify_true:
                new_available_main_priority = min([approver.priority for approver in approvers_with_notify_true])
            else:
                new_available_main_priority = min([approver.priority for approver in approve_requests])

            approvers_with_main_priority = [approver for approver in approve_requests
                                            if approver.notify is True or
                                            approver.priority == new_available_main_priority]
        else:
            # добавить людей с notify=True
            approvers_with_main_priority.extend([approve_request for approve_request in approve_requests
                                                 if
                                                 approve_request.notify and not approve_request.is_notification_sent])

        # если у всех был notify=False, то они попали в этот список
        # надо снова их убрать
        approvers_with_main_priority = [
            approver for approver in approvers_with_main_priority if approver.notify is not False
        ]
        return new_available_main_priority, approvers_with_main_priority

    def notify_approvers(self, requests_to_notify, comment=''):
        """
        Высылает уведомления аппруверам
        :param requests_to_notify: List[ApproveRequest]
        """
        notified_approvers = set()

        for request in requests_to_notify:
            if request.approver not in notified_approvers and not request.is_notification_sent:
                notified_approvers.add(request.approver)
                request.send_email(comment)
                request.is_notification_sent = True
                request.save(update_fields=['is_notification_sent'])

    def recalculate_main_priority(self, approve=None):
        """
             Пересчитывает основной приоритет у всех аппруверов раз в 15 минут и отправляет уведомления
             при смене основновного приоритета
        """
        from idm.core.models import ApproveRequest

        filtered_approve_requests = (
            Q(approve__approved=None) & Q(approve__role_request__is_done=False) &
            ~Q(approve__role_request__role__is_public=False) &
            Q(approve__role_request__role__node__is_public=True) &
            Q(added__gte=constance.config.RECALCULATE_MAIN_PRIORITY_START_DATE) &
            ~Q(decision=APPROVEREQUEST_DECISION.IGNORE)
        )

        if approve is not None:
            filtered_approve_requests &= Q(approve=approve)

        approve_requests = (
            ApproveRequest.objects
            .filter(filtered_approve_requests).
            select_related(
                'approver', 'approve', 'approve__role_request__role__system',
                'approve__role_request__role__user', 'approve__role_request__role__group',
                'approve__role_request__role__node',
                'approve__role_request__role', 'approve__role_request__requester'
            )
            .order_by('approve', 'priority')
        )

        approvers_to_notify = self.get_approvers_to_notify(approve_requests, exclude_active_roles=True)
        self.notify_approvers(approvers_to_notify)

    def get_approvers_to_notify(self, approve_requests, exclude_active_roles):
        """
            Функция возвращает аппруверов с основным приоритетом для всех групп одного запроса(RoleRequest)
        """

        from idm.core.models import ApproveRequest

        approvers_to_notify = []

        for approve, requests in itertools.groupby(approve_requests, lambda request: request.approve):
            if approve.approved is not None:
                continue
            requests = list(requests)

            # отсортируем по приоритету и выберем всех с основным приоритетом
            main_priority, filtered_requests = ApproveRequest.objects.get_approvers_by_main_priority(requests)
            # если основной приоритет группы изменился
            if approve.main_priority != main_priority:
                approve.main_priority = main_priority
                approve.save(update_fields=['main_priority'])

            # приоритеты перепросчитываем всегда, но уведомлялки в случае перезапроса не отправляем
            if not (exclude_active_roles and approve.role_request.role.is_active):
                approvers_to_notify.extend(filtered_requests)

        return approvers_to_notify

