import copy
import datetime

from django.db.models import QuerySet, Q
from django.db.models.query_utils import InvalidQuery
from django.utils.timezone import make_naive, utc

from idm.core.constants.groupmembership import GROUPMEMBERSHIP_STATE
from idm.core.constants.system import SYSTEM_REQUEST_POLICY
from idm.framework.requester import requesterify
from idm.users.constants.group import GROUP_TYPES
from idm.users.constants.user import USER_TYPES


def _can_view_all_in_system(requester, system):
    if system.request_policy == SYSTEM_REQUEST_POLICY.ANYONE:
        return True

    system.fetch_root_role_node()
    return requester.impersonated.has_perm('core.idm_view_roles', system.root_role_node)


def invert(value):
    result = value
    if isinstance(value, datetime.datetime):
        result = - (make_naive(value, utc) - datetime.datetime(1970, 1, 1)).total_seconds()
    return result


class UnwrapMixin(object):
    def unwrap(self: QuerySet, use_random_sorting: bool = False) -> QuerySet:
        query = self.query
        if query.can_filter():  # does not have limits
            raise InvalidQuery('Cannot unwrap unlimited query')

        if use_random_sorting and self.query.order_by:
            # добавим случайную сортировку, чтобы постгрес не пытался идти по индексу
            self.query.order_by = (*self.query.order_by, '?')

        ids = list(self.values_list('pk', flat=True))

        # We don't need WHERE once we know ids, clear it
        # self.query.where.children = []
        clone = self._clone()
        clone.query.clear_limits()
        slice_ = clone.filter(pk__in=ids)
        return slice_


class BasePermittedQuerySet(UnwrapMixin, QuerySet):
    def permitted_query(self, requester, base=None, system=None, **kwargs):
        raise NotImplementedError

    def permitted_for(self, requester, system=None, permission_params=None, **kwargs):
        queryset = self
        q_set = self.permitted_query(requester, system=system, permission_params=permission_params, **kwargs)

        # фильтр может быть пустой для людей с широкими полномочиями,
        # поэтому нет смысла его добавлять тогда
        if q_set:
            queryset.permitted_q = q_set
            queryset = queryset.filter(q_set)

        return queryset

    def _clone(self, *args, **kwargs):
        clone = super(BasePermittedQuerySet, self)._clone(*args, **kwargs)
        clone.permitted_q = getattr(self, 'permitted_q', None)
        return clone


class BasePermittedRolesQuerySet(BasePermittedQuerySet):
    """
    Базовый менеджер для работы с объектами, для которых учитываются разрешения
    """
    prefix = ''

    def p(self, s):
        return self.prefix + s

    def make_subquery(self, fieldname, qs, values_list_field='id'):
        q_filter = Q(**{
            self.p('%s__in' % fieldname): qs.values_list(values_list_field, flat=True)
        })
        return q_filter

    def _get_get_tvm_apps_subquery(self, requester):
        from idm.users.models import GroupMembership, GroupResponsibility

        tvm_services_ids = (
            GroupResponsibility.objects
            .active()
            .filter(
                group__type=GROUP_TYPES.TVM_SERVICE,
                user=requester.impersonated,
            )
            .values('group_id')
        )
        tvm_apps_memberships = GroupMembership.objects.active().filter(group_id__in=tvm_services_ids)
        q = self.make_subquery('user_id', tvm_apps_memberships, 'user_id')
        return q

    def _is_allowed_user(self, requester, user):
        if user is None:
            return False

        if requester == user:
            return True

        user.fetch_department_group()
        return (
            requester.impersonated.is_head_for(user)
            or requester.impersonated.is_owner_of(user)
        )

    def permitted_query(self, requester, base=None, system=None, permission_params=None, **kwargs):
        """
        Фильтровать queryset в соответствии с разрешениями пользователя, сделавшего запрос
        """
        from idm.core.models import Role, RoleNode, RoleRequest
        from idm.users.models import User, GroupClosure

        q_filter = base or Q()
        permission_params = permission_params or {}
        requester = requesterify(requester)

        p = self.p

        # если есть широкие права - отдаем все сущности
        # TODO: убрать проверку env.name != 'intranet' когда в коннекте появятся нормальные права @polosate
        if requester.impersonated.has_perm('core.idm_view_roles'):
            return Q()

        # если в системе все могут всё или у человека есть право все смотерть, то не надо ничего проверять
        if 'system' in permission_params and _can_view_all_in_system(requester, permission_params['system']):
            return Q()

        # если пользователь смотрит сам на себя или на своего подчиненного, то нет смысла проверять права
        if (
            self._is_allowed_user(requester, permission_params.get('user')) or
            self._is_allowed_user(requester, permission_params.get('approver'))
        ):
            return Q()

        # Права на TVM-приложения проверяем только в нужных системах
        if (
            permission_params.get('user_type') == USER_TYPES.TVM_APP or
            (
                'user_type' not in permission_params and
                ('system' not in permission_params or permission_params['system'].use_tvm_role)
            )
        ):
            q_filter |= self._get_get_tvm_apps_subquery(requester)

        if not permission_params.get('groups') or permission_params.get('skip_users'):
            # в самом простом случае пользователь без особых прав смотрит лишь свои роли
            q_filter |= Q(**{p('user'): requester.impersonated})

            # роли своих подчинённых
            subordinates_qs = User.objects.filter(
                department_group__groupclosure_parents__parent__responsibilities__user=requester.impersonated,
                department_group__groupclosure_parents__parent__responsibilities__is_active=True,
            )

            q_filter |= self.make_subquery('user_id', subordinates_qs)

            # и роли своих роботов
            robots_qs = User.objects.filter(
                responsibles=requester.impersonated,
                is_active=True,
            )

            q_filter |= self.make_subquery('user_id', robots_qs)

        # если роли в системе могут запрашивать все для всех,
        # то нет смысла прятать от всех роли в этой системе
        q_filter |= Q(**{p('system__request_policy'): 'anyone'})

        # групповые роли, выданные на группу, где user ответственный, а также групповые роли, выданные на группы,
        # дочерние или родительские по отношению к группам, где user ответственный
        groups_down_responsible_qs = GroupClosure.objects.filter(
            parent__responsibilities__user=requester.impersonated,
            parent__responsibilities__is_active=True,
        )
        q_filter |= self.make_subquery('group_id', groups_down_responsible_qs, 'child_id')
        groups_up_responsible_qs = GroupClosure.objects.filter(
            child__responsibilities__user=requester.impersonated,
            child__responsibilities__is_active=True,
        )
        q_filter |= self.make_subquery('group_id', groups_up_responsible_qs, 'parent_id')

        # персональные роли, выданные по групповой на группы, где user ответственный, а также персональные роли,
        # выданные по групповой на группы, дочерние по отношению к тем, где user ответственный
        parent_role_qs = Role.objects.filter(
            group__groupclosure_parents__parent__responsibilities__user=requester.impersonated,
            group__groupclosure_parents__parent__responsibilities__is_active=True,
        )

        q_filter |= self.make_subquery('parent_id', parent_role_qs)

        # групповые роли, выданные на группы, где user член, а также
        # групповые роли, выданные на группы, родительские для тех, где user член
        groups_up_membership_qs = GroupClosure.objects.filter(
            child__memberships__user=requester.impersonated,
            child__memberships__state__in=GROUPMEMBERSHIP_STATE.ACTIVE_STATES,
        )

        q_filter |= self.make_subquery('group_id', groups_up_membership_qs, 'parent_id')

        # специальные полномочия в системах
        node_permitted_query = RoleNode.objects.get_permitted_query(
            requester.impersonated,
            'core.idm_view_roles',
            system,
            for_view=True,
        )
        q_filter |= Q(**{
            p('node_id__in'): node_permitted_query,
        })

        # роли, которые запросил пользователь
        last_requests_qs = RoleRequest.objects.filter(requester=requester.impersonated)
        q_filter |= self.make_subquery('last_request_id', last_requests_qs)

        # роли, которые подтверждал или мог подтверждать пользователь
        role_approves_qs = Role.objects.filter(requests__approves__requests__approver=requester.impersonated)
        q_filter |= self.make_subquery('id', role_approves_qs)

        return (q_filter & Q(**{p('system__in'): requester.allowed_systems})
                if requester.allowed_systems is not None
                else q_filter)
