import collections
import datetime
import itertools
import logging
import operator
from itertools import groupby, chain
from typing import Set, DefaultDict, Union, Dict, Any

import constance
import waffle
import yenv
from django.conf import settings
from django.db import transaction, models, connection
from django.db.models import Q, NOT_PROVIDED
from django.db.models.aggregates import Max
from django.utils import timezone
from django.utils.encoding import force_text
from django.utils.translation import ugettext_lazy as _
from django_pgaas import atomic_retry
from idm.framework.utils import add_to_instance_cache

from idm.core.constants.action import ACTION
from idm.core.workflow import exceptions
from idm.core.constants.affiliation import AFFILIATION
from idm.core.constants.email_templates import (
    EXPIRATION_ROLES_TEMPLATES,
    ONHOLD_ROLES_TEMPLATES,
    REREQUESTED_ROLES_TEMPLATES,
)
from idm.core.constants.groupmembership import GROUPMEMBERSHIP_STATE
from idm.core.constants.role import ROLE_STATE
from idm.core.constants.system import SYSTEM_GROUP_POLICY
from idm.core.constants.workflow import RUN_REASON, REQUEST_TYPE
from idm.core.querysets.base import BasePermittedRolesQuerySet
from idm.core.workflow.common.subject import subjectify
from idm.framework.requester import requesterify, Requester
from idm.framework.queryset import NestedValuesMixin
from idm.notification.utils import send_reminder
from idm.permissions import shortcuts
from idm.permissions.utils import get_permission
from idm.users.constants.group import GROUP_TYPES
from idm.users.constants.user import USER_TYPES
from idm.users.models import Group, User
from idm.utils.actions import parent_action_completion
from idm.utils.log import log_duration
from idm.utils.mongo import get_queue_size
from idm.utils.query import parametrize_sql, maybe_subquery

log = logging.getLogger('idm.core.querysets')


class RoleQuerySet(NestedValuesMixin, BasePermittedRolesQuerySet):
    def basic_related(self):
        return self.select_related('system', 'user', 'group', 'node')

    def reports_related(self):
        qs = self.select_related(
            'user',  # owner
            'group',  # group_type
            'user__department_group',  # department
            'system',  # system
            'node',  # role, role_code
            'parent',  # approvers
            'last_request',  # approvers
            'parent__last_request',  # approvers
        ).prefetch_related(
            'last_request__approves__requests__approver',
            'parent__last_request__approves__requests__approver',
        )
        return qs

    def active(self):
        return self.filter(is_active=True)

    def inactive(self):
        return self.filter(is_active=False)

    def returnable(self):
        return self.filter(state__in=ROLE_STATE.RETURNABLE_STATES)

    def onhold(self):
        return self.filter(state='onhold')

    def awaiting(self):
        return self.filter(state='awaiting')

    def public(self):
        return self.exclude(is_public=False).exclude(node__is_public=False, is_public=None)

    def allowed_for_review(self):
        return self.filter(Q(user=None) | (Q(user__is_robot=False) & ~Q(user__type=USER_TYPES.TVM_APP)))

    def get_systems_resources(self):
        system_to_resources = collections.defaultdict(set)
        resources = (
            self
            .values_list('system__slug', 'fields_data')
            .distinct('fields_data')
            .order_by()
        )
        for system, fields_data in resources:
            system_to_resources[system].add(fields_data['resource_id'])

        return system_to_resources

    def hanging_approved(self, threshold):
        if threshold is None:
            threshold = constance.config.IDM_HANGING_APPROVED_OLD
        else:
            threshold = int(threshold)

        old = timezone.now() - timezone.timedelta(seconds=threshold)
        too_old = timezone.now() - timezone.timedelta(days=3)

        return self.of_operational_system().filter(
            state='approved',
            actions__added__gt=too_old,
        ).annotate(
            last_action_added=Max('actions__added')
        ).filter(last_action_added__lt=old)

    def hanging_depriving(self):
        old = timezone.now() - timezone.timedelta(seconds=constance.config.IDM_HANGING_DEPRIVING_OLD)
        too_old = timezone.now() - timezone.timedelta(days=3)

        return self.of_operational_system().filter(
            state='depriving',
            actions__added__gt=too_old,
            actions__action__in=('expire', 'deprive'),
            actions__error='',
        ).annotate(
            last_action_added=Max('actions__added')
        ).filter(last_action_added__lt=old)

    def of_operational_system_query(self, prefix=''):
        p = lambda string: prefix + string

        q_filter = Q(**{
            p('system__is_broken'): False,
            p('system__is_active'): True,
        })
        return q_filter

    def of_operational_system(self):
        return self.filter(self.of_operational_system_query())

    def parentless(self):
        return self.filter(parent=None)

    def parentful(self):
        return self.exclude(parent=None)

    def deprivable_query(self, prefix=''):
        p = lambda string: prefix + string

        q_filter = Q(**{
            p('state__in'): ROLE_STATE.RETURNABLE_STATES,
        })
        return q_filter

    def deprivable(self):
        return self.filter(self.deprivable_query())

    def personal_by_group(self):
        return self.of_user().filter(parent__group__isnull=False)

    def of_user(self, user=None):
        qs = self.filter(group=None)
        if user is not None:
            qs = qs.filter(user=user)
        return qs

    def of_group(self, group=None, look_up=False):
        qs = self.filter(user=None)
        if group is not None:
            if look_up:
                qs = qs.filter(group__in=group.get_ancestors(include_self=True))
            else:
                qs = qs.filter(group=group)
        return qs

    def of_role_type(self, role):
        if role.user_id is not None:
            qs = self.of_user()
        else:
            qs = self.of_group()
        return qs

    def askable_to_rerequest(self, transfer):
        qs = self.parentless().filter(state='granted').filter(granted_at__lt=transfer.created_at)
        return qs

    def rerunnable(self):
        return self.filter(Q(parent=None) | Q(parent__user__isnull=False) | Q(user__isnull=True)).returnable()

    def of_same_owner(self, subj, look_groups_up=False):
        if subj.is_user:
            qs = self.of_user(subj.user)
        else:
            qs = self.of_group(subj.group, look_up=look_groups_up)
        return qs

    def conflictable(self, subj: 'Subject', node: 'RoleNode', system: 'System' = None):
        qs = self.of_same_owner(subj, look_groups_up=True)
        if system is not None:
            qs = qs.filter(system=system)
        else:
            qs = qs.filter(system=node.system)
        qs = qs.returnable().select_related('node')
        return qs

    def holdable(self):
        return self.active().filter(state='granted')

    def of_inactive_groups(self):
        from idm.users.models import Group
        return self.filter(group__state=Group.DEPRIVED)

    def children_of_inactive_groups(self):
        from idm.users.models import Group
        qs = self.filter(parent__group__state=Group.DEPRIVED)
        return qs

    def children_of_inactive_refs(self):
        qs = self.filter(
            parent__group__isnull=True,
            parent__state__in=ROLE_STATE.ALREADY_INACTIVE_STATES,
        )
        return qs

    def get_roles_of_groups(self, groups):
        return self.filter(group__in=groups)

    def get_inapplicable_robot_roles(self):
        return self.deprivable().personal_by_group().filter(with_robots=False, user__is_robot=True)

    def get_inapplicable_external_roles(self):
        return self.deprivable().personal_by_group().filter(
            with_external=False,
            user__affiliation=AFFILIATION.EXTERNAL,
            user__is_robot=False,
        )

    # border between -able and verbs

    def deprive_or_decline(self, *args, **kwargs):
        assert 'parent_action' in kwargs
        with parent_action_completion(kwargs['parent_action']):
            for role in self.select_related('system', 'parent', 'depriver').iterator():
                try:
                    role.deprive_or_decline(*args, **kwargs)
                except Exception:
                    # почему-то не удалось отозвать роль, но надо попробовать отозвать остальные
                    log.exception('Can\'t deprive role %s.', role)

    def ask_rerequest(self, transfer, parent_action):
        """
        Переводит роли в need_request
        """
        for role in self.select_related('system').iterator():
            role.ask_rerequest(transfer, parent_action=parent_action)

    def keep_granted(self, transfer, parent_action):
        """
        Оставляет роли в granted, но пишет action
        """
        for role in self.select_related('system').iterator():
            role.keep_granted(transfer, parent_action)

    def rerun_workflow(self, parent_action, sync=False):
        if sync:
            self.rerun_workflow_sync(parent_action)
        else:
            self.rerun_workflow_async(parent_action)

    def rerun_workflow_sync(self, parent_action):
        from idm.users.models import User
        robot = User.objects.get_idm_robot()

        for role in self.select_related('system__actual_workflow'):
            role.rerun_workflow(robot, parent_action=parent_action)

    def rerun_workflow_async(self, parent_action):
        from idm.core.tasks.roles import RerunWorkflow
        for role in self:
            RerunWorkflow.apply_async(
                kwargs={'role_id': role.pk, 'parent_action_id': parent_action.id},
                countdown=settings.IDM_PLUGIN_TASK_COUNTDOWN,
            )

    def put_on_hold(self, *args, **kwargs):
        assert 'parent_action' in kwargs
        for role in self:
            role.put_on_hold(*args, **kwargs)

    def hold_or_decline_or_deprive(self, parent_action):
        """
        Если роль ещё не была выдана – отклоняем.
        Если роль уже активна, но должна отзываться мгновенно (without_hold=True) – отзываем
        Если же роль активна и не должна отзываться мгновенно (without_hold=False) – откладываем
        Предполагается, что эта функция вызывается только при удалении групп
        """
        for role in self.filter(
            state__in=ROLE_STATE.NOT_YET_ACTIVE_STATES | ROLE_STATE.ACTIVE_RETURNABLE_STATES
        ).select_related('system', 'parent'):
            if role.state in ROLE_STATE.NOT_YET_ACTIVE_STATES:  # отзываем/отклоняем неактивные
                comment = 'Роль отклонена в связи с удалением группы'
                role.deprive_or_decline(
                    User.objects.get_idm_robot(),
                    comment=comment,
                    bypass_checks=True,
                    deprive_all=True,
                    parent_action=parent_action
                )
            elif role.state in ROLE_STATE.ACTIVE_RETURNABLE_STATES - {ROLE_STATE.ONHOLD}:
                if role.without_hold:
                    comment = 'Роль отозвана в связи с удалением группы'
                    role.deprive_or_decline(
                        User.objects.get_idm_robot(),
                        comment=comment,
                        bypass_checks=True,
                        deprive_all=True,
                        parent_action=parent_action
                    )
                else:
                    comment = 'Роль отложена в связи с удалением группы'
                    role.put_on_hold(comment=comment, parent_action=parent_action)


    def restore_holded_on_group_restore(self, parent_action):
        """Предполагается, что эта функция вызывается только при восстановлении групп"""
        for role in self.filter(state=ROLE_STATE.ONHOLD):
            role.set_state('granted', parent_action=parent_action)

    def rerequest_without_approvers(self, comment):
        from idm.core.models import Role

        failed = []

        for role in self.iterator():
            try:
                role.restore(comment)
            except Exception as e:
                failed.append((role, e))

        Role.objects.poke_hanging_approved_roles()

        return failed

    def review_roles(self, system, expiration_date):
        """Запускает пересмотр ролей в указанной системе"""
        roles = self.all()
        for role in roles:
            assert role.system == system

        log.info('Reviewing %d roles in system "%s"', self.count(), system.slug)
        failed_roles = []
        for role in roles:
            add_to_instance_cache(role.node, 'system', role.system)
            with transaction.atomic():
                try:
                    role.review(expiration_date=expiration_date)
                except Exception as ex:
                    failed_roles.append(role.pk)
                    role.create_action(
                        action=ACTION.REVIEW_ROLE_ERROR,
                        comment=_('Не удалось пересмотреть роль'),
                        error=str(ex)
                    )
                    log.warning(
                        'Role id=%d for %s cannot be rerequested',
                        role.pk,
                        role.get_subject().get_ident(),
                        exc_info=True,
                    )
        return failed_roles



class RoleManager(models.Manager.from_queryset(RoleQuerySet)):
    """
    Менеджер для работы с моделью роли, учитывающий разрешения
    """

    # Dirty crutch for testing b2b
    DEFAULT_ORGANIZATION = None

    def get_filtered_query(self, prefix='',
                           role_ids=None, role_id__gt=None, role_id__lt=None, sox=None, state_set=None, parent=None,
                           states=None, ownership=None, system=None, ancestor_node=None, fields_data=NOT_PROVIDED,
                           search_term=None, internal_role=None, requester=None, nodesets=None,
                           users=None, groups=None, user_type=None,
                           node=NOT_PROVIDED, system_specific=NOT_PROVIDED,
                           with_parents=False, filter_fields_data=None):
        permission_params = {}

        def p(s):
            return prefix + s

        state_set_mapping = {
            'active': Q(**{
                p('is_active'): True
            }),
            'requested': Q(**{
                p('state__in'): ROLE_STATE.REQUESTED_STATES
            }),
            'inactive': Q(**{
                p('is_active'): False, p('state__in'): ROLE_STATE.INACTIVE_STATES - ROLE_STATE.REQUESTED_STATES
            }),
            'returnable': Q(**{
                p('state__in'): ROLE_STATE.RETURNABLE_STATES
            }),
            'almost_active': Q(**{
                p('state__in'): ROLE_STATE.ALMOST_ACTIVE_STATES,
            }),
        }

        q_filter = Q()
        if role_ids is not None:
            q_filter &= Q(**{p('id__in'): role_ids})

        if role_id__gt is not None:
            q_filter &= Q(**{p('id__gt'): role_id__gt})

        if role_id__lt is not None:
            q_filter &= Q(**{p('id__lt'): role_id__lt})

        if sox is not None:
            q_filter &= Q(**{p('system__is_sox'): sox})

        if state_set is not None:
            q_filter &= state_set_mapping[state_set]

        if parent is not None:
            if parent == 'absent':
                q_filter &= Q(**{p('parent'): None})
            elif parent == 'present':
                q_filter &= Q(**{p('parent__isnull'): False})
            elif parent == 'user':
                q_filter &= Q(**{p('parent__user__isnull'): False})
            elif parent == 'group':
                q_filter &= Q(**{p('parent__group__isnull'): False})
            else:
                q_filter &= Q(**{p('parent'): parent})

        if states is not None:
            q_filter &= Q(**{p('state__in'): states})

        if system_specific is not NOT_PROVIDED:
            q_filter &= Q(**{p('system_specific'): system_specific})

        if node is not NOT_PROVIDED:
            q_filter &= Q(**{p('node'): node})

        if fields_data is not NOT_PROVIDED:
            q_filter &= Q(**{
                p('fields_data'): fields_data
            })

        if filter_fields_data is not None:
            for field, value in filter_fields_data.items():
                q_filter &= Q(**{
                    p('fields_data__%s' % field): value
                })

        if ownership is not None:
            if ownership == 'personal':
                q_filter &= Q(**{
                    p('group__isnull'): True
                })
            elif ownership == 'group':
                permission_params['skip_persons'] = True
                q_filter &= Q(**{
                    p('group__isnull'): False
                })

        if system is not None:
            permission_params['system'] = system
            q_filter &= Q(**{
                p('system'): system
            })

            if ancestor_node is not None:
                ancestor_node.fetch_system()
                if search_term is not None:
                    q_filter &= Q(**{
                        p('node__rolenodeclosure_parents__parent_id__in'): ancestor_node.search_grandchildren(
                            search_term
                        )
                    })
                elif ancestor_node.level > 0:
                    q_filter &= Q(**{
                        p('node__value_path__startswith'): ancestor_node.value_path,
                    })

            if internal_role is not None and requester is not None:
                q_filter &= Q(**{
                    p('node__id__in'): self.get_internal_role_query(requester, system, internal_role, 'core.idm_view_roles')
                })

            if nodesets is not None:
                q_filter &= Q(**{
                    p('node__nodeset__in'): nodesets
                })

        if users is not None or groups is not None or user_type is not None:
            # Роли пользователей и групп фильтруются через ИЛИ
            or_q = Q()

            if users is not None:
                if len(users) == 1:
                    permission_params['user'] = users[0]

                user_ids = {user.pk for user in users}
                user_filters = {p('user_id__in'): user_ids}
                if user_type is not None:
                    user_filters[p('user__type')] = user_type
                or_q |= Q(**user_filters)

                if ownership != 'personal':
                    users_of_type = users
                    if user_type is not None:
                        # FIXME: кажется, что потенциальный дополнительный запрос тут – меньшее из зол, но всё равно зло
                        users_of_type = users.filter(type=user_type)

                    aware_system_q = Q(**{
                        p('user'): None,
                        p('group__in'): set(Group.objects.get_membership_query(users_of_type))
                    })

                    if system is None:
                        or_q |= Q(**{
                            p('system__group_policy__in'): SYSTEM_GROUP_POLICY.AWARE_OF_GROUPS}) & aware_system_q
                    elif system.group_policy in SYSTEM_GROUP_POLICY.AWARE_OF_GROUPS:
                        or_q |= aware_system_q
            elif user_type is not None:
                permission_params['user_type'] = user_type
                or_q |= Q(**{p('user__type'): user_type})

            if groups is not None:
                if 'user' in permission_params:
                    del permission_params['user']
                else:
                    permission_params['groups'] = groups

                group_ids = {group.id for group in groups}

                if with_parents:
                    or_q |= Q(**{p('group__groupclosure_children__child_id__in'): group_ids})
                else:
                    or_q |= Q(**{p('group__id__in'): group_ids})

            q_filter &= or_q

        return q_filter, permission_params

    def get_internal_role_query(self, requester, system, internal_role, permission):
        from idm.core.models import InternalRole, RoleNode, InternalRoleUserObjectPermission
        permission_obj = get_permission(permission)

        return maybe_subquery(
            '''
            SELECT
                 nc.child_id
            FROM %(RoleNode)s rn
            JOIN %(InternalRole)s ir ON
                 ir.node_id=rn.id
                 AND rn.level>1
            JOIN %(InternalRoleUserObjectPermission)s uop ON
                 ir.id=uop.content_object_id
            JOIN %(RoleNodeClosure)s nc ON
                 nc.parent_id=rn.id
            WHERE
                 uop.permission_id=%%s
                 AND rn.system_id=%%s
                 AND uop.user_id=%%s
                 AND ir.role=%%s
            ''',
            [permission_obj.pk, system.pk, requester.impersonated.pk, internal_role],
            [
                InternalRole, RoleNode, RoleNode._closure_model, InternalRoleUserObjectPermission,
            ]
        )

    def get_applicable_roles(self, group, include_inactive=False):
        """Фильтруем групповые роли, которые должны быть применены к данной группе"""
        qs = self.filter(group__in=group.get_ancestors(include_self=True), system__group_policy='unaware')
        if not include_inactive:
            # такой способ удалить из активных depriving
            qs = qs.filter(state__in=ROLE_STATE.ACTIVE_RETURNABLE_STATES)
        return qs

    def public_query(self, prefix=''):
        """Публичные роли"""

        def p(s):
            return prefix + s

        return Q(**{p('is_public'): True}) | Q(**{p('is_public'): None, p('node__is_public'): True})

    def get_public_roles(self):
        return self.filter(self.public_query())

    def create_role(
            self,
            subject: Union[User, Group, Requester],
            system: 'System',
            node: 'RoleNode',
            fields_data: Dict[str, Any],
            system_specific: Dict[str, Any] = None,
            ttl_days: int = None,
            ttl_date: datetime.datetime = None,
            review_date: datetime.datetime = None,
            parent: 'Role' = None,
            is_public: bool = None,
            inconsistency: 'Inconsistency' = None,
            save: bool = False,
            organization_id: int = None,
            with_inheritance: bool = True,
            with_robots: bool = True,
            with_external: bool = True,
            without_hold: bool = False,
            request_fields: Dict[str, Any] = None,
    ) -> 'Role':
        if not organization_id:
            organization_id = self.DEFAULT_ORGANIZATION
        subject = subjectify(subject)
        log.info(
            'Create role %s for %s (%s) in system %s, data %s from view',
            node.data, subject.get_ident(), subject.inflect('en'), system.slug, fields_data
        )

        role = self.model(
            system=system,
            node=node,
            is_active=False,
            ttl_days=ttl_days,  # сколько роли жить осталось
            ttl_date=ttl_date,  # когда отозвать роль
            review_date=review_date,
            fields_data=fields_data,
            system_specific=system_specific,
            parent=parent,
            is_public=is_public,
            inconsistency=inconsistency,
            organization_id=organization_id,
            with_inheritance=with_inheritance,
            with_robots=with_robots,
            with_external=with_external,
            without_hold=without_hold,
            request_fields=request_fields,
            **subject.get_role_kwargs()
        )
        role.remove_redundant_ttl()
        role.check_fields_data()
        if save:
            role.save()
        return role

    @transaction.atomic
    def request_role(
            self,
            requester: Union[User, Group, Requester],
            subject: Union[User, Group],
            system: 'System',
            comment: str,
            data: Dict[str, Any],
            fields_data: Dict[str, Any] = None,
            ttl_days: int = None,
            ttl_date: datetime.datetime = None,
            review_date: datetime.datetime = None,
            parent_action: 'Action' = None,
            parent: 'Role' = None,
            is_public: bool = None,
            inconsistency: 'Inconsistency' = None,
            silent: bool = False,
            organization_id: int = None,
            with_inheritance: bool = True,
            with_robots: bool = True,
            with_external: bool = True,
            without_hold: bool = False,
            request_fields: Dict[str, Any] = None,
            from_api: bool = False,
    ) -> 'Role':
        if not organization_id:
            organization_id = self.DEFAULT_ORGANIZATION
        if comment is None:
            comment = ''
        requester = requesterify(requester)
        role, workflow_context = self.request_or_simulate(
            requester, subject, system, data, fields_data, ttl_days, ttl_date,
            review_date, parent, is_public=is_public, save=True,
            inconsistency=inconsistency, organization_id=organization_id,
            with_inheritance=with_inheritance, with_robots=with_robots, with_external=with_external,
            without_hold=without_hold, request_fields=request_fields,
        )

        role.report_conflicts(workflow_context)

        reason_action = role.set_state(
            'requested',
            requester=requester,
            comment=comment,
            parent_action=parent_action,
            from_api=from_api,
        )
        role.fetch_node()
        role.create_approve_requests(
            requester, workflow_context,
            reason_action, comment,
            silent=silent, from_api=from_api,
        )
        return role

    def simulate_role_request(self, requester, subject, system, data, fields_data, ttl_days=None, ttl_date=None,
                              review_date=None, parent=None, is_public=None, silent=False, organization_id=None,
                              with_inheritance=True, with_robots=True, with_external=True,
                              without_hold=False, **kwargs):
        if not organization_id:
            organization_id = self.DEFAULT_ORGANIZATION
        requester = requesterify(requester)
        subject = subjectify(subject)
        role, workflow_result = self.request_or_simulate(
            requester, subject, system, data, fields_data, ttl_days, ttl_date,
            review_date, parent,
            is_public=is_public,
            save=False,
            organization_id=organization_id,
            with_inheritance=with_inheritance,
            with_robots=with_robots,
            with_external=with_external,
            without_hold=without_hold,
        )
        approvers = workflow_result.get('approvers')
        all_approvers = []
        if approvers:
            # в списке есть approver'ы
            # заносим всех аппруверов (объединенных по OR) в all_approvers
            for approvers_group in approvers:
                approvers_or = []
                for approver in approvers_group:
                    # если подтверждающий == запрашивающему, его подтверждение не нужно
                    if approver == requester.impersonated or (subject.is_user and approver == subject.user):
                        break

                    approvers_or.append(approver)
                else:
                    all_approvers.append(approvers_or)

        # На сколько дней выдаётся роль и будет ли выслана sms при ее выдаче
        send_sms = None
        if workflow_result.get('send_sms'):
            send_sms = subject.get_sms_message()

        ttl_days, expire_date = self.calculate_expiration(ttl_days, ttl_date, workflow_result.get('ttl_days'))
        review_days, review_date = self.calculate_expiration(None, review_date, workflow_result.get('review_days'))
        additional = {
            'ttl_days': ttl_days,
            'expire_date': expire_date.strftime('%Y-%m-%dT%H:%M:%S') if expire_date else None,
            'review_days': review_days,
            'review_date': review_date.strftime('%Y-%m-%dT%H:%M:%S') if review_date else None,
            'sms': send_sms,
            'warnings': workflow_result['warnings'],
            'silent': silent,
        }
        return all_approvers, additional

    def calculate_expiration(self, days, date, workflow_days):
        if workflow_days:
            if days:
                return min(days, workflow_days), None
            elif date:
                dt = timezone.now() + datetime.timedelta(days=workflow_days)
                if dt < date:
                    return workflow_days, None
                else:
                    return None, date
            else:
                return workflow_days, None

        else:
            if days:
                return days, None
            elif date:
                return None, date
            else:
                return None, None

    def check_role_request(self, requester, subject, system, node=None, fields_data=None, parent_id=None, is_public=None):
        """Проверяет возможность запросить роль
        """
        subject = subjectify(subject)

        system.check_group_policy(subject)

        if system.is_broken:
            raise exceptions.BrokenSystemError(_('Система "%(system)s" сломана. Роль не может быть запрошена.') % {
                'system': system.get_name()
            })

        elif not system.is_active:
            raise exceptions.InactiveSystemError(_('Система "%(system)s" неактивна. Роль не может быть запрошена.') % {
                'system': system.get_name()
            })

        if not requester.is_allowed_for(system):
            raise exceptions.Forbidden(_(
                'Роль не может быть запрошена, так как '
                '%(impersonator)s не имеет права исполнять роль %(impersonated)s в данной системе'
            ) % {
                'impersonator': requester.impersonator,
                'impersonated': requester.impersonated,
            })

        if not subject.is_active():
            log.warning(
                '%s tried to request role for inactive %s', requester, subject.inflect('en')
            )
            message = _('Роль не может быть запрошена, так как %(subject)s %(reason)s') % {
                'subject': subject.inflect(),
                'reason': subject.get_inactivity_reason()
            }
            raise exceptions.Forbidden(message)

        if node is not None and not node.is_active():
            raise exceptions.Forbidden(_(
                'Роль не может быть запрошена, так как система больше не поддерживает данный узел дерева ролей'
            ))

        if subject.is_detached():
            log.warning(
                '%s tried to request role for detached %s', requester, subject.get_ident()
            )
            message = _('Роль не может быть запрошена, так как у %(subject)s не указано подразделение')
            message %= {
                'subject': subject.inflect('кого-чего')
            }
            raise exceptions.NoDepartmentError(message)

        # запрашивать могут лишь пользователи, у которых есть права на это
        result = shortcuts.can_request_role(requester, subject, system, node, fields_data, parent_id)
        if not result:
            log.warning(
                'requester %s has no rights to request role for %s in system "%s"',
                requester,
                subject.get_ident(),
                system.slug
            )
            message = _('У %(requester)s нет прав на запрос роли для %(subject)s в системе "%(system)s": %(reason)s')
            message %= {
                'requester': subjectify(requester.impersonated).inflect('кого-чего'),
                'subject': subject.inflect('кого-чего'),
                'system': system.slug,
                'reason': force_text(result.message),
            }
            raise exceptions.Forbidden(message)

        if node is not None:
            if not node.is_requestable():
                message = _('Роль не может быть запрошена на данный узел дерева ролей')
                raise exceptions.Forbidden(message)

            if is_public is True and node.is_public is False:
                message = _('Невозможно запросить публичную роль для скрытого узла')
                raise exceptions.Forbidden(message)

    def check_role_uniqueness(self, role):
        already_exists = not role.is_unique(ignore=['system_specific'],
                                            among_states=ROLE_STATE.RETURNABLE_STATES)
        if already_exists:
            raise exceptions.RoleAlreadyExistsError(role, with_details=True)

    def request_or_simulate(self, requester, subject, system, data, fields_data, ttl_days, ttl_date,
                            review_date, parent,
                            is_public=None, save=True, inconsistency=None, organization_id=None,
                            with_inheritance=True, with_robots=True, with_external=True, without_hold=False,
                            request_fields=None):
        """
        Запрашивает от имени requester в системе system для сущности subject роль
        с идентификатором data, данными fields_data, временем жизни ttl_days и ttl_date,
        временем пересмотра review_date, родительской ролью parent.
        Сущность subject может быть пользователем или группой.
        """
        if not organization_id:
            organization_id = self.DEFAULT_ORGANIZATION
        from idm.core.models import RoleNode
        subject = subjectify(subject)
        if subject.is_user and subject.user.type == USER_TYPES.TVM_APP and not system.use_tvm_role:
            raise exceptions.TVMRolePolicyError(_('Система `%s` не поддерживает роли для tvm-приложений') % system.slug)
        if isinstance(data, RoleNode):
            role_node = data
        else:
            try:
                role_node = RoleNode.objects.get_node_by_data(system, data, for_request=True)
            except RoleNode.DoesNotExist as e:
                raise exceptions.RoleNodeDoesNotExist(str(e))

        self.check_role_request(
            requester, subject, system, role_node, fields_data, parent.id if parent else None, is_public
        )
        cleaned_data = role_node.get_valid_fields_data(
            fields_data,
            user=getattr(subject, 'user', None),
            group=getattr(subject, 'group', None),
        )
        system.check_passport_policy(subject, role_node, cleaned_data)

        role = self.create_role(subject, system, role_node, cleaned_data, ttl_days=ttl_days, ttl_date=ttl_date,
                                review_date=review_date, parent=parent,
                                is_public=is_public, inconsistency=inconsistency, organization_id=organization_id,
                                with_inheritance=with_inheritance, with_robots=with_robots, with_external=with_external,
                                without_hold=without_hold, request_fields=request_fields)
        self.check_role_uniqueness(role)

        reason = RUN_REASON.REQUEST if save else RUN_REASON.SIMULATE
        workflow_result = role.apply_workflow(
            requester.impersonated, save=False, reason=reason, request_type=REQUEST_TYPE.REQUEST
        )
        if save:
            role.save()
        return role, workflow_result

    def send_reminders(self):
        returnable_without_hold = ROLE_STATE.ACTIVE_RETURNABLE_STATES - {'onhold'}
        roles_user_can_handle = self.public().of_operational_system().select_related('user', 'group', 'system', 'node')
        # фильтр по перезапрошенным ролям ограничим по дате сверху
        # +1 день, чтоб попали все роли до конца дня, лишние отбросим при итерировании по ролям
        expire_at = timezone.now() + timezone.timedelta(days=max(settings.IDM_REMIND_ABOUT_REREQUESTED_ROLES_DAYS)+1)
        reminderable = (
            roles_user_can_handle
            .filter(
                Q(state__in=('need_request', 'onhold'))
                | Q(state__in=ROLE_STATE.REMINDER_ABOUT_UPCOMING_DEPRIVED_STATES, expire_at__lte=expire_at)
            )
        )

        group_roles = groupby(reminderable.of_group().select_related('parent__group', 'parent__user', 'user', 'group').order_by('group__external_id'), operator.attrgetter('group'))
        # выбираем только пользовательские роли, не являющиеся связанными
        user_roles = groupby(reminderable.of_user().select_related('parent__group', 'parent__user', 'user', 'group').order_by('user__center_id'), operator.attrgetter('user'))
        for subject, roles in chain(group_roles, user_roles):
            subject = subjectify(subject)
            roles_for_state = {'need_request': [], 'onhold': [], 'rerequested': []}
            for role in roles:
                if role.state == 'need_request':
                    roles_for_state['need_request'].append(role)
                elif role.state == 'onhold':
                    is_unique = role.is_unique(ignore=['system_specific', 'parent'], among_states=returnable_without_hold)
                    if subject.is_user and is_unique:
                        roles_for_state['onhold'].append(role)
                else:
                    days_to_deprived = (role.expire_at.date() - timezone.now().date()).days
                    if days_to_deprived in settings.IDM_REMIND_ABOUT_REREQUESTED_ROLES_DAYS:
                        roles_for_state['rerequested'].append(role)
            if roles_for_state['need_request']:
                log.info('Sending reminder about roles that need request to the %(subject)s' % {
                    'subject': subject.get_ident()
                })
                self.send_reminder(subject, roles_for_state['need_request'], EXPIRATION_ROLES_TEMPLATES)
            if roles_for_state['onhold']:
                log.info('Sending reminder about roles that are on hold to the %(subject)s' % {
                    'subject': subject.get_ident()
                })
                if not waffle.switch_is_active('dont_send_onhold_reminders'):
                    self.send_reminder(subject, roles_for_state['onhold'], ONHOLD_ROLES_TEMPLATES)
            if roles_for_state['rerequested']:
                log.info('Sending reminder about roles that rerequest but not approved to the %(subject)s' % {
                        'subject': subject.get_ident()
                    })
                self.send_reminder(
                    subject,
                    sorted(roles_for_state['rerequested'], key=operator.attrgetter('expire_at')),
                    REREQUESTED_ROLES_TEMPLATES,
                )

    def send_reminder(self, subject, roles, templates, **kwargs):
        if subject.is_user:
            subject.user.fetch_department_group()
        try:
            send_reminder(subject, templates, roles=roles, **kwargs)
        except Exception:
            log.exception('Error while trying to send reminder about roles to the %s' % subject.get_ident())

    def poke_hanging_depriving_roles(self, system=None, parent_action=None):
        """Пытаемся подпихнуть роли, зависшие в depriving"""
        date_from = timezone.now() - datetime.timedelta(minutes=settings.DEPRIVING_HANGING_ROLES_MINUTES)

        hanging_depriving_roles = (
            self
            .of_operational_system()
            .filter(state=ROLE_STATE.DEPRIVING, updated__lte=date_from)
        )
        if system:
            hanging_depriving_roles = hanging_depriving_roles.filter(system=system)

        hanging_depriving_roles.deprive_or_decline(
            parent_action=parent_action,
            depriver=User.objects.get_idm_robot(),
            comment=_('Повторный запуск отзыва роли'),
        )

    # non-db side-effects are the last here
    @atomic_retry
    def poke_hanging_approved_roles(self, system=None):
        """Пытаемся подпихнуть роли, зависшие в approved"""
        from idm.core.tasks.roles import RoleAdded

        if yenv.type != 'development.unittest':
            roles_queue_size = get_queue_size('roles')
            if roles_queue_size > settings.IDM_MAX_ROLES_QUEUE_SIZE:
                raise ValueError(f'Roles queue size is too high: {roles_queue_size}')

        hanging_approved_roles = self.of_operational_system().filter(state=ROLE_STATE.APPROVED)
        if system:
            hanging_approved_roles = hanging_approved_roles.filter(system=system)

        for role_id, requester_id in hanging_approved_roles.select_related('last_request').values_list('pk', 'last_request__requester_id').iterator():
            log.info('Processing hanging role %d which was approved but not granted', role_id)
            task_args = {
                'kwargs': {
                    'role_id': role_id,
                },
                'countdown': settings.IDM_PLUGIN_TASK_COUNTDOWN,
            }
            if requester_id and requester_id == settings.CRM_ROBOT:
                task_args['queue'] = settings.PINCODE_QUEUE
                task_args['countdown'] = settings.PINCODE_COUNTDOWN

            RoleAdded.apply_async(**task_args)

    def poke_awaiting_roles(self, system=None, query=None, parent_action=None):
        potential_roles = self.of_operational_system().select_related('system')
        if system:
            potential_roles = potential_roles.filter(system=system)
        if query:
            potential_roles = potential_roles.filter(query)
        awaiting_roles = potential_roles.awaiting().select_related('parent', 'user')
        for role in awaiting_roles.iterator():
            target_state, _ = role.check_awaiting_conditions(passport_login_id=NOT_PROVIDED)
            if target_state == ROLE_STATE.GRANTED:
                role.set_state('approved', comment='Все условия выполнены', parent_action=parent_action)

    def request_applicable_ref_roles(self, system=None, parent_action=None):
        """Пытаемся выдать невыданные связанные роли"""

        potential_roles = self.of_operational_system()
        if system:
            potential_roles = potential_roles.filter(system=system)

        roles_with_refs = potential_roles.filter(
            is_active=True,
            ref_roles__isnull=False,
        ).exclude(
            ref_roles=[]
        ).select_related(
            'system', 'user', 'group', 'node'
        )

        for role in roles_with_refs.iterator():
            role.request_ref_roles(parent_action=parent_action)

    def request_applicable_personal_roles(self, system=None, group_type=None, retry_failed=False):
        """Пытаемся выдать пользовательские роли, выданные по активным групповым ролям,
        но по какой-то причине не выдавшиеся"""

        from idm.users.models import Group, GroupMembership, User
        from idm.core.models import System, Role

        if group_type is None:
            group_types = GROUP_TYPES.USER_GROUPS
        else:
            group_types = [group_type]

        if system:
            system_condition = 'AND system.slug = \'%s\'' % system.slug
        else:
            system_condition = ''

        if waffle.switch_is_active('idm.enable_ignore_failed_roles_on_poke') and not retry_failed:
            retry_failed_condition = "OR child_role.state = 'failed'"
        else:
            retry_failed_condition = ''

        sql = parametrize_sql('''
            SELECT parent_role.id, u.id
            FROM %(Role)s parent_role
            JOIN %(System)s system ON (system.id = parent_role.system_id)
            JOIN %(Group)s g ON (g.id = parent_role.group_id)
            JOIN %(GroupMembership)s groupmembership ON (groupmembership.group_id = parent_role.group_id)
            JOIN %(User)s u ON (u.id = groupmembership.user_id)
            LEFT JOIN %(Role)s child_role
            ON (
                child_role.user_id = groupmembership.user_id
                AND child_role.parent_id = parent_role.id
                AND (child_role.is_active {retry_failed_condition})
                AND child_role.state NOT IN ('onhold', 'depriving_validation')
            )

            WHERE
                child_role.id IS NULL
                AND groupmembership.state = 'active'
                AND g.state = 'active'
                AND g.type IN ({group_types})
                AND parent_role.state IN ({role_states})
                AND NOT system.is_broken
                AND system.group_policy = 'unaware'
                AND system.is_active
                AND (parent_role.with_robots OR NOT u.is_robot)
                AND (parent_role.with_inheritance OR groupmembership.is_direct)
                AND (parent_role.with_external OR NOT u.affiliation = '{EXTERNAL_AFFILIATION}' OR u.is_robot)
                {system_condition}
            ;
        ''', [self.model, System, Group, GroupMembership, User])

        sql = sql.format(
            role_states=','.join("'%s'" % state for state in ROLE_STATE.ACTIVE_RETURNABLE_STATES),
            group_types=','.join("'%s'" % group_type for group_type in group_types),
            system_condition=system_condition,
            EXTERNAL_AFFILIATION=AFFILIATION.EXTERNAL,
            retry_failed_condition=retry_failed_condition
        )

        with connection.cursor() as cursor:
            cursor.execute(sql)

            parent_role_to_user_ids = collections.defaultdict(list)
            for parent_role_id, user_id in cursor.fetchall():
                parent_role_to_user_ids[parent_role_id].append(user_id)

        role_ids = list(parent_role_to_user_ids.keys())
        for role in Role.objects.filter(id__in=role_ids).select_related('system', 'group'):
            role.request_group_roles(user_ids=parent_role_to_user_ids[role.id])

    def deprive_refs_of_inactive_roles(self, system=None):
        """Пытаемся отозвать активные пользовательские роли, выданные по неактивным групповым ролям"""

        comment = _('Отзыв связанной роли, выданной по неактивной родительской роли')

        potential_roles = self.of_operational_system()
        if system:
            potential_roles = potential_roles.filter(system=system)

        inactive_roles_with_active_refs = (
            potential_roles.filter(
                is_active=False,
                refs__is_active=True,
            ).distinct().prefetch_related('refs')
        )
        for role in inactive_roles_with_active_refs.iterator():
            role.deprive_refs(bypass_checks=True, comment=comment)

    def request_or_deprive_personal_roles(self, system=None, group_type=None, retry_failed=False):
        """Запросим недостающие персональные роли, отзовём лишние персональные роли, выданные по групповым"""
        with log_duration(log, 'Requesting applicable roles for group type %s', group_type):
            self.request_applicable_personal_roles(system=system, group_type=group_type, retry_failed=retry_failed)
        with log_duration(log, 'Depriving inapplicable roles for group type %s', group_type):
            self.deprive_inapplicable_personal_roles(system=system, group_type=group_type)

    def deprive_inapplicable_personal_roles(self, system=None, group_type=None):
        """Пытаемся отозвать те персональные роли, которые выданы по групповым, хотя пользователь не входит
        в соответствующие группы"""
        from idm.users.models import Group, GroupMembership, User
        from idm.core.models import System, Action

        if group_type is None:
            group_types = GROUP_TYPES.USER_GROUPS
        else:
            group_types = [group_type]

        # onhold – активный статус, но мы не должны его отзывать при синхронизации.
        # depriving - тоже активный статус, но доотзывом из depriving занимается допинывалка, а не синхронизация
        # depriving_validation - роль уже почти отозвалась
        ignored_states = (ROLE_STATE.ONHOLD, ROLE_STATE.DEPRIVING, ROLE_STATE.DEPRIVING_VALIDATION)

        comment = _('Отзыв персональной роли, выданной по групповой для группы, не являющейся родительской '
                    'ни для одной из групп пользователя')

        if system:
            system_condition = 'AND system.slug = \'%s\'' % system.slug
        else:
            system_condition = ''

        sql = parametrize_sql('''
        SELECT role.id
        FROM %(Role)s role
        JOIN %(System)s system ON (role.system_id = system.id)
        JOIN %(Role)s parent_role ON (role.parent_id = parent_role.id)
        JOIN %(Group)s g ON (parent_role.group_id = g.id)
        LEFT JOIN %(GroupMembership)s gm ON (gm.user_id = role.user_id AND gm.group_id = parent_role.group_id AND gm.state = 'active')
        WHERE
        role.is_active AND
        role.user_id IS NOT NULL AND
        role.state NOT IN ({ignored_states}) AND
        parent_role.state IN ({states}) AND
        g.type IN ({group_types}) AND
        g.state = 'active' AND
        gm.id IS NULL
        {system_condition}
        ''', [self.model, System, Group, GroupMembership])

        sql = sql.format(
            states=','.join("'%s'" % state for state in ROLE_STATE.ACTIVE_RETURNABLE_STATES),
            ignored_states=','.join("'%s'" % state for state in ignored_states),
            group_types=','.join("'%s'" % group_type for group_type in group_types),
            type_specifier='::integer' if connection.vendor == 'postgresql' else '',
            system_condition=system_condition,
            gm_active_states=','.join("'%s'" % gm_active_state for gm_active_state in GROUPMEMBERSHIP_STATE.ACTIVE_STATES),
        )

        with connection.cursor() as cursor:
            cursor.execute(sql)
            pks = cursor.fetchall()
            roles_from_sql = self.filter(pk__in=itertools.chain(*pks)).select_related('system', 'user', 'parent')

        inapplicable_robot_roles = (self.get_inapplicable_robot_roles().select_related('system', 'user', 'parent')
                                    if waffle.switch_is_active('hold_inapplicable_robots_roles')
                                    else self.none())
        inapplicable_external_roles = (self.get_inapplicable_external_roles().select_related('system', 'user', 'parent')
                                       if waffle.switch_is_active('hold_inapplicable_external_roles')
                                       else self.none())
        inapplicable_refs = chain(
            roles_from_sql,
            inapplicable_robot_roles,
            inapplicable_external_roles,
        )

        robot = User.objects.get_idm_robot()
        for ref in inapplicable_refs:
            if ref.should_be_put_on_hold():
                ref.put_on_hold()
            elif ref.system.is_operational():  # не отзываем роли в сломанных и неактивных системах.
                # отзываем от None, считаем, что None означает "робот"
                # FIXME: надо делать это от имени robot-idm
                try:
                    ref.deprive_or_decline(robot, bypass_checks=True, comment=comment)
                except Exception:
                    # в случае ошибки, выставляем состояние onhold
                    ref.put_on_hold(expiration_date=timezone.now())
                    log.exception('Cannot deprive inapplicable role')
        if waffle.switch_is_active('deprive_inapplicable_robots_roles'):
            robot_comment = _('Отзыв роли, не выдаваемой роботам, у пользователя, ставшего роботом')
            parent_action = Action.objects.create(action=ACTION.MASS_ACTION, data={'name': 'deprive_robots'})
            inapplicable_robot_roles.deprive_or_decline(
                robot,
                comment=robot_comment,
                parent_action=parent_action,
            )

        if waffle.switch_is_active('deprive_inapplicable_external_roles'):
            external_comment = _('Отзыв роли, не выдаваемой внешним, у пользователя, ставшего внешним')
            parent_action = Action.objects.create(action=ACTION.MASS_ACTION, data={'name': 'deprive_external'})
            inapplicable_external_roles.deprive_or_decline(
                robot,
                comment=external_comment,
                parent_action=parent_action,
            )

    def get_need_attach_passport_login_groups_id_in_aware_for_membership_systems(self):
        from idm.core.models import RoleNode

        groups_id = set()
        node_id_group_id = list(
            self
            .of_group()
            .active()
            .filter(system__group_policy=SYSTEM_GROUP_POLICY.AWARE_OF_MEMBERSHIPS_WITH_LOGINS)
            .values_list('node_id', 'group_id')
            .distinct()
        )

        nodes_ids = {x for x, _ in node_id_group_id}
        nodes_should_have_login = {
            node.pk: node.role_should_have_login(is_required=True)
            for node in RoleNode.objects.filter(pk__in=nodes_ids)
        }

        for node_id, group_id in node_id_group_id:
            if nodes_should_have_login[node_id]:
                groups_id.add(group_id)

        return groups_id

    def get_need_attach_passport_login_groups_id_in_unaware_systems(self) -> DefaultDict[int, Set[int]]:
        groups_users_id = set(
            self
            .awaiting()
            .filter(parent__group__isnull=False,
                    user__isnull=False)
            .values_list('parent__group_id', 'user_id')
        )
        users_per_groups = collections.defaultdict(set)
        for group_id, user_id in groups_users_id:
            users_per_groups[group_id].add(user_id)
        return users_per_groups

    def remove_roles_when_detaching_resource(self, role):
        from idm.core.models import Action
        from idm.users.models import User

        robot = User.objects.get_idm_robot()
        resource_id = role.fields_data.get('resource_id') if role.fields_data else None
        if resource_id is None:
            return
        resource_type = role.fields_data.get('resource_type')
        query = Q(
            organization_id=role.organization_id,
            system_id=role.system_id,
            fields_data__resource_id=resource_id,
        )
        if resource_type is not None:
            query &= Q(fields_data__resource_type=resource_type)

        parent_action = Action.objects.create(
            action=ACTION.MASS_ACTION,
            data={
                'name': 'remove_roles_when_detaching_resource',
                'resource_id': resource_id,
            }
        )
        self.active().filter(query).deprive_or_decline(
            robot,
            comment=_('Ресурс отвязан от организации'),
            with_push=False,
            parent_action=parent_action,
        )

    def get_roles_for_review(self, system, since_date) -> 'RoleQuerySet':
        if not system.has_review:
            log.error('System "%s" has no review', self.slug)
            return self.none()
        granted_date = since_date - timezone.timedelta(days=system.roles_review_days)
        qs = (
            self.get_public_roles()
            .allowed_for_review()
            .filter(state=ROLE_STATE.GRANTED, parent=None, system=system)
            .filter(
                Q(review_at__isnull=True, granted_at__lte=granted_date) |
                Q(review_at__isnull=False, review_at__lte=since_date)
            )
            .select_related('system__actual_workflow', 'node')
        )
        if system.review_required_default is not None:
            qs = qs.filter(
                Q(node__review_required=True) |
                Q(node__review_required__isnull=True, system__review_required_default=True)
            )
        return qs
