# coding: utf-8

from typing import *

import logging

import waffle
from django.utils.translation import ugettext_lazy as _

from idm.core.constants.system import SYSTEM_GROUP_POLICY
from idm.core.models import GroupMembershipSystemRelation
from idm.core.role_validators.base import BaseValidator

log = logging.getLogger(__name__)


class GroupMembershipsIsActivatedValidator(BaseValidator):
    def check(self, **kwargs) -> Tuple[bool, Optional[str]]:
        """
        Проверяет, что членства запушены в систему перед выдачей роли
        """
        ok_result = (True, None)

        if waffle.switch_is_active('dont_check_memberships_activations'):
            return ok_result

        if self.role.is_user_role():
            return ok_result

        self.role.fetch_system()
        # Тут именно WITHOUT_LOGINS, так как один человек может долго не привязывать свой логин к членству
        if self.role.system.group_policy != SYSTEM_GROUP_POLICY.AWARE_OF_MEMBERSHIPS_WITHOUT_LOGINS:
            return ok_result

        self.role.fetch_group()
        memberships = self.role.group.memberships.active()

        activated_relations = (
            GroupMembershipSystemRelation.objects
            .activated()
            .filter(membership__in=memberships)
        )

        if activated_relations.values('membership_id').distinct().count() == memberships.count():
            return ok_result

        return False, _('Членства в группе ожидают синхронизации с системой')
