# coding: utf-8

import logging

from django.db.models import NOT_PROVIDED
from django.utils.translation import ugettext_lazy as _

from idm.core.constants.groupmembership import GROUPMEMBERSHIP_STATE
from idm.core.workflow.exceptions import PassportLoginPolicyError
from idm.core.models import UserPassportLogin
from idm.core.role_validators.base import BaseValidator
from idm.users.models import GroupMembership

log = logging.getLogger(__name__)


class PassportLoginValidator(BaseValidator):
    def check(self, passport_login_id, **kwargs):
        result = True
        comment = None

        login = self._get_passport_login_to_check_awaiting_condition(passport_login_id)
        if self.role.should_have_login(is_required=True) and login is None:
            result = False
            comment = _('Роль требует указания паспортного логина')
        elif login:
            if not self.role.passport_logins.filter(pk=login.pk).exists():
                result, comment = self._try_add_passport_login_to_role(login)
            if not login.is_fully_registered:
                result = False
                comment = _('Требуется дорегистрировать паспортный логин')
        return result, comment

    def _try_add_passport_login_to_role(self, login):
        try:
            self.role.system.check_passport_policy(self.role.user, self.role.node, direct_passport_login=login)
            self.role.add_passport_login(login.login, approve=False)
            result = True
            comment = None
        except PassportLoginPolicyError:
            result = False
            comment = _('Нарушение политики системы в отношении паспортных логинов')
            # добавление логина нарушает установленные политики, в таком случае логин не прицепляется
            # роль пока не трогаем, но при ближайшем ежедневном синке она уйдёт в ошибку
        except Exception:
            result = False
            comment = _('Возникла ошибка при добавлении паспортного логина к роли')
            log.exception(
                'Could not attach login "%s" to awaiting personal role with pk="%s"',
                login.login,
                self.role.pk,
            )
        return result, comment

    def _get_passport_login_to_check_awaiting_condition(self, passport_login_id):
        if passport_login_id is NOT_PROVIDED:
            # Если passport_login_id не передан, то попробуем получить login из роли или членства в группе
            if self.role.is_personal_role():
                mb = (
                    GroupMembership
                    .objects
                    .filter(user_id=self.role.user_id, group_id=self.role.parent.group_id, state=GROUPMEMBERSHIP_STATE.ACTIVE)
                    .select_related('passport_login')
                    .first()
                )
                login = mb.passport_login if mb else None
            else:
                login_value = self.role.fields_data.get('passport-login') if self.role.fields_data else None
                login = self.role.passport_logins.filter(login=login_value).first() if login_value else None
        else:
            # Если передан passport_login_id, то попробуем получить login по переданному id
            # (используется из таски по выдаче ролей)
            login = UserPassportLogin.objects.get(id=passport_login_id) if passport_login_id else None
        return login
