# coding: utf-8

import logging

from django.db.models import Q
from django.utils.translation import ugettext_lazy as _

from idm.core.constants.role import ROLE_STATE
from idm.core.models import Role
from idm.core.role_validators.base import BaseValidator
from idm.users.constants.user import USER_TYPES

log = logging.getLogger(__name__)


class ResourceAssociateValidator(BaseValidator):
    def check(self, **kwargs):
        """
        Проверяет, что ресурс привязан к организации, прежде чем выдать пользовательскую роль
        """
        result = True
        comment = None
        self.role.fetch_user()
        if self.role.user is not None and self.role.user.type == USER_TYPES.ORGANIZATION:
            # Роль на организацию, проверять ничего не нужно
            return result, comment

        resource_id = self.role.fields_data.get('resource_id') if self.role.fields_data else None
        # Если fields_data не содержит resource_id, то ничего не валидируем
        # Считаем, что запрос роли не на ресурс
        if resource_id is not None:
            resource_type = self.role.fields_data.get('resource_type')
            query = Q(
                organization_id=self.role.organization_id,
                state=ROLE_STATE.GRANTED,
                system_id=self.role.system_id,
                user__type=USER_TYPES.ORGANIZATION,
                fields_data__resource_id=resource_id,
            )
            if resource_type is not None:
                query &= Q(fields_data__resource_type=resource_type)
            if not Role.objects.filter(query).exists():
                result = False
                comment = _('Ресурс, на который запрашивается роль, ещё не привязан к организации')
        return result, comment
