# -*- coding: utf-8 -*-


from django.conf import settings
from django.utils import timezone
from django.utils.translation import ugettext_noop as _
from django_pgaas import atomic_retry

from idm.core.constants.action import ACTION
from idm.core.models import Action
from idm.celery_app import app
from idm.framework.task import BaseTask
from idm.sync.ldap import connector as ldap
from idm.users.models import User
from idm.utils import http
from idm.utils.tasks import get_object_or_fail_task


class DisableAccount(BaseTask):
    """Этот таск запускается для отключения в AD учетной записи уволенного сотрудника."""

    monitor_success = False  # таски этого типа запускаются ad-hoc

    def init(self, username, move_to_old=True, reason=None, **kwargs):
        self.log.info('Disable LDAP account of user %s', username)
        user = get_object_or_fail_task(User, username=username)
        reason_data = {}
        if reason:
            reason_data = {'reason': reason}

        with ldap.NewLDAP() as ldap_connection:
            deactivated, _, _ = ldap_connection.disable_user(
                user,
                move_to_old,
                ad_reason_data=reason_data,
            )

        if deactivated:
            return {'step': 'update_db', 'username': username, 'move_to_old': move_to_old}
        else:
            self.log.warning('Unable to deactivate account of %s. Will try next time.', username)

    # method has no non-db side effects
    @atomic_retry
    def update_db(self, username=None, move_to_old=None, **kwargs):
        user = get_object_or_fail_task(User, username=username)
        if move_to_old:
            user.ldap_active = False
            user.ldap_blocked = False  # пользователь уже не просто заблокирован, а уволен
            user.ldap_blocked_timestamp = timezone.now()
        else:
            user.ldap_blocked = True  # пользователь заблокирован в AD

        user.save(update_fields=('ldap_active', 'ldap_blocked', 'ldap_blocked_timestamp'))
        return {'step': 'logout', 'username': username}

    def logout(self, username, **kwargs):
        """ Завершить паспортные сессии"""

        # IDM-4997: Пока у СИБ нет тестинга мы не должны пытаться делать logout из тестинга и дева
        if settings.PASSPORT_CLOSE_SESSIONS_URL is None:
            self.log.info('Cannot logout user %s in no production environment', username)
            return

        try:
            response = http.post(
                settings.PASSPORT_CLOSE_SESSIONS_URL,
                json={'username': username},
                tvm_id=settings.PASSPORT_CLOSE_SESSIONS_TVM_ID,
            )

            result = response.json()

            if 'error' in result:
                self.log.warning('Cannot logout %s: returned %s "%s"', username, response.code, result['error'])
            else:
                self.log.info('Successfully logged out %s', username)

        except Exception:
            self.log.warning('Cannot logout user %s', username, exc_info=1)


class DismissUser(BaseTask):
    """Этот таск запускается, если мы понимаем, что сотрудник был уволен, и отзывает у него все роли."""

    monitor_success = False  # таски этого типа запускаются ad-hoc

    # method has non-db side effects, they are idempotent
    @atomic_retry
    def init(self, username, **kwargs):
        self.log.info('Depriving all roles for dismissed user %s', username)

        user = get_object_or_fail_task(User, username=username)

        user_roles = user.roles.deprivable().of_operational_system()
        parent_action = Action.objects.create(
            action=ACTION.MASS_ACTION,
            data={'name': 'dismiss_user', 'user': username}
        )
        user_roles.deprive_or_decline(
            None,
            bypass_checks=True,
            deprive_all=True,
            parent_action=parent_action,
            comment=_('Сотрудник уволен'),
        )


DisableAccount = app.register_task(DisableAccount())
DismissUser = app.register_task(DismissUser())
