import logging
from functools import wraps

from django.db import transaction
from django.utils import timezone

from idm.celery_app import app
from idm.core.models import System
from idm.framework.requester import requesterify, Requester
from idm.framework.task import BaseTask, UnrecoverableError
from idm.inconsistencies.models import Inconsistency
from idm.utils.lock import lock
from idm.utils.log import log_duration
from idm.utils.rolenode import deprive_nodes
from idm.utils.tasks import get_object_or_fail_task

log = logging.getLogger(__name__)


ALL_STEPS = [
    'sync_nodes',
    'deprive_nodes',
    'report_unchecked',
    'check_inconsistencies',
    'report_inconsistencies',
    'resolve_inconsistencies',
]


class Next(object):
    nextkwargs = {}

    def __init__(self, **nextkwargs):
        self.nextkwargs = nextkwargs


def step(func):
    step_name = func.__name__
    assert step_name in ALL_STEPS

    @wraps(func)
    def inner(*args, **kwargs):
        system_slug = kwargs['system_slug']
        block = kwargs['block']

        with lock('idm.tasks.everysync.EverySync:%s' % system_slug, block=block) as acquired:
            if not acquired:
                log.info('Cannot acquire lock for system %s, dropping task', system_slug)
                return

            result = func(*args, **kwargs)

            if isinstance(result, Next):
                nextkwargs = result.nextkwargs
                result = kwargs.copy()
                result.update(nextkwargs)

                steps = kwargs.get('steps')
                allowed_steps = (step_ for step_ in ALL_STEPS if step_ in steps)

                # съедаем итератор до текущего степа
                for i, step_ in enumerate(allowed_steps):
                    if step_ == step_name:
                        break

                try:
                    next_step = next(allowed_steps)
                except StopIteration:
                    result = None
                else:
                    result['step'] = next_step

        return result
    return inner


class EverySync(BaseTask):
    # Время пишем кастомным образом в модельку SystemMeta
    monitor_success = False

    @step
    def sync_nodes(self, system_slug, requester, force_nodes=False, **kwargs):
        system = get_object_or_fail_task(System.objects.select_related('metainfo'), slug=system_slug)
        requester = Requester.from_dict(requester)
        sync_key = None
        try:
            has_auto_updated_nodes = system.has_auto_updated_nodes()
            sync_system = has_auto_updated_nodes or force_nodes

            if sync_system:
                self.log.info('Starting role tree synchronization for system %s', system_slug)
                _, _, sync_key = system.synchronize(
                    user=requester.impersonated,
                    force_update=force_nodes
                )
                self.log.info('Role tree synchronization for system %s has successfully finished', system_slug)
            else:
                self.log.info(
                    'System %s was not synchronized, because it has nor auto updated branches neither forced sync',
                    system_slug
                )
        except Exception:
            self.log.exception('Role tree synchronization for system "%s" has failed', system_slug)
        return Next(sync_key=sync_key)

    @step
    @transaction.atomic
    def deprive_nodes(self, system_slug, requester, sync_key=None, from_api=False, **kwargs):
        system = get_object_or_fail_task(System.objects.select_related('metainfo'), slug=system_slug)
        requester = Requester.from_dict(requester)
        with log_duration(self.log, 'Depriving depriving nodes of system %s', system_slug):
            deprive_nodes(system, requester, sync_key, from_api=from_api)
        return Next()

    @step
    def report_unchecked(self, system_slug, **kwargs):
        system = get_object_or_fail_task(System.objects.select_related('metainfo'), slug=system_slug)
        if system.is_unchecked():
            system.report_unchecked()

    @step
    def check_inconsistencies(self, system_slug, requester, threshold=None, log_check_time=False, **kwargs):
        system = get_object_or_fail_task(System.objects.select_related('metainfo'), slug=system_slug)
        requester = Requester.from_dict(requester)
        if not system.is_operational():
            raise UnrecoverableError('System %s is broken. Cannot proceed.' % system_slug)

        if log_check_time:
            system.metainfo.last_check_inconsistencies_start = timezone.now()
            system.metainfo.save(update_fields=('last_check_inconsistencies_start',))

        with log_duration(self.log, 'Checking system %s', system.slug):
            result = False
            try:
                result = Inconsistency.objects.check_system(
                    system,
                    threshold=threshold,
                    requester=requester.impersonated
                )
            except Exception:
                self.log.exception('Error while checking system %s', system.slug)
            else:
                # логгируем только время успешной сверки и только ежедневные ночные сверки
                if log_check_time and result:
                    system.metainfo.last_check_inconsistencies_finish = timezone.now()
                    system.metainfo.save(update_fields=('last_check_inconsistencies_finish',))
        if result is False:
            # сравнение не удалось, нет смысла слать отчёт или разрешать неконсистентности
            return None
        return Next()

    @step
    def report_inconsistencies(self, system_slug, log_check_time=False, **kwargs):
        system = get_object_or_fail_task(System.objects.select_related('metainfo'), slug=system_slug)
        if log_check_time:
            system.metainfo.last_report_inconsistencies_start = timezone.now()
            system.metainfo.save(update_fields=('last_report_inconsistencies_start',))
        result = Inconsistency.objects.send_report(system)
        if log_check_time and result:
            system.metainfo.last_report_inconsistencies_finish = timezone.now()
            system.metainfo.save(update_fields=('last_report_inconsistencies_finish',))
        return Next()

    @step
    def resolve_inconsistencies(self, system_slug, requester, threshold=None, force_roles=False,
                                resolve_in_idm_direct=False, log_check_time=False, **kwargs):
        system = get_object_or_fail_task(System.objects.select_related('metainfo'), slug=system_slug)
        requester = Requester.from_dict(requester)
        if log_check_time:
            system.metainfo.last_resolve_inconsistencies_start = timezone.now()
            system.metainfo.save(update_fields=('last_resolve_inconsistencies_start',))
        result = False
        with log_duration(self.log, 'Resolving inconsistencies in system %s forced=%s', system_slug, force_roles):
            try:
                result = Inconsistency.objects.resolve_system(
                    system,
                    requester=requester.impersonated,
                    threshold=threshold,
                    force=force_roles,
                    resolve_in_idm_direct=resolve_in_idm_direct,
                )
            except Exception:
                self.log.exception('Error while resolving inconsistencies for system %s', system.slug)
        if log_check_time and result:
            system.metainfo.last_resolve_inconsistencies_finish = timezone.now()
            system.metainfo.save(update_fields=('last_resolve_inconsistencies_finish',))


EverySync = app.register_task(EverySync())
