from django.conf import settings
from django.db import connection
from django_pgaas import atomic_retry

from idm.celery_app import app
from idm.core import signals
from idm.core.constants.rolefield import FIELD_STATE
from idm.core.constants.rolenode import ROLENODE_STATE
from idm.core.models import Action, RoleNode, System, SystemRoleField, Transfer
from idm.core.node_pipeline import NodePipeline
from idm.framework.task import BaseTask
from idm.utils.lock import lock
from idm.utils.tasks import get_object_or_fail_task, get_object_or_retry_task


class RoleNodeRemoved(BaseTask):
    default_retry_delay = 60
    monitor_success = False  # таски этого типа запускаются ad-hoc

    def init(self, action_id, rolenode_id, rolenode_path, system_slug):
        action = get_object_or_retry_task(Action.objects.select_related('role_node'), pk=action_id)
        node = action.role_node

        moved_to_id = None
        moved_to = action.data.get('moved_to')
        if moved_to:
            moved_to_id = moved_to['id']

        return {
            'step': 'remove_node',
            'action_id': action.pk,
            'node_id': node.pk,
            'moved_to_id': moved_to_id,
        }

    # method has no non-db side effects
    @atomic_retry
    def remove_node(self, action_id, node_id, moved_to_id):
        action = get_object_or_retry_task(Action, pk=action_id)
        node = get_object_or_retry_task(RoleNode, pk=node_id)

        if node.state != ROLENODE_STATE.DEPRIVING:
            return

        node.fetch_system()
        node.remove_fields('*', requester=None, sync_key=None)
        node.deprive_emailed_at = None
        node.state = ROLENODE_STATE.DEPRIVED
        node.save(update_fields=['deprive_emailed_at', 'state', 'updated_at'])
        node.fetch_nodeset()
        if node.nodeset and node.nodeset.is_active:
            nodes = RoleNode.objects.filter(state='active', nodeset=node.nodeset)
            if not nodes.exists():
                node.nodeset.is_active = False
                node.nodeset.save(update_fields=['is_active'])

        return {
            'step': 'remove_aliases_and_responsibilities',
            'action_id': action.pk,
            'node_id': node.pk,
            'moved_to_id': moved_to_id,
        }

    @atomic_retry
    def remove_aliases_and_responsibilities(self, action_id, node_id, moved_to_id):
        node = get_object_or_retry_task(RoleNode.objects.prefetch_related('responsibilities__user'), pk=node_id)
        node.fetch_system()
        node.remove_aliases('*', requester=None, sync_key=None)
        node.remove_responsibilities('*', requester=None, sync_key=None)

        return {
            'step': 'deprive_roles',
            'action_id': action_id,
            'node_id': node.pk,
            'moved_to_id': moved_to_id,
        }

    def deprive_roles(self, action_id, node_id, moved_to_id):
        action = Action.objects.select_related('requester').get(pk=action_id)
        node = RoleNode.objects.get(pk=node_id)
        moved_to = RoleNode.objects.get(pk=moved_to_id) if moved_to_id else None

        # Отзываем все привязанные к узлу роли.
        # Если оба узла были листовыми, то к этому моменту роли уже были бы привязаны к новому узлу.
        # С другой стороны, новый узел с тех пор мог успеть стать нелистовым,
        # и если он им действительно стал, надо забрать роли назад и отозвать их.
        if moved_to:
            if not moved_to.is_requestable():
                moved_to.roles.update(node=node)
            reason = 'target node is not requestable'
        else:
            reason = 'node is deprived'

        from_api = action.data.get('from_api', False)
        node.deprive_roles_async(action.requester, from_api, reason)

        return {
            'step': 'post_remove_node',
            'action_id': action_id,
            'node_id': node.pk,
            'moved_to_id': moved_to_id,
        }

    def post_remove_node(self, action_id, node_id, moved_to_id):
        action = Action.objects.get(pk=action_id)
        node = RoleNode.objects.get(pk=node_id)
        moved_to = RoleNode.objects.get(pk=moved_to_id) if moved_to_id else None

        # Пушим узлы в интрапоиск и, быть может, делаем что-нибудь ещё
        signals.role_node_deprived.send(sender=node, role_node=node, moved_to=moved_to, action=action)


class RoleNodeRolesDeprived(BaseTask):
    default_retry_delay = 60
    monitor_success = False  # таски этого типа запускаются ad-hoc

    def init(self, action_id):
        action = get_object_or_retry_task(Action.objects.select_related('role_node', 'role_node__system'), pk=action_id)
        node = action.role_node
        reason = action.data['reason']

        self.log.info('Depriving all roles for node %s[%d] in system %s, because: %s',
                      node.slug_path, node.pk, node.system, reason)

        node.deprive_deleted_roles()


class ResolveTransfersTask(BaseTask):
    """Таск для разрешения трансферов. TODO: Удалить этот класс"""
    monitor_success = False  # таски этого типа запускаются ad-hoc

    def init(self, pks, decision, requester_username):
        from idm.users.models import User

        assert decision in ('accept', 'reject')
        queryset = Transfer.objects.filter(pk__in=pks).old().undecided()
        if decision == 'accept':
            method = queryset.accept
        else:
            method = queryset.reject
        requester = User.objects.users().get(username=requester_username)
        method(requester)


class RecalcNodePipelineTask(BaseTask):
    monitor_success = False  # таски этого типа запускаются слишком часто и могут отпинываться из-за лока

    def init(self, system_id):
        system = System.objects.get(pk=system_id)
        NodePipeline(system).locked_run(block=False)


class CreateRoleFieldIndex(BaseTask):
    monitor_success = False

    def init(self, field_id):
        self.log.info('Creating index for role fields, system field id %s', field_id)
        rolefield = get_object_or_fail_task(SystemRoleField, pk=field_id)
        slug_for_index = preprocess_field_slug(rolefield.slug)
        if not rolefield.state == FIELD_STATE.CREATED:
            self.log.info("Wrong field state '%s', skip it", rolefield.state)
            return

        concurrently = ' CONCURRENTLY ' if settings.CREATE_INDEX_CONCURRENTLY else ' '
        idx_name = 'role_fields_data_system_{}_{}_idx'.format(rolefield.system_id, slug_for_index)
        drop_query = 'DROP INDEX{}IF EXISTS {}'.format(concurrently, idx_name)
        create_query = ('CREATE INDEX{}IF NOT EXISTS {} ON upravlyator_role ((fields_data->%s)) '
                     'WHERE system_id=%s'.format(concurrently, idx_name))

        with lock('idm.tasks.nodes.RoleFieldIndex:{}'.format(idx_name), block=False):
            with connection.cursor() as cursor:
                cursor.execute(drop_query)  # Удаляем индекс, на случай если он существует, но не валиден
                cursor.execute(create_query, [rolefield.slug, rolefield.system_id])

            rolefield.state = FIELD_STATE.ACTIVE
            rolefield.save(update_fields=['state', 'updated_at'])
            self.log.info('Index {} is successfully created'.format(idx_name))


class DropRoleFieldIndex(BaseTask):
    monitor_success = False

    def init(self, field_id):
        self.log.info('Drop index for role fields, system field id %s', field_id)
        rolefield = get_object_or_fail_task(SystemRoleField, pk=field_id)
        slug_for_index = preprocess_field_slug(rolefield.slug)
        if not rolefield.state == FIELD_STATE.DEPRIVING:
            self.log.info("Wrong field state '%s', skip it", rolefield.state)
            return

        concurrently = ' CONCURRENTLY ' if settings.CREATE_INDEX_CONCURRENTLY else ' '
        idx_name = 'role_fields_data_system_{}_{}_idx'.format(rolefield.system_id, slug_for_index)
        sql_query = 'DROP INDEX{}IF EXISTS {}'.format(concurrently, idx_name)

        with lock('idm.tasks.nodes.RoleFieldIndex:{}'.format(idx_name), block=False):
            with connection.cursor() as cursor:
                cursor.execute(sql_query)

                rolefield.state = FIELD_STATE.DEPRIVED
                rolefield.save(update_fields=['state', 'updated_at'])
                self.log.info('Index {} is successfully dropped'.format(idx_name))


def preprocess_field_slug(field_slug: str) -> str:
    return field_slug.replace('-', '_')


CreateRoleFieldIndex = app.register_task(CreateRoleFieldIndex())
DropRoleFieldIndex = app.register_task(DropRoleFieldIndex())
RecalcNodePipelineTask = app.register_task(RecalcNodePipelineTask())
ResolveTransfersTask = app.register_task(ResolveTransfersTask())
RoleNodeRemoved = app.register_task(RoleNodeRemoved())
RoleNodeRolesDeprived = app.register_task(RoleNodeRolesDeprived())
