from django_pgaas import atomic_retry
from django.db import transaction

from idm.celery_app import app
from idm.core.constants.passport_login import PASSPORT_LOGIN_STATE, CREATED_FLAG_REASON
from idm.core.models import UserPassportLogin
from idm.core.tasks.roles import RegisterPassportLoginMixin
from idm.framework.task import BaseTask
from idm.sync import passport
from idm.users.models import GroupMembership
from idm.utils.tasks import get_object_or_fail_task
from idm.utils.lock import lock


class SubscribePassportLogin(BaseTask):
    """Подписывает паспортный логин (sid=67)"""
    max_retries = 10
    default_retry_delay = 10
    monitor_success = False  # таски этого типа запускаются ad-hoc

    # method has
    @atomic_retry
    def init(self, passport_login, **kwargs):
        self.log.info('Trying to subscribe passport login %s', passport_login)
        login = get_object_or_fail_task(UserPassportLogin, login=passport_login)

        try:
            successful = passport.set_strongpwd(passport_login)
            if successful:
                self.log.info('Mark passport login %s as subscribed', passport_login)

                login.state = PASSPORT_LOGIN_STATE.SUBSCRIBED
                login.save()

                login.actions.create(action='subscribed')

        except passport.UnknownLoginError:
            self.log.warning('Passport login %s is unknown', passport_login)
            UserPassportLogin.objects.filter(login=passport_login).update(state=PASSPORT_LOGIN_STATE.UNKNOWN)


class UnsubscribePassportLogin(BaseTask):
    """Снимает подписку с логина (sid=67)"""
    max_retries = 10
    default_retry_delay = 10
    monitor_success = False  # таски этого типа запускаются ad-hoc

    @atomic_retry
    def init(self, passport_login, **kwargs):
        self.log.info('Trying to unsubscribe passport login %s', passport_login)
        login = get_object_or_fail_task(UserPassportLogin, login=passport_login)

        try:
            successful = passport.remove_strongpwd(passport_login)
            if successful:
                self.log.info('Mark passport login %s as unsubscribed', passport_login)

                login.state = PASSPORT_LOGIN_STATE.UNSUBSCRIBED
                login.save()

                login.actions.create(action='unsubscribed')

        except passport.UnknownLoginError:
            self.log.warning('Passport login %s is unknown', passport_login)
            UserPassportLogin.objects.filter(login=passport_login).update(state=PASSPORT_LOGIN_STATE.UNKNOWN)


class RegisterPassportLogin(RegisterPassportLoginMixin, BaseTask):
    """Регистрирует паспортный логин в паспорте"""
    max_retries = 10
    default_retry_delay = 10
    monitor_success = False  # таски этого типа запускаются ad-hoc

    @atomic_retry
    def init(self, passport_login, membership_id, **kwargs):
        membership = GroupMembership.objects.select_related('user').get(pk=membership_id)
        self.init_attributes(membership=membership)
        with lock('idm.tasks.roles.RoleAdded.add_passport_login:%s' % passport_login, block=True):
            self.check_or_register_login(passport_login)
            # Добавляем логин в базу
            with transaction.atomic():
                login = membership.user.passport_logins.create(
                    login=passport_login,
                    state=PASSPORT_LOGIN_STATE.NEW,
                    created_by_idm=True,
                    created_flag_reason=CREATED_FLAG_REASON.EXACT,
                )
                membership.passport_login = login
                membership.save(update_fields=['passport_login'])
                self.log.info(
                    'Passport login <%s> successfully created and assigned with <%s> membership',
                    passport_login, membership.pk
                )

        return {
            'step': 'poke_personal_roles',
            'membership_id': membership_id,
        }

    @atomic_retry
    def poke_personal_roles(self, membership_id, **kwargs):
        membership = GroupMembership.objects.select_related('passport_login', 'user').get(pk=membership_id)
        membership.poke_awaiting_personal_roles()
        membership.deprive_roles_with_old_logins()


RegisterPassportLogin = app.register_task(RegisterPassportLogin())
SubscribePassportLogin = app.register_task(SubscribePassportLogin())
UnsubscribePassportLogin = app.register_task(UnsubscribePassportLogin())
