import ast
import itertools
import re
from typing import Union, Collection, Iterator, List

from django import template
from django.conf import settings
from django.template.base import Parser, Token
from django.utils.translation import gettext_lazy as _

from idm.core.models import Role
from idm.core.workflow.common.subject import subjectify
from idm.users import ranks
from idm.users.models import User, Group
from idm.utils.human import format_username
from idm.utils.i18n import LANG_UNIONS, ROLE_VERB

register = template.Library()


@register.filter
def union(value: str) -> str:
    return LANG_UNIONS.get(value, value)


@register.filter
def role_action_verb(user: User, state: str) -> str:
    sex = 'M'
    if isinstance(user, User):
        sex = user.sex
    return ROLE_VERB.get(state, {'M': _(state), 'F': _(state)}).get(sex or 'M')


@register.filter
def who(user_or_login: Union[str, int, User, Group], form: str = 'кто') -> str:
    """Темплейттег, работает аналогично фильтру username, но результат не форматируется как html"""
    return format_username(user_or_login, form)


@register.filter
def inflect(word: str, form: str = 'кто') -> str:
    return inflect(form, word)


@register.filter
def ident(subject: Union[User, Group]) -> str:
    return subjectify(subject).get_ident()


@register.filter
def sub(a: int, b: int) -> int:
    return a - b


@register.filter
def check_hidden(groups: Collection[Collection[Group]]) -> bool:
    return any(len(group) >= settings.HIDE_AFTER for group in groups)


@register.filter
def get_personal_pronoun_for_groups(approver_group: List[List[User]]) -> str:
    if len(approver_group) != 1:
        return 'их'
    return get_personal_pronoun(approver_group[0])


@register.filter
def get_personal_pronoun(approver_group: List[User]) -> str:
    if len(approver_group) != 1:
        return 'их'
    return approver_group[0].is_female() and 'её' or 'его'


@register.filter
def main_approvers_are_notified(pronoun: str) -> str:
    return {
        'его': _('Основной подтверждающий, оповещен'),
        'её': _('Основная подтверждающая, оповещена'),
        'их': _('Основные подтверждающие, оповещены'),
    }[pronoun]


@register.simple_tag
def absolute_uri(relative_url: str = '/') -> str:
    return settings.IDM_BASE_URL + relative_url


@register.tag('line')
class LineNode(template.Node):
    def __init__(self, parser: Parser, token: Token):
        self.nodelist = parser.parse(('endline',))
        parser.delete_first_token()

    def render(self, context: template.Context) -> str:
        output = self.nodelist.render(context).strip()
        return re.sub(r'\s+', ' ', output.strip())


@register.tag('lines')
class LinesNode(template.Node):
    def __init__(self, parser: Parser, token: Token):
        self.strip = False
        if len(options := token.split_contents()) > 1:
            self.strip = 'strip' in options
        self.nodelist = parser.parse(('endlines',))
        parser.delete_first_token()

    def render(self, context: template.Context) -> str:
        output = []
        for line in self.nodelist.render(context).splitlines():
            if self.strip:
                line = line.strip(' ')
            if line:
                output.append(line)
        return '\n'.join(output)


@register.tag('stripbreaks')
def paragraphize(parser: Parser, token: Token) -> template.Node:
    nodelist = parser.parse(('endstripbreaks',))
    parser.delete_first_token()
    return StripBreaksNode(nodelist)


class StripBreaksNode(template.Node):
    def __init__(self, nodelist: template.NodeList):
        self.nodelist = nodelist

    def render(self, context: template.Context) -> str:
        lines = self.nodelist.render(context).splitlines()
        prev = False
        output = []
        for line in lines:
            stripped = line.strip()
            if stripped:
                output.append(line)
                prev = False
            elif not prev:
                prev = True
                output.append(stripped)
        return '\n'.join(output)


@register.filter
def joined_responsibles(group: Group) -> str:
    return ', '.join(user.get_full_name() for user in group.get_responsibles(ranks.HEAD_OR_DEPUTY))


@register.filter
def request_link(role: Role) -> str:
    link = settings.IDM_BASE_URL + '/#rf=1,rf-mode=personally,rf-role=%s@%s%s' % (
        role.user.username, role.system.slug, role.node.value_path.rstrip('/')
    )
    if role.fields_data:
        for k, v in role.fields_data.items():
            link += ';%s:%s' % (k, v)
    return link


@register.filter
def islice(iterable, stop) -> Iterator:
    return itertools.islice(iterable, stop)


@register.tag('strip')
class StripNode(template.Node):
    def __init__(self, parser: Parser, token: Token):
        options = token.split_contents()
        if len(options) > 1:
            try:
                self.strip_chars = ast.literal_eval(options[1])
            except SyntaxError:
                raise SyntaxError(
                    'Argument to strip tag ("{0}") is not a valid python string literal'.format(options[1])
                )
        else:
            self.strip_chars = None
        self.nodelist = parser.parse(('endstrip',))
        parser.delete_first_token()

    def render(self, context: template.Context) -> str:
        content = self.nodelist.render(context)
        if self.strip_chars:
            return content.strip(self.strip_chars)
        return content.strip()
