import itertools

from django.utils import timezone
from django.utils.translation import ugettext

from idm.core.constants.role import ROLE_STATE
from idm.utils.attributes import get_attribute
from idm.utils.i18n import get_localized_fieldname


def get_localized_value(obj, field, lang=None):
    field = get_localized_fieldname(field, lang=lang)
    return get_attribute(obj, [field])


def get_localized_node_name(node, lang=None):
    return get_localized_value(node, 'name', lang=lang)


def get_localized_node_description(node, lang=None):
    return get_localized_value(node, 'description', lang=lang)


def get_system_state(system):
    if get_attribute(system, ['is_active']):
        if get_attribute(system, ['is_broken']):
            state = 'broken'
        else:
            state = 'active'
    else:
        state = 'inactive'
    return state


def humanize_node(node, lang=None, format='full'):
    assert format in ('short', 'full')

    fieldname = get_localized_fieldname('name', lang)
    fullname = get_attribute(node, ['fullname'])
    is_key = get_attribute(node, ['is_key'])

    keys = [item[fieldname] for item in fullname[::2]]
    values = [item[fieldname] for item in fullname[1::2]]
    tuples = list(itertools.zip_longest(keys, values))
    result = ''
    if format == 'full':
        if is_key:
            length = len(tuples)
            result = ', '.join('%s: %s' % (key.capitalize(), value) if index != length - 1
                               else key.capitalize()
                               for index, (key, value) in enumerate(tuples))
        else:
            result = ', '.join('%s: %s' % (key.capitalize(), value) for key, value in tuples)
    elif format == 'short':
        values = [value for _, value in tuples]
        if is_key:
            values[-1] = ugettext('*Ключ*')
        result = ' / '.join(values)
    return result


def get_role_review_at(role):
    review_at = None
    try:
        parent_id = get_attribute(role, ['parent_id'])
    except KeyError:
        parent_id = get_attribute(role, ['parent', 'id'])

    if (
        get_attribute(role, ['state']) == 'granted' and
        get_attribute(role, ['system', 'has_review']) and
        parent_id is None and
        get_attribute(role, ['granted_at']) and
        is_public_role(role)
    ):
        if get_attribute(role, ['review_at']):
            review_date = get_attribute(role, ['review_at'])
        else:
            roles_review_days = timezone.timedelta(get_attribute(role, ['system', 'roles_review_days']))
            review_date = get_attribute(role, ['granted_at']) + roles_review_days

        if not get_attribute(role, ['expire_at']) or review_date < get_attribute(role, ['expire_at']):
            review_at = review_date

    return review_at


def humanize_role(role, lang=None, format='full'):
    assert format in ('short', 'full')

    node = get_attribute(role, ['node'])
    if node is not None:
        humanized = humanize_node(node, lang=lang, format=format)
    else:
        humanized = _('Неизвестная роль')

    return humanized


def humanize_role_state(role):
    return ROLE_STATE.STATE_CHOICES[get_attribute(role, ['state'])]


def is_public_role(role):
    role_is_public = get_attribute(role, ['is_public'])
    if role_is_public:
        return True

    return role_is_public is None and get_attribute(role, ['node', 'is_public'])


def update_model(obj, fields):
    """Обновляет поля объекта obj в соответствии со словарем fields.
    Возвращает True, если хотя бы одно из полей изменилось"""
    changed_fields = []
    for key, value in fields.items():
        old_value = getattr(obj, key)

        if old_value != value:
            setattr(obj, key, value)
            changed_fields.append(key)
    if changed_fields:
        obj.save(update_fields=changed_fields)
    return bool(changed_fields)


def create_or_update_model(model: 'Model', obj_filter: dict, defaults: dict):
    obj = model.objects.filter(**obj_filter).first()
    if not obj:
        obj = model.objects.create(**obj_filter, **defaults)
    else:
        update_model(obj, defaults)
    return obj
