# coding: utf-8


from django.core.exceptions import ValidationError
from django.core.validators import validate_email
from django.utils.encoding import force_text

from idm.core.workflow.exceptions import RecipientValidationError
from idm.core.constants.role import ROLE_STATE
from idm.core.workflow.sandbox.serializable import Serializable, serialize
from idm.framework.mixins import Representable


def recipientify(thing):
    if isinstance(thing, Recipient):
        result = thing
    elif isinstance(thing, str):
        result = Recipient(thing, granted=True)
    elif hasattr(thing, 'email'):
        result = Recipient(thing.email, granted=True)
    else:
        raise RecipientValidationError('Cannot recipientify %s' % repr(thing))
    return result


class Recipient(Representable, Serializable):
    def __init__(self, email, lang='ru', pass_to_personal=False, **kwargs):
        if isinstance(email, str):
            email = force_text(email)
        elif hasattr(email, 'email'):
            email = force_text(email.email)
        else:
            raise RecipientValidationError('Cannot recipientify %s' % repr(email))
        try:
            validate_email(email)
        except ValidationError:
            raise RecipientValidationError('Wrong email: %s' % email)
        self.email = email

        if lang not in ('ru', 'en'):
            raise RecipientValidationError('Wrong language key for recipientify: %s' % lang)
        self.language = lang

        self.pass_to_personal = pass_to_personal

        if 'states' in kwargs:
            self.validate_states(kwargs['states'])
            self.states = kwargs['states']
        else:
            self.validate_states(list(kwargs.keys()))
            self.states = {key for key, value in kwargs.items() if value}

    def __str__(self):
        return 'email=%s, lang=%s, pass_to_personal=%s, states=%s' % (
            self.email,
            self.language,
            self.pass_to_personal,
            ', '.join(self.states)
        )

    def validate_states(self, keys):
        for key in keys:
            if key not in ROLE_STATE.ALL_STATES and key != 'reminders':
                raise RecipientValidationError('Wrong key for recipientify: %s' % key)

    @classmethod
    def from_dict(cls, data, context=None):
        return cls(**data)

    def as_dict(self):
        return {
            'email': self.email,
            'lang': self.language,
            'pass_to_personal': self.pass_to_personal,
            'states': serialize(self.states),
        }
