# coding: utf-8

from __future__ import unicode_literals

import logging
import operator
import pprint
import types

from django.core.exceptions import PermissionDenied
from django.utils.encoding import force_text
from django.utils.translation import ugettext as _
from six import python_2_unicode_compatible

try:
    from idm.core.constants.workflow import REQUEST_TYPE
except ImportError:
    from workflow_constants import REQUEST_TYPE

log = logging.getLogger(__name__)


class Forbidden(PermissionDenied):
    def __init__(self, message=None, data=None):
        super(PermissionDenied, self).__init__(message)
        self.data = data


@python_2_unicode_compatible
class BaseRuntimeError(RuntimeError):
    """ Базовый класс для исключений, который нормально работает с Unicode строками.
    """

    def __init__(self, *args, **kwargs):
        args = tuple(force_text(a, strings_only=True) for a in args)
        kwargs = {key: force_text(value) for key, value in kwargs.items()}
        super(BaseRuntimeError, self).__init__(*args, **kwargs)

    def __str__(self):
        result = ''
        if len(self.args) == 1:
            result = force_text(self.args[0])
        elif len(self.args) > 1:
            result = force_text(self.args)
        return result

    def __repr__(self):
        return '{classname:s}{repr:s}'.format(classname=type(self).__name__, repr=repr(self.args))


class RoleRequestError(BaseRuntimeError):
    """Ошибка при запросе роли"""


class WorkflowError(RoleRequestError):
    """Ошибка при запросе роли, связанная с прогоном workflow"""

    # а-ля Serializable, только без зависимостей от него
    @classmethod
    def from_dict(cls, data, context=None):
        try:
            from idm.core.workflow.sandbox.serializable import deserialize
        except ImportError:
            from serializable import deserialize
        args = deserialize(data['args'])
        return cls(*args)

    def as_dict(self):
        try:
            from idm.core.workflow.sandbox.serializable import serialize
        except ImportError:
            from serializable import serialize
        return {'args': serialize(self.args)}


class Return(WorkflowError):
    """завершить workflow, исползуется место обычного return"""


class WorkflowDismissiedApproverError(WorkflowError):
    def __init__(self, node_name, request_type, users_str):
        if request_type == REQUEST_TYPE.REQUEST:
            err_msg_action_verb = _('подтвердить')
            err_msg_users = _('следующие подтверждающие')
        else:
            err_msg_action_verb = _('отозвать')
            err_msg_users = _('все кто может отозвать')
        super(WorkflowDismissiedApproverError, self).__init__(
            'Невозможно {err_msg_action_verb} роль {node_name}, из-за того, что {err_msg_users} '
            'уволены: {users_str}. '
            'Ответственные за систему уже предупреждены.'.format(
                err_msg_action_verb=err_msg_action_verb,
                node_name=node_name,
                err_msg_users=err_msg_users,
                users_str=users_str))


class WorkflowContainerInitError(WorkflowError):
    """Ошибка или таймаут при поднятии контейнера с workflow"""


class WorkflowContainerTimeoutError(WorkflowError):
    """Ошибка при получении таймаута из контейнера с выполняющимся workflow"""


class WorkflowContainerProtocolError(WorkflowError):
    """Ошибка при получении некорректного ответа от контейнера (ошибки или неправильного типа ответа)"""


class WorkflowForbiddenOperationError(WorkflowError):
    """Функция или атрибут недоступны из workflow"""


class WorkflowForbiddenSystemError(WorkflowError):
    """Данные о системе недоступны из workflow текущей системы"""


class NoMobilePhoneError(WorkflowError):
    """Ошибка, возникающая, если в WF прописано, что нужно слать sms, а у пользователя не прописан номер мобильного"""


class BrokenSystemError(BaseRuntimeError):
    """Роль не может быть запрошена, так как система сломана.
    Ошибка возникает не только при запросе роли, поэтому это не подкласс RoleRequestError"""


class InactiveSystemError(RoleRequestError):
    """Роль не может быть запрошена, так как система неактивна"""


class ApproverNotFoundError(WorkflowError):
    """Роль не может быть запрошена, так как подтверждающий не найден"""

    def __init__(self, msg, username):
        self.msg = msg
        self.username = username


class NoBossError(WorkflowError):
    """Общая ошибка для проблем с запросом информации о главе и/или заместителях департамента"""


class BossNotFoundError(NoBossError):
    """При прогоне WF была запрошена информация о главе подразделения, а у подразделения нет главы"""


class NoDepartmentError(NoBossError):
    """У пользователя нет департамента"""


class NoGroupError(NoBossError):
    """Запрошенная группа не существует или неактивна"""


class AccessDenied(WorkflowError):
    """Доступ запрещён"""


class NoApproversDefined(WorkflowError):
    """В workflow не описаны подтверждающие"""


class TooManyApproversDefined(WorkflowError):
    """Для данной роли определено слишком много подтверждающих"""


class RoleAlreadyExistsError(RoleRequestError):
    """У пользователя уже есть такая роль"""

    def __init__(self, role, with_details=False):
        from idm.core.constants.role import ROLE_STATE
        subject = role.get_subject()
        message = _('У %(subject)s уже есть такая роль (%(role)s) в системе "%(system)s"') % {
            'subject': subject.inflect('кого-чего'),
            'role': role.node.humanize(),
            'system': role.system.get_name(),
        }
        detailed_message = _('У %(subject)s уже есть такая роль (%(role)s) в системе "%(system)s" '
                             'в состоянии "%(state)s"')
        if with_details:
            alike_roles = role.get_alike(ignore=['system_specific'], among_states=ROLE_STATE.RETURNABLE_STATES)
            alike_role = alike_roles.first()
            if alike_role is not None:
                message = detailed_message % {
                    'subject': subject.inflect('кого-чего'),
                    'role': role.node.humanize(),
                    'system': role.system.get_name(),
                    'state': alike_role.get_state_display(),
                }
        super(RoleAlreadyExistsError, self).__init__(message)


class GroupHasNoParentError(WorkflowError):
    """У группы нет родительской, так как группа корневая"""


class RoleNodeDoesNotExist(WorkflowError):
    """Узел не найден"""


class GroupDoesNotExist(WorkflowError):
    """Группа не найдена"""


class NodeHasNoParentError(WorkflowError):
    """У узла нет родительского, так как он корневой"""


class RecipientValidationError(WorkflowError):
    """Ошибка в описании email_cc"""


class BaseReferenceRolesValidationError(WorkflowError):
    """Базовая ошибка в ref_roles"""


class NotifyReferenceRolesValidationError(BaseReferenceRolesValidationError):
    """Ошибка в ref_roles для которой нужно отправлять письмо"""


class SilentReferenceRolesValidationError(BaseReferenceRolesValidationError):
    """Ошибка в ref_roles, для которой не нужно отправлять письмо"""


class ConflictValidationError(WorkflowError):
    """Ошибка в описании правила конфликта"""


@python_2_unicode_compatible
class DataValidationError(RoleRequestError):
    """Предоставленные для запроса роли данные не прошли проверку"""

    def __init__(self, message, errors):
        self.message = message
        self.errors = errors

    def __str__(self):
        return force_text(self.message)


class PassportLoginPolicyError(RoleRequestError):
    """Выдача роли запрещена паспортной политикой системы"""


class GroupPolicyError(RoleRequestError):
    """Выдача роли запрещена политикой системы в отношении групп"""


class TVMRolePolicyError(RoleRequestError):
    """Запрос роли на tvm-приложение в системе, которая не поддерживает такие роли"""


@python_2_unicode_compatible
class WorkflowSyntaxError(WorkflowError):
    """Ошибка, связанная с проблемами в синтаксисе."""

    def __init__(self, exception, line, lineno, local_variables, global_varibles):
        super(WorkflowSyntaxError, self).__init__(exception, line, lineno, local_variables)
        self.exception = exception
        self.line = line
        self.lineno = lineno
        self.locals = local_variables
        self.globals = global_varibles

    def __repr__(self):
        return str(self)

    def __str__(self):
        variables = self.globals.copy()
        variables.update(self.locals)
        variables.pop('__builtins__', None)
        variables = sorted([(key, value) for key, value in variables.items()
                            if not isinstance(value, (types.FunctionType, types.MethodType, type))],
                           key=operator.itemgetter(0))

        variables = '\n'.join(['    %(key)s = %(value)s' % {
            'key': force_text(key),
            'value': force_text(pprint.pformat(value))
        } for key, value in variables])
        result = 'Ошибка: %(exception)s\nСтрока %(lineno)d: %(line)s\nПеременные:\n%(variables)s' % {
            'exception': repr(self.exception),
            'lineno': self.lineno,
            'line': self.line,
            'variables': variables
        }
        return result

    def as_dict(self):
        try:
            from idm.core.workflow.sandbox.serializable import serialize
        except ImportError:
            from serializable import serialize

        try:
            serialized_exception = serialize(self.exception)
        except ValueError:
            serialized_exception = str(self.exception)

        _locals = self.locals.flatten() if hasattr(self.locals, 'flatten') else self.locals
        _globals = self.globals.flatten() if hasattr(self.globals, 'flatten') else self.globals

        serialized_locals = serialize(_locals, skip_functions=True)
        serialized_globals = serialize(_globals, skip_functions=True)

        return {
            'exception': serialized_exception,
            'line': serialize(self.line),
            'lineno': serialize(self.lineno),
            'locals': serialized_locals,
            'globals': serialized_globals,
        }

    @classmethod
    def from_dict(cls, data, context=None):
        try:
            from idm.core.workflow.sandbox.serializable import deserialize
        except ImportError:
            from serializable import deserialize
        return cls(
            exception=deserialize(data['exception']),
            line=deserialize(data['line']),
            lineno=deserialize(data['lineno']),
            local_variables=deserialize(data['locals']),
            global_varibles=deserialize(data['globals']),
        )


@python_2_unicode_compatible
class BossAndZamNotAvailableError(NoBossError):
    @property
    def message(self):
        if not self.zams:
            return _('Руководитель {boss}, подтверждающий роль, отсутствует.').format(boss=self.boss.username)
        if len(self.zams) == 1:
            return _('Руководитель {boss} и его заместитель {zam}, подтверждающие роль, '
                     'отсутствуют.').format(boss=self.boss.username, zam=self.zams[0].username)
        return _('Руководитель {boss} и его заместители {zams}, подтверждающие роль, '
                 'отсутствуют.').format(boss=self.boss.username, zams=', '.join(z.username for z in self.zams))

    def __init__(self, boss, zams=None):
        self.boss = boss
        self.zams = zams

    def __str__(self):
        return self.message


class NoSuchPriorityPolicyError(WorkflowError):
    """Такого priority policy не существует"""


class InvalidPriorityError(WorkflowError):
    """ Приоритет должен быть целым числом от 0 до 32767"""
