# coding: utf-8


import itertools

from django.db.models import Q
from django.utils import six

from idm.core.constants.workflow import PRIORITY_POLICIES, DEFAULT_PRIORITY
from idm.core.constants.groupmembership import GROUPMEMBERSHIP_STATE
from idm.core.workflow.exceptions import NoBossError, NoDepartmentError, BossNotFoundError, BossAndZamNotAvailableError
from idm.core.workflow.plain.mixins import SubjectWrapperMixin
from idm.framework.mixins import Representable
from idm.sync import gap
from idm.users import ranks
from idm.users import ranks as ranks_const
from idm.users.models import User, Group, GroupMembership, GroupResponsibility


def userify(login_or_user):
    """Возвращает Пользователя, но с дополнительными методами.

    Используется в workflow для различных проверок.
    """
    if login_or_user is None:
        result = login_or_user
    elif isinstance(login_or_user, UserWrapper):
        result = login_or_user
    elif isinstance(login_or_user, User):
        result = UserWrapper(login_or_user)
    else:
        result = UserWrapper(User.objects.select_related('department_group').get(username=login_or_user))
    return result


def try_userify(login_or_user):
    try:
        return userify(login_or_user)
    except User.DoesNotExist:
        return None


@six.python_2_unicode_compatible
class UserWrapper(Representable, SubjectWrapperMixin):
    """Класс-обертка для пользователя, добавляющая некоторые методы.
    """
    object_attribute_name = '_user'
    context = {}  # контекст исполнения, тут во время исполнения кода workflow должны быть ссылки на систему и т.д.
    userify_func = staticmethod(userify)

    @staticmethod
    def groupify_func(thing):
        from idm.core.workflow.sandbox.manager.group import groupify
        return groupify(thing)

    def __init__(self, user, context=None, rank=None, priority=DEFAULT_PRIORITY):
        self._user = user
        self.context = context or {}
        self.rank = rank
        self.priority = priority

    def __eq__(self, other):
        if isinstance(other, User):
            return self._user == other
        elif isinstance(other, UserWrapper):
            return self._user == other._user

        return NotImplemented

    def __ne__(self, other):
        return not self == other

    def __str__(self):
        return six.text_type(self._user)

    def __getattr__(self, name):
        return getattr(self._user, name)

    @property
    def department_group(self):
        from idm.core.workflow.plain.group import groupify
        return groupify(self._user.department_group)

    def is_boss_of(self, other_user):
        if isinstance(other_user, six.string_types):
            other_user = self.userify_func(other_user)
            other_user.fetch_department_group()
        try:
            return other_user.get_boss() == self._user
        except NoBossError:
            # у other_user нет вышестоящего начальника
            return False

    def is_head_of(self, *deps):
        """deps - список слагов департаментов или список id департаментов
        Возвращает True, только если начальник – непосредственный.
        """
        if not deps:
            return False

        query = Q()
        for dep_id in deps:
            if isinstance(dep_id, six.string_types):
                query |= Q(slug=dep_id)
            else:
                query |= Q(external_id=dep_id)
        qs = list(Group.objects.active().department().filter(query).prefetch_related('responsibilities'))
        result = any(self._user in group_.get_responsibles() for group_ in qs)
        return result

    def get_boss(self):
        """
        Вернуть ближайшего начальника по дереву отделов
        """
        boss, _ = self.get_boss_and_dep()
        return boss

    def get_boss_and_dep(self):
        """
        Вернуть ближайшего по дереву начальника и отдел, в котором он найден.
        Это важно, т.к. человек может быть начальников нескольких отделов, при этом ни в одном из них не работая,
        потому надо сразу говорить, где его нашли. Прошлый вариант выдавал ошибки для таких людей, см. RULES-937
        """
        dep = self.department_group.group
        if dep is not None:
            # Ищем ближайшего начальника вверх по иерархии
            while dep is not None:
                resps = dep.get_responsibles()
                if resps and self._user not in resps:
                    return self.userify_func(resps[0]), dep
                dep = dep.parent
        else:
            raise NoDepartmentError('Сотрудник {0.username} не состоит ни в одном департаменте.'.format(self))

        raise BossNotFoundError('Сотрудник {0.username} не имеет вышестоящего начальника.'.format(self))

    def get_boss_or_zam(self):
        """Возвращает ближайшего существующего начальника,
        если он отсутствует на работе - заместителя, иначе - raise Error"""

        boss, boss_dep = self.get_boss_and_dep()
        if not gap.is_user_absent(boss.username) or self.context.get('requester') == boss:
            return boss

        # проверяем отсутствия его замов.
        # Для поиска замов используется именно тот департамент, где нашли начальника.
        zams = get_department_zams(boss_dep.slug)
        for zam in zams:
            zam_user = self.userify_func(zam)
            if not gap.is_user_absent(zam) or self.context.get('requester') == zam_user:
                return zam_user
        raise BossAndZamNotAvailableError(boss=boss, zams=list(map(self.userify_func, zams)))

    def works_in_dep(self, *deps):
        """deps - список слагов департаментов или список id департаментов"""
        if not deps:
            return False

        if self.department_group_id is None:
            return False

        if isinstance(deps[0], six.string_types):
            result = self.departments_chain.filter(slug__in=deps).exists()
        else:
            result = self.departments_chain.filter(external_id__in=deps).exists()

        return result

    @property
    def departments_chain(self):
        return self._user.departments_chain

    def get_chain_of_heads(self, up_to_level=None, ranks=ranks_const.HEAD_OR_DEPUTY,
                           priority_policy=PRIORITY_POLICIES.HEAD_LESS):
        """
            Возвращает список руководителей и заместителей начиная с департамента на один уровень выше
            текущего департамента пользователя для руководетелей. Для всех остальных текущий департамент учитывается
        """
        # to avoid circular import
        from idm.core.workflow.plain.approver import AnyApprover

        self._user.fetch_department_group()
        # для руководителей поиск начинается с департамента на уровень выше
        if GroupResponsibility.objects.active().filter(
            user=self._user,
            rank=ranks_const.HEAD,
            group=self._user.department_group).exists():
            include_self = False
        else:
            include_self = True

        chain = self.department_group.get_responsibles(up_to_level=up_to_level, ranks=ranks, include_self=include_self,
                                                       priority_policy=priority_policy)

        # если текущий пользователь является руководителем или заместителем, то не возвращать самого себя
        flattened = [approver for approver in itertools.chain.from_iterable(chain) if self != approver]
        # если у сотрудника нет руководителей или замов, то вернуть себя
        if not flattened:
            self.priority = 1
            flattened = [self]

        return AnyApprover(flattened)

    @property
    def all_roles(self):
        """
        Вернуть список данных ролей пользователя.

        :rtype: [{}, ...]
        """
        system = self.context['system']._system
        codes = (
            self._user.roles.filter(system=system, is_active=True).
            values_list('node__data', flat=True).
            order_by('node__data').
            distinct()
        )
        return list(codes)

    @property
    def groups(self):
        groups = Group.objects.active().filter(
            memberships__user=self._user,
            memberships__state=GROUPMEMBERSHIP_STATE.ACTIVE
        )
        return [self.groupify_func(group) for group in groups]

    @property
    def is_robot(self):
        return self._user.is_robot

    @property
    def affiliation(self):
        return self._user.affiliation

    def get_robot_owners(self):
        return [self.userify_func(user) for user in self._user.responsibles.all()]

    def get_tvm_app_responsibles(self):
        responsibilities = GroupResponsibility.objects.active().filter(
            group_id__in=GroupMembership.objects.active().filter(user=self)
        )
        resposibles = User.objects.filter(id__in=responsibilities.values('user_id'))
        return [self.userify_func(user) for user in resposibles]

    def is_owner_of(self, robot):
        robot = self.userify_func(robot)
        return self._user.is_owner_of(robot._user)

    @property
    def all_heads(self):
        return [self.userify_func(x) for x in self._user.all_heads]

    @property
    def head(self):
        return self.userify_func(self._user.head)

    @property
    def internal_head(self):
        head = self._user.internal_head
        return self.userify_func(head)


def all_heads_of(dep_slug):
    """Возвращает список руководителей отделов: dep_slug и всех подотделов"""
    lst_heads = []

    def get_heads_of(group):
        resps = group.get_responsibles()
        if resps:
            lst_heads.extend(resps)
        for dep in group.children.active():
            get_heads_of(dep)

    dep = Group.objects.active().department().get(slug=dep_slug)
    get_heads_of(dep)

    return lst_heads


def get_head_of(dep_slug):
    """Возвращает руководителя подразделения dep_slug"""
    from idm.users.models import Group
    try:
        dep = Group.objects.active().get(slug=dep_slug)
    except Group.DoesNotExist:
        raise NoDepartmentError('Подразделение {0} не существует.'.format(dep_slug))

    # todo: check
    resps = dep.get_responsibles()
    if resps:
        return userify(resps[0])

    raise BossNotFoundError('Подразделение {0} не имеет руководителя.'.format(dep_slug))


def get_head_of_or_zam(dep_slug):
    """Возвращает руководителя, если он на работе, если нет - его заместитлеля, если нет - ошибку"""
    boss = get_head_of(dep_slug)
    if not gap.is_user_absent(boss.username):
        return boss

    zams = get_department_zams(dep_slug)
    for zam in zams:
        if not gap.is_user_absent(zam):
            return userify(zam)

    raise BossAndZamNotAvailableError(boss=boss, zams=list(map(userify, zams)))


def get_department_zams(dep_slug):
    """Возвращает всех заместителей подразделения dep_slug"""
    try:
        dep = Group.objects.active().get(slug=dep_slug)
    except:
        raise NoDepartmentError('Подразделение {0} не существует.'.format(dep_slug))

    return dep.get_responsibles(ranks=(ranks.DEPUTY,))
