# coding: utf-8

from __future__ import unicode_literals

from copy import copy

from django.template.context import BaseContext, ContextDict

try:
    from workflow_constants import PRIORITY_POLICIES, REQUEST_TYPE, RUN_REASON
    from workflow_exceptions import AccessDenied, Return
    from serializable import Serializable, serialize, deserialize, basestring_type
    from shortcuts import groupify, systemify, userify, find_conflicts, try_userify
    from wrappers import (
        Approver,
        AnyApprover,
        IDMFunction,
        Recipient,
    )
except ImportError:
    from idm.core.constants.workflow import PRIORITY_POLICIES, REQUEST_TYPE, RUN_REASON
    from idm.core.workflow.exceptions import AccessDenied, Return
    from idm.core.workflow.sandbox.container.shortcuts import groupify, systemify, userify, find_conflicts, try_userify
    from idm.core.workflow.sandbox.container.wrappers import (
        Approver,
        AnyApprover,
        IDMFunction,
        Recipient,
    )
    from idm.core.workflow.sandbox.serializable import Serializable, serialize, deserialize, basestring_type


# копипсата из idm.workflow.plain.context
class BaseWorkflowContext(BaseContext):
    def update(self, other_dict):
        "Pushes other_dict to the stack of dictionaries in the Context"
        if not hasattr(other_dict, '__getitem__'):
            raise TypeError('other_dict must be a mapping (dictionary-like) object.')
        return ContextDict(self, other_dict)

    def copy(self):
        """Метод для соответствия интерфейсу словаря"""
        return copy(self)

    def clear(self):
        self.pop()

    def items(self):
        flattened = self.flatten()
        return list(flattened.items())


class WorkflowContext(Serializable, BaseWorkflowContext):
    builtins = {
        'approver': Approver,
        'any_from': AnyApprover,
        'AccessDenied': AccessDenied,
        'Return': Return,
        'recipient': Recipient,
        'system': None,
        'REQUEST_TYPE': REQUEST_TYPE,
        'RUN_REASON': RUN_REASON,
        # -ify functions
        'systemify': systemify,
        'userify': userify,
        'try_userify': try_userify,
        'groupify': groupify,
        # other functions
        'find_conflicts': find_conflicts,
    }
    defaults = {
        'notify_everyone': False,
        'review_days': None,  # будет проставлен preprocess()
        'send_sms': False,
        'no_email': False,
    }
    default_lists = (
        'ad_groups',
        'email_cc',
        'ref_roles',
    )

    def __init__(self):
        builtins = {}
        defaults = {}
        for base in reversed(self.__class__.__mro__):
            if hasattr(base, 'builtins'):
                builtins.update(base.builtins)
            if hasattr(base, 'defaults'):
                defaults.update(defaults)
        builtins.update(self.builtins or {})
        defaults.update(self.defaults or {})

        for key in self.default_lists:
            defaults[key] = []

        self.builtins = builtins
        self.defaults = defaults
        super(WorkflowContext, self).__init__()

    def as_dict(self):
        d = {}
        for key, value in self.items():
            if not isinstance(key, basestring_type):
                raise ValueError('Serializable dict keys must be strings')
            d[key] = serialize(value)
        return d

    @classmethod
    def from_dict(cls, data, context=None):
        obj = cls()
        for key, value in data.items():
            if not isinstance(key, basestring_type):
                raise ValueError('Serializable dict keys must be strings')
            obj[key] = deserialize(value)
        return obj

    def preprocess(self, role, system, requester, node, **kwargs):
        self.update(self.builtins)
        self.update(self.defaults)
        self.update(kwargs)
        self['review_days'] = None
        self.push()
        self['role'] = role
        self['system'] = system
        self['requester'] = requester
        self['node'] = node


class UserWorkflowContext(WorkflowContext):
    builtins = {
        'PRIORITY_POLICIES': PRIORITY_POLICIES,
        'get_head_of_or_zam': IDMFunction(None, 'get_head_of_or_zam', is_callable=True),
        'all_heads_of': IDMFunction(None, 'all_heads_of', is_callable=True),
        'get_head_of': IDMFunction(None, 'get_head_of', is_callable=True),
    }

    def preprocess(self, role, system, requester, user, **kwargs):
        super(UserWorkflowContext, self).preprocess(role, system, requester, user=user, **kwargs)
        self['user'] = user


class GroupWorkflowContext(WorkflowContext):
    builtins = {
        'all_heads_of': IDMFunction(None, 'all_heads_of', is_callable=True),
    }

    def preprocess(self, role, system, requester, group, **kwargs):
        super(GroupWorkflowContext, self).preprocess(role, system, requester, group=group, **kwargs)
        self['group'] = group
