# coding: utf-8

from __future__ import unicode_literals

import doctest
import io
import socket
import sys

from django.utils.encoding import force_text

try:
    from connection import Connection
    from executor import ContainerWorkflowExecutor
    from serializable import deserialize, serialize
    from workflow_constants import WORKFLOW_CONTAINER_RESPONSE, WORKFLOW_IDM_RESPONSE
except ImportError:
    from idm.core.constants.workflow import WORKFLOW_CONTAINER_RESPONSE, WORKFLOW_IDM_RESPONSE
    from idm.core.workflow.sandbox.connection import Connection
    from idm.core.workflow.sandbox.container.executor import ContainerWorkflowExecutor
    from idm.core.workflow.sandbox.serializable import deserialize, serialize


class FakeRepresentation:
    def __init__(self, repr):
        self.repr = repr

    def __repr__(self):
        return self.repr


class Runner(object):
    def __init__(self, socker_addr):
        self.socker_addr = socker_addr
        self.connection = Connection(socket.socket(socket.AF_UNIX, socket.SOCK_STREAM))

    def wait_and_run_workflow(self):
        try:
            from context import GroupWorkflowContext, UserWorkflowContext
        except ImportError:
            from .context import GroupWorkflowContext, UserWorkflowContext

        operation, response = self.connection.get_data()
        if operation == WORKFLOW_IDM_RESPONSE.RUN_DOCTESTS:
            response_dict = deserialize(response)
            workflow_code = response_dict['code']
            system = response_dict['system']
            result = self.run_doctests(workflow_code, system)
            self.connection.send_data(WORKFLOW_CONTAINER_RESPONSE.FINISH, serialize(result))
            return

        response_dict = deserialize(response)
        context_dict = response_dict['context']
        if 'user' in context_dict:
            context = UserWorkflowContext()
        else:
            context = GroupWorkflowContext()

        context.preprocess(**context_dict.flatten())
        workflow_code = response_dict['code']

        ContainerWorkflowExecutor(workflow_code, context).run()

        context_dict = serialize(context.flatten(), skip_functions=True)
        self.connection.send_data(WORKFLOW_CONTAINER_RESPONSE.FINISH, context_dict)

    def run_doctests(self, workflow_code, system):
        try:
            from wrappers import UserWrapper, GroupWrapper
            from context import GroupWorkflowContext, UserWorkflowContext
            from shortcuts import userify, groupify
        except ImportError:
            from .wrappers import UserWrapper, GroupWrapper
            from .context import GroupWorkflowContext, UserWorkflowContext
            from .shortcuts import userify, groupify

        def run_doctest(requester, subject, role, expected_fields=None, **kwargs):
            node = system.get_node_by_data(role)
            context_dict = {'requester': requester, 'role': role, 'system': system, 'node': node}
            context_dict.update(kwargs)
            if isinstance(subject, UserWrapper):
                context = UserWorkflowContext()
                context_dict['user'] = subject
            elif isinstance(subject, GroupWrapper):
                context = GroupWorkflowContext()
                context_dict['group'] = subject
            else:
                return "Subject is not user or group"
            context_dict['__expected_fields'] = expected_fields
            context.preprocess(**context_dict)
            ContainerWorkflowExecutor(workflow_code, context).run()
            context_dict = serialize(context.flatten(), skip_functions=True)
            self.connection.send_data(WORKFLOW_CONTAINER_RESPONSE.POSTPROCESS_REQUEST, context_dict)
            _, result = self.connection.get_data()

            return FakeRepresentation(result)

        doctest_finder = doctest.DocTestFinder(recurse=False)
        doctest_runner = doctest.DocTestRunner(optionflags=doctest.ELLIPSIS)
        status = True
        messages = []
        doctest_context = {'run': run_doctest, 'user': userify, 'group': groupify}

        for test in doctest_finder.find(workflow_code, b'workflow.py', globs=doctest_context):
            with (io.StringIO() if sys.version_info[0] == 3 else io.BytesIO()) as out:
                test_result = doctest_runner.run(test, out=out.write)
                messages.append(force_text(out.getvalue()))
                if test_result.failed > 0:
                    status = False
        return {'status': status, 'messages': messages}

    def run(self):
        self.connection.connect(self.socker_addr)
        self.connection.send_data(0, {'status': 'container_start'})
        while True:
            self.wait_and_run_workflow()


runner = Runner('/socket.sock')
