# coding: utf-8

from __future__ import unicode_literals

import itertools
import six

from django.core.exceptions import ValidationError
from django.core.validators import validate_email
from django.utils.encoding import force_text
from django.utils.translation import ugettext as _

try:
    from serializable import Serializable, serialize, deserialize
    from role_constants import ROLE_STATE
    from runner import runner
    from workflow_constants import DEFAULT_PRIORITY, WORKFLOW_CONTAINER_RESPONSE, WORKFLOW_IDM_RESPONSE
    from workflow_exceptions import WorkflowForbiddenOperationError
except ImportError:
    from idm.core.constants.role import ROLE_STATE
    from idm.core.constants.workflow import DEFAULT_PRIORITY, WORKFLOW_CONTAINER_RESPONSE, WORKFLOW_IDM_RESPONSE
    from idm.core.workflow.exceptions import WorkflowForbiddenOperationError
    from idm.core.workflow.sandbox.container.runner import runner
    from idm.core.workflow.sandbox.serializable import Serializable, serialize, deserialize


if six.PY2:
    basestring_type = basestring
else:
    basestring_type = str


class InvalidPriorityError(Exception):
    pass


class RecipientValidationError(Exception):
    pass


class IDMFunction(object):
    def __init__(self, instance, function_name, is_callable):
        self.instance = instance
        self.is_callable = is_callable
        self.name = function_name

    def __call__(self, *args, **kwargs):
        data = serialize({
            'instance': self.instance,
            'name': self.name,
            'args': args,
            'kwargs': kwargs,
            'is_callable': self.is_callable,
        })

        runner.connection.send_data(WORKFLOW_CONTAINER_RESPONSE.REQUEST, data)
        response_code, response_data = runner.connection.get_data()
        response_obj = deserialize(response_data)

        if response_code == WORKFLOW_IDM_RESPONSE.RESPONSE_OK:
            return response_obj
        elif response_code == WORKFLOW_IDM_RESPONSE.RESPONSE_EXCEPTION:
            raise response_obj
        else:
            raise AssertionError('Wrong IDM response code %s' % response_code)


class BaseWrapper(Serializable):
    functions = []
    properties = []

    def __getattr__(self, item):
        if item in self.functions:
            return IDMFunction(self, item, is_callable=True)

        elif item in self.properties:
            result = IDMFunction(self, item, is_callable=False)()
            setattr(self, item, result)
            return result

        else:
            raise WorkflowForbiddenOperationError(
                'Method or attribute {} of {} is not available from within workflow'.format(
                    item, self.__class__.__name__))

    def set_initial_properties(self, initial_properties):
        if initial_properties:
            for attr, value in deserialize(initial_properties).items():
                setattr(self, attr, value)

    @classmethod
    def from_dict(cls, data, context=None):
        initial_properties = data.pop('initial_properties', None)
        obj = cls(**data)
        obj.set_initial_properties(initial_properties)
        return obj


class UserWrapper(BaseWrapper):
    functions = [
        'get_chain_of_heads', 'has_role', 'get_tvm_app_responsibles', 'get_full_name', 'get_short_name', 'get_boss',
        'is_head_of', 'is_boss_of', 'works_in_dep', 'get_boss_or_zam', 'get_robot_owners',
        'is_external',
    ]
    properties = [
        'department_group', 'first_name', 'last_name', 'email', 'actual_mobile_phone', 'internal_head',
        'is_homeworker', 'is_robot', 'lang_ui', 'head', 'all_heads', 'username', 'is_tvm_app', 'is_active', 'all_roles',
        'groups', 'departments_chain', 'date_joined',
    ]

    def __init__(self, username, rank=None, priority=DEFAULT_PRIORITY):
        self.rank = rank
        self.priority = priority
        if not isinstance(username, six.string_types):
            self.username = None
            raise ValidationError('Username must be a string.')
        self.username = username

    def __eq__(self, other):
        if other is None:
            return False

        assert isinstance(other, (UserWrapper, Approver) + six.string_types)
        if isinstance(other, Approver):
            return self.username == other.username
        elif isinstance(other, UserWrapper):
            return self.username == other.username
        elif isinstance(other, six.string_types):
            return self.username == other

    def __ne__(self, other):
        return not self == other

    def get_username(self):
        return self.username

    def works_in_dep(self, *deps):
        if not deps:
            return False

        existing_deps_slugs = {x.slug for x in self.departments_chain}
        for dep in deps:
            if isinstance(dep, six.string_types):
                if dep in existing_deps_slugs:
                    return True
            elif isinstance(dep, GroupWrapper):
                if dep.slug in existing_deps_slugs:
                    return True
            else:
                raise ValueError('Wrong group instance %s' % dep)

        return False

    def as_dict(self):
        return {
            'username': self.username,
            'rank': self.rank,
            'priority': self.priority,
        }


class AnyApprover(BaseWrapper):
    def __init__(self, approvers, notify=None, priority=DEFAULT_PRIORITY):
        self.approvers = []
        self.approver_dict = {}
        self.priority = priority
        self._add_approvers(approvers, notify)

    def __or__(self, other):
        # приоритет группы больше не нужен
        self.priority = DEFAULT_PRIORITY
        if isinstance(other, Approver):
            self._add_approvers(other, notify=other.notify)
            return self
        elif isinstance(other, AnyApprover):
            # проставить новый приоритет сотрудникам без приоритета
            self._add_approvers(other.approvers)
            return self
        elif isinstance(other, (str, UserWrapper)):
            self._add_approvers([other])
            return self

        return NotImplemented

    def _flatten_approverify(self, approvers):
        for approver in approvers:
            if isinstance(approver, Approver):
                yield approver
            elif isinstance(approver, AnyApprover):
                for other_approver in approver.approvers:
                    yield other_approver
            elif isinstance(approver, dict):
                if approver['type'] == 'approver':
                    yield Approver.from_dict(approver)
                elif approver['type'] == 'any_approver':
                    for approver in AnyApprover.from_dict(approver).approvers:
                        yield approver
                else:
                    raise AssertionError('Wrong approver dict %s' % approver)

            else:
                yield Approver(approver)

    def _combine_notifications(self, *notifications):
        result = None
        for notify in notifications:
            if notify is True:
                return True
            elif notify is False:
                result = False
        return result

    def _add_approvers(self, approvers, notify=None):
        for approver in self._flatten_approverify(approvers):

            # если у approverа не выставлен приоритет
            if self.priority != DEFAULT_PRIORITY and approver.priority == DEFAULT_PRIORITY:
                approver.priority = self.priority

            if approver.username in self.approver_dict:
                existing_approver = self.approver_dict[approver.username]
                existing_approver.notify = self._combine_notifications(existing_approver.notify, approver.notify)
                # выбирать минимальный приоритет у повторяющихся сотрудников
                existing_approver.priority = min(existing_approver.priority, approver.priority)
            else:
                if notify is not None:
                    approver.notify = notify
                self.approvers.append(approver)
                self.approver_dict[approver.username] = approver

    def __eq__(self, other):
        if hasattr(other, '__iter__'):
            return all(our_approver == their_approver for our_approver, their_approver in itertools.zip_longest(self, other))

        return NotImplemented

    def __ne__(self, other):
        return not self == other

    def __iter__(self):
        """Возвращает Iterable по всем login."""
        for approver in self.approvers:
            yield approver

    def __len__(self):
        return len(self.approvers)

    @classmethod
    def from_dict(cls, data, context=None):
        obj = cls(approvers=deserialize(data['approvers']), priority=data.get('priority', DEFAULT_PRIORITY))
        obj.set_initial_properties(data.get('initial_properties'))
        return obj

    def as_dict(self):
        return {
            'approvers': serialize(self.approvers),
            'priority': self.priority
        }

    @property
    def user(self):
        if len(self) == 1:
            return self.approvers[0].user
        else:
            raise AttributeError('"AnyApprover" object has no attribute "user"')

    @property
    def priority(self):
        return self._priority

    @priority.setter
    def priority(self, priority):
        if not isinstance(priority, int) or priority < 0 or priority > DEFAULT_PRIORITY:
            raise InvalidPriorityError(_("Приоритет должен быть целым числом от 0 до {}".format(DEFAULT_PRIORITY)))

        self._priority = priority


class Approver(BaseWrapper):
    properties = ['user']

    def __init__(self, username, notify=None, priority=DEFAULT_PRIORITY):
        # берем приоритет из UserWrapper если пользователь не задал приоритет извне
        if isinstance(username, UserWrapper):
            self.username = username.username

            if priority == DEFAULT_PRIORITY:
                self.priority = username.priority
            else:
                self.priority = priority

        else:
            user = UserWrapper(username)
            self.priority = priority
            self.username = user.username

        self.notify = notify

    def __hash__(self):
        return hash(self.username)

    def __eq__(self, other):
        assert isinstance(other, (Approver, UserWrapper))

        if isinstance(other, Approver):
            return self.username == other.username
        elif isinstance(other, UserWrapper):
            return self.username == other.username

    def __ne__(self, other):
        return not self == other

    def __or__(self, other):
        return AnyApprover([self, other])

    def __iter__(self):
        """Возвращает Iterable по пользователям."""
        yield self

    def __len__(self):
        return 1

    def as_dict(self):
        return {
            'username': self.username,
            'notify': self.notify,
            'priority': self.priority,
        }

    @classmethod
    def from_dict(cls, data, context=None):
        obj = cls(username=data['username'], notify=data['notify'], priority=data['priority'])
        obj.set_initial_properties(data.get('initial_properties'))
        return obj

    @property
    def approvers(self):
        return [self]

    @property
    def priority(self):
        return self._priority

    @priority.setter
    def priority(self, priority):
        if not isinstance(priority, int) or priority < 0 or priority > DEFAULT_PRIORITY:
            raise InvalidPriorityError(_("Приоритет должен быть целым числом от 0 до {}".format(DEFAULT_PRIORITY)))

        self._priority = priority


class AllList(Serializable, list):
    def all(self):
        return self

    @classmethod
    def from_dict(cls, data, context=None):
        return cls(deserialize(data['data']))

    def as_dict(self):
        return {
            'data': serialize(list(self))
        }


class ChainLevel(Serializable, list):
    def __init__(self, level, items):
        self.level = level
        super(ChainLevel, self).__init__(items)

    @classmethod
    def from_dict(cls, data, context=None):
        level = data['level']
        items = deserialize(data['items'], context=context)
        return cls(level, items)

    def as_dict(self):
        return {
            'level': self.level,
            'items': serialize([x for x in self])
        }


class GroupWrapper(BaseWrapper):
    functions = [
        'has_role', 'get_parent', 'is_root', 'get_root', 'get_ancestor', 'get_ancestors', 'is_descendant_of',
        'get_children', 'get_chain_of_heads', 'get_all_responsibles', 'get_responsibles', 'get_heads', 'get_deputies',
        'is_subgroup_of', 'get_name', 'get_descendant_members',
    ]
    properties = ['staff_id', 'parent', 'name', 'name_en', 'description', 'type', 'members', 'slug', 'head', 'level']

    def __init__(self, group_id):
        self.group_id = group_id

    def __eq__(self, other):
        assert isinstance(other, GroupWrapper)
        return self.group_id == other.group_id

    def __ne__(self, other):
        return not self == other

    def __hash__(self):
        return hash(self.group_id)

    def as_dict(self):
        return {
            'group_id': self.group_id,
        }


class NodeWrapper(BaseWrapper):
    functions = [
        'get_parent', 'is_root', 'get_root', 'get_ancestor', 'get_ancestors', 'is_descendant_of', 'get_children',
        'get_aliases', 'get_responsibilities', 'get_owners',
    ]
    properties = [
        'parent', 'name', 'name_en', 'data', 'fullname', 'description', 'description_en', 'slug', 'slug_path',
        'value_path', 'parent_path',
    ]

    def __init__(self, id):
        self.id = id

    def __eq__(self, other):
        assert isinstance(other, NodeWrapper)
        return self.group_id == other.group_id

    def __ne__(self, other):
        return not self == other

    def as_dict(self):
        return {
            'id': self.id
        }


class SystemWrapper(BaseWrapper):
    functions = ['get_name', 'get_state', 'all_users_with_role', 'all_groups_with_role', 'get_aliased_nodes',
                 'get_node_by_data']
    properties = ['name', 'name_en', 'is_active', 'is_broken', 'slug', 'creator']

    def __init__(self, slug):
        self.slug = slug

    def __eq__(self, other):
        assert isinstance(other, SystemWrapper)
        return self.system_slug == other.system_slug

    def __ne__(self, other):
        return not self == other

    def as_dict(self):
        return {
            'slug': self.slug
        }


class Recipient(BaseWrapper):
    def __init__(self, email, lang='ru', pass_to_personal=False, **kwargs):
        if isinstance(email, basestring_type):
            email = force_text(email)
        elif hasattr(email, 'email'):
            email = force_text(email.email)
        else:
            raise RecipientValidationError('Cannot recipientify %s' % repr(email))
        try:
            validate_email(email)
        except ValidationError:
            raise RecipientValidationError('Wrong email: %s' % email)
        self.email = email

        if lang not in ('ru', 'en'):
            raise RecipientValidationError('Wrong language key for recipientify: %s' % lang)
        self.language = lang

        self.pass_to_personal = pass_to_personal

        if 'states' in kwargs:
            self.validate_states(kwargs['states'])
            self.states = kwargs['states']
        else:
            self.validate_states(list(kwargs.keys()))
            self.states = {key for key, value in kwargs.items() if value}

    def __str__(self):
        return 'email=%s, lang=%s, pass_to_personal=%s states=%s' % (
            self.email,
            self.language,
            self.pass_to_personal,
            ', '.join(self.states)
        )

    def validate_states(self, keys):
        for key in keys:
            if key not in ROLE_STATE.ALL_STATES and key != 'reminders':
                raise RecipientValidationError('Wrong key for recipientify: %s' % key)

    def as_dict(self):
        return {
            'email': self.email,
            'language': self.language,
            'pass_to_personal': self.pass_to_personal,
            'states': serialize(self.states),
        }


class AliasWrapper(BaseWrapper):
    functions = []
    properties = [
        'node', 'type', 'name', 'name_en', 'is_active',
    ]

    def __init__(self, id):
        self.id = id

    def as_dict(self):
        return {
            'id': self.id
        }


class ResponsibilityWrapper(BaseWrapper):
    functions = []
    properties = [
        'node', 'is_active', 'user', 'notify',
    ]

    def __init__(self, id):
        self.id = id

    def as_dict(self):
        return {
            'id': self.id
        }


class RoleWrapper(BaseWrapper):
    functions = []
    properties = ['system', 'node', 'user', 'group']

    def __init__(self, id):
        self.id = id

    def as_dict(self):
        return {
            'id': self.id
        }


class ConflictWrapper(BaseWrapper):
    functions = []
    properties = ['requested_path', 'conflicting_path', 'conflicting_system', 'email', 'subj', 'node']

    def as_dict(self):
        return {
            'requested_path': self.requested_path,
            'conflicting_path': self.conflicting_path,
            'email': self.email,
            'conflicting_system': serialize(self.conflicting_system),
            'node': serialize(self.node),
            'subj': serialize(self.subj),
        }

    @classmethod
    def from_dict(cls, data, context=None):
        obj = cls()
        obj.subj = deserialize(data['subj'])
        obj.node = deserialize(data['node'])
        obj.conflicting_system = deserialize(data['conflicting_system'])
        for k in ['requested_path', 'conflicting_path', 'email']:
            setattr(obj, k, data[k])
        return obj
