# coding: utf-8


import logging

from idm.core.constants.workflow import PRIORITY_POLICIES, REQUEST_TYPE, RUN_REASON
from idm.core.workflow.exceptions import AccessDenied, Return
from idm.core.workflow.plain.context import (WorkflowContext as PlainWorkflowContext,
                                             UserContextMixin as PlainUserContextMixin,
                                             GroupContextMixin as PlainGroupContextMixin)
from idm.core.workflow.sandbox.manager.approver import AnyApprover, Approver
from idm.core.workflow.sandbox.manager.conflcit import ConflictWrapper
from idm.core.workflow.sandbox.manager.group import groupify
from idm.core.workflow.sandbox.manager.node import NodeWrapper
from idm.core.workflow.sandbox.manager.role import RoleWrapper
from idm.core.workflow.common.recipient import Recipient
from idm.core.workflow.sandbox.serializable import Serializable, serialize, deserialize, basestring_type
from idm.core.workflow.sandbox.manager.system import SystemWrapper, systemify
from idm.core.workflow.sandbox.manager.user import userify, all_heads_of, get_head_of, get_head_of_or_zam, try_userify


log = logging.getLogger(__name__)


class WorkflowContext(Serializable, PlainWorkflowContext):
    class WRAPPERS:
        CONFLICT = ConflictWrapper

    builtins = {
        # classes
        'approver': Approver,
        'any_from': AnyApprover,
        'AccessDenied': AccessDenied,
        'Return': Return,
        'recipient': Recipient,
        'REQUEST_TYPE': REQUEST_TYPE,
        'RUN_REASON': RUN_REASON,
        # functions
        'systemify': systemify,
        'userify': userify,
        'try_userify': try_userify,
        'groupify': groupify,
        'all_heads_of': all_heads_of,
        # other
        'find_conflicts': ConflictWrapper.find_conflicts_wrapper,
    }

    def as_dict(self):
        d = {}
        for key, value in self.items():
            if not isinstance(key, basestring_type):
                raise ValueError('Serializable dict keys must be strings')
            d[key] = serialize(value)
        return d

    @classmethod
    def from_dict(cls, data, context=None):
        obj = cls()
        for key, value in data.items():
            if not isinstance(key, basestring_type):
                raise ValueError('Serializable dict keys must be strings')
            obj[key] = deserialize(value)
        return obj

    def preprocess(self, role_data, system, requester, node, parent=None, **kwargs):
        self.update(self.defaults)
        self.update(kwargs)
        self['parent'] = RoleWrapper(parent) if parent else None
        self['review_days'] = None
        self['role'] = role_data
        self['system'] = SystemWrapper(system)
        self['requester'] = userify(requester)
        self['original_requester'] = userify(kwargs.get('original_requester'))
        self['node'] = NodeWrapper(node)
        if self['requester']:
            self['requester'].context = self


class UserWorkflowContext(PlainUserContextMixin, WorkflowContext):
    builtins = {
        'PRIORITY_POLICIES': PRIORITY_POLICIES,
        'get_head_of_or_zam': get_head_of_or_zam,
        'get_head_of': get_head_of,
    }

    def preprocess(self, role_data, system, requester, node, user, **kwargs):
        WorkflowContext.preprocess(self, role_data, system, requester, node, **kwargs)
        self['user'] = userify(user)
        self['user'].context = self


class GroupWorkflowContext(PlainGroupContextMixin, WorkflowContext):
    def preprocess(self, role_data, system, requester, node, group, **kwargs):
        WorkflowContext.preprocess(self, role_data, system, requester, node, **kwargs)
        self['group'] = groupify(group)
        self['group'].context = self
        self['inheritance_settings'] = kwargs.get('inheritance_settings', {})
