# coding: utf-8

import collections
import contextlib
import logging
import os
import shutil
import socket
import sys
import uuid
from pathlib import Path
from typing import Tuple, ContextManager

import porto
from django.conf import settings
from django.utils.encoding import force_bytes, force_text
from django.utils.translation import ugettext_lazy as _
from django_tools_log_context.profiler import ExecutionProfiler
import time

from idm.core.constants.system import SYSTEM_WORKFLOW_INTERPRETER
from idm.core.constants.workflow import (
    WORKFLOW_CONTAINER_START_STATUS,
    WORKFLOW_CONTAINER_RESPONSE,
    WORKFLOW_IDM_RESPONSE,
)
from idm.core.workflow.exceptions import (
    WorkflowError, WorkflowContainerInitError, WorkflowContainerTimeoutError,
    WorkflowContainerProtocolError, WorkflowForbiddenOperationError,
)
from idm.core.workflow.sandbox.connection import Connection
from idm.core.workflow.sandbox.manager.context import WorkflowContext, UserWorkflowContext, GroupWorkflowContext
from idm.core.workflow.sandbox.manager.sanboxed import SandboxedMixin
from idm.core.workflow.sandbox.manager.system import SystemWrapper
from idm.core.workflow.sandbox.serializable import serialize, deserialize
from idm.utils.log import log_subcontext
from idm.utils.math import quantile

# local (хост-контейнер)
LOCAL_ROOT = Path('/')
PORTO_ROOT = LOCAL_ROOT
CONTAINER_ROOT = Path('/')

CONTAINERS_DIRECTORY = Path('idm_container')
PYTHON_PORTO_LAYER_PATH = CONTAINERS_DIRECTORY / 'python23'
DIRS_TO_LINK = ['bin', 'coredumps', 'etc', 'home', 'lib', 'lib64', 'place', 'root', 'sbin', 'srv', 'usr', 'var']
CONTAINER_USER = 'container'
CONTAINER_CODE_DIR = 'idm'
ENTRYPOINT_NAME = 'main.py'
PROJECT_ROOT = Path(settings.MODULE_ROOT)
RUNNER_CODE_PATH = PROJECT_ROOT / 'core' / 'workflow' / 'sandbox' / 'container'
MODULES_TO_COPY = [
    (PROJECT_ROOT / 'core' / 'constants' / 'role.py', 'role_constants.py'),
    (PROJECT_ROOT / 'core' / 'constants' / 'workflow.py', 'workflow_constants.py'),
    (PROJECT_ROOT / 'core' / 'workflow' / 'exceptions.py', 'workflow_exceptions.py'),
    (PROJECT_ROOT / 'core' / 'workflow' / 'sandbox' / 'connection.py', 'connection.py'),
    (PROJECT_ROOT / 'core' / 'workflow' / 'sandbox' / 'serializable.py', 'serializable.py'),
]

log = logging.getLogger(__name__)
_porto_connection = None

# monkeypatch бага в venv/lib/python3.8/site-packages/porto/exceptions.py:22
setattr(porto.exceptions.PortoException, '__str__', Exception.__str__)


def get_porto_connection():
    global _porto_connection
    if not _porto_connection:
        _porto_connection = porto.Connection()
        _porto_connection.connect()
    return _porto_connection


class WorkflowExecutionProfiler(ExecutionProfiler):
    def __init__(self, executor, *args, **kwargs):
        super(WorkflowExecutionProfiler, self).__init__(*args, **kwargs)
        self.executor = executor

    def __exit__(self, exc_type, exc_val, exc_tb):
        with log_subcontext(
                'workflow',
                container_creation_time=self.executor.container_creation_time,
                container_destruction_time=self.executor.container_destruction_time,
                container_function_calls=self.executor.postprocess_profiler_entries(),
                # агрегируем все вызовы по функциям
        ):
            args = (None, None, None)
            if exc_type and not issubclass(exc_type, WorkflowError):
                args = (exc_type, exc_val, exc_tb)
            return super(WorkflowExecutionProfiler, self).__exit__(*args)


class SandboxedWorkflowExecutor:
    context_class = WorkflowContext

    def __init__(self, code: str, system):
        self.code = force_text(code)
        if 'coding:' not in self.code:
            self.code = '# coding: utf-8\n' + self.code
        self.code = self.code.replace('\r', '').strip()
        self.system = system
        self.context = self.context_class()

        self.container_creation_time = None
        self.container_destruction_time = None
        self.container_function_calls = collections.defaultdict(self._init_profiler_entry)
        self.cached_objects = collections.defaultdict(dict)

    @staticmethod
    def _init_profiler_entry():
        return {
            '_calls': [],
        }

    def postprocess_profiler_entries(self):
        result = []
        for funcname in sorted(self.container_function_calls):
            entry = self.container_function_calls[funcname]
            calls = sorted(entry['_calls'])
            result.append({
                'function': funcname,
                'calls': len(calls),
                'total': sum(calls),
                'mean': sum(calls) / len(calls),
                'perc90': quantile(calls, 0.90, already_sorted=True),
                'perc100': calls[-1],
            })
        return result

    @staticmethod
    def _get_user_and_group_by_name(name: str) -> Tuple[int, int]:
        passwd_file = LOCAL_ROOT / PYTHON_PORTO_LAYER_PATH / 'etc' / 'passwd'
        if not passwd_file.exists():
            raise WorkflowContainerInitError(_('Файл /etc/passwd контейнера не существует'))
        with passwd_file.open() as f:
            for line in f.readlines():
                username, password_hash, uid, gid, rest = line.split(':', 4)
                if force_bytes(username) == force_bytes(name):
                    return uid, gid
        raise WorkflowContainerInitError(_('Пользователь контейнера не найден в /etc/passwd'))

    @staticmethod
    def _create_socket(main_socket_name: str) -> socket.socket:
        connection = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
        # при таймауте в дальнейшем будем считать, что контейнер умер
        connection.settimeout(settings.IDM_WORKFLOW_SOCKET_TIMEOUT)
        connection.bind(main_socket_name)
        connection.listen(1)
        return connection

    @staticmethod
    def _create_connection(socket_connection: socket.socket) -> Connection:
        connection, addr = socket_connection.accept()
        connection.settimeout(settings.IDM_WORKFLOW_SOCKET_TIMEOUT)
        return Connection(connection)

    def _init_container(self, container_name: str, container_root: Path) -> porto.api.Container:
        try:
            container = get_porto_connection().Create(f'self/{container_name}')  # porto fix: RTCSUPPORT-14508
        except porto.exceptions.PortoException:
            log.exception(f'Error occurred during container creation')
            raise WorkflowContainerInitError(_('Не получилось инициализировать контейнер с workflow'))

        # Настраиваем кастомный чрут для дочернего контейнера
        if container_root.exists():
            shutil.rmtree(container_root)
        container_root.mkdir()
        container.SetProperty('root', str(PORTO_ROOT / CONTAINERS_DIRECTORY / container_name))
        container.SetProperty('cwd', '/')

        # Всё, что можно, readonly-биндим в контейнер (симлинк по сути), так как копировать долго
        bounds = [
            f'{PORTO_ROOT / PYTHON_PORTO_LAYER_PATH / directory.name} {CONTAINER_ROOT / directory.name} ro'
            for directory in (LOCAL_ROOT / PYTHON_PORTO_LAYER_PATH).iterdir()
            if directory.name in DIRS_TO_LINK
        ]
        container.SetProperty('bind', '; '.join(bounds))

        # Запускаем под отдельным юзером
        uid, gid = self._get_user_and_group_by_name(CONTAINER_USER)
        container.SetProperty('user', uid)
        container.SetProperty('group', gid)

        # Определяем и прокидываем Python-код
        if self.system.workflow_python_version == SYSTEM_WORKFLOW_INTERPRETER.PYTHON_2_7:
            interpreter = 'python2'
        elif self.system.workflow_python_version == SYSTEM_WORKFLOW_INTERPRETER.PYTHON_3_6:
            interpreter = 'python3'
        else:
            raise ValueError('Invalid interpreter version: {}'.format(self.system.workflow_python_version))
        container_code_path = container_root / CONTAINER_CODE_DIR
        shutil.copytree(RUNNER_CODE_PATH, container_code_path)
        for module, destination in MODULES_TO_COPY:
            shutil.copy(module, container_code_path / destination)
        container.SetProperty(
            'command',
            '{} {}'.format(interpreter, CONTAINER_ROOT / CONTAINER_CODE_DIR / ENTRYPOINT_NAME)
        )

        # Всякие настройки контейнера, улучшающие изоляцию
        container.SetProperty('memory_limit', settings.IDM_WORKFLOW_MEMORY_LIMIT)  # ограничиваем по памяти
        container.SetProperty('cpu_limit', settings.IDM_WORKFLOW_CPU_LIMIT)  # ограничиваем по потреблению cpu
        container.SetProperty('root_readonly', 'false')  # readonly не получится (нужно как минимум писать в сокет)
        container.SetProperty('enable_porto', 'false')  # отключаем доступ к porto api у дочернего контейнера
        container.SetProperty('isolate', 'true')  # изолируем контейнер от отца
        container.SetProperty('net', 'none')  # отключаем сеть
        container.SetProperty('weak', 'true')  # киляем контейнер при смерти запускалки
        container.SetProperty('stdin_path', '/dev/null')  # не пробрасываем stdin ниже
        # Редиректим stdout и stderr в файлики
        # Несекьюрно: атакующий может забить диск и теоретически свалить idm
        for descriptor in ('stdout', 'stderr'):
            (container_root / descriptor).touch()
            (container_root / descriptor).chmod(0o666)

        container.SetProperty('stdout_path', str(CONTAINER_ROOT / 'stdout'))
        container.SetProperty('stderr_path', str(CONTAINER_ROOT / 'stderr'))
        return container

    def _start_container(self, container: porto.api.Container, socket_path: Path) -> Connection:
        # Создаём сокет, по которому будем общаться с выполнялкой в контейнере
        container_socket = self._create_socket(str(socket_path))
        socket_path.chmod(0o666)

        try:
            container.Start()
            connection = self._create_connection(container_socket)
            status_code, response = connection.get_data()
        except porto.exceptions.PortoException:
            log.exception(f'Error occurred during container creation')
            raise WorkflowContainerInitError(_('Не получилось инициализировать контейнер с workflow'))
        except socket.timeout:
            raise WorkflowContainerInitError(_('Не получилось инициализировать контейнер с workflow'))

        if response['status'] != WORKFLOW_CONTAINER_START_STATUS.CONTAINER_START_SUCCESS:
            raise WorkflowContainerInitError(
                _('Некорректный статус инициализации контейнера: ') + force_text(response['status'])
            )
        return connection

    def _get_unique_identifier(self) -> str:
        return '{}_{}_{}_{}'.format(self.system.slug, os.getpid(), int(time.time() * 1000), str(uuid.uuid4())[:18])

    @contextlib.contextmanager
    def create_container(self) -> ContextManager[Connection]:
        container = None
        log_ctx = None

        containers_path = LOCAL_ROOT / CONTAINERS_DIRECTORY
        if not containers_path.exists():
            raise WorkflowContainerInitError(_('Директория контейнеров не существует'))

        container_name = self._get_unique_identifier()
        container_root = containers_path / container_name
        start_time = time.monotonic()
        try:
            container = self._init_container(container_name, container_root)
            log_ctx = log_subcontext('workflow', container_name=container_name)
            log_ctx.__enter__()
            connection = self._start_container(container, container_root / 'socket.sock')
            self.container_creation_time = (time.monotonic() - start_time) * 1000  # в миллисекундах
            yield connection
        except Exception as err:
            error_message = ','.join([str(x) for x in err.args])
            log.exception('Could not init container for system=%s: error="%s"', self.system.slug, error_message)
            raise
        finally:
            if container:
                start_time = time.monotonic()
                container.Destroy()
            if container_root.exists():
                shutil.rmtree(container_root)
            if log_ctx:
                log_ctx.__exit__(*sys.exc_info())
            self.container_destruction_time = (time.monotonic() - start_time) * 1000  # в миллисекундах

    def _run(self, **kwargs):
        # Тут мы поднимаем контейнер, прокидываем туда всё, общаемся с ним, а потом собираем результат и прибиваем
        serialized_result = None
        start_time = time.time()
        exception_raised = False
        with self.create_container() as container:
            input_data = serialize({
                'code': self.code,
                'context': self.context,
            })
            container.send_data(WORKFLOW_IDM_RESPONSE.RUN_WORKFLOW, input_data)
            while True:
                try:
                    response_code, response = container.get_data()
                except socket.timeout:
                    raise WorkflowContainerTimeoutError(_('Ответ от контейнера workflow не был получен в срок'))

                accepted_codes = [
                    WORKFLOW_CONTAINER_RESPONSE.FINISH,
                    WORKFLOW_CONTAINER_RESPONSE.REQUEST,
                    WORKFLOW_CONTAINER_RESPONSE.ERROR,
                ]
                if response_code not in accepted_codes:
                    raise WorkflowContainerProtocolError(
                        _('Некорректный код ответа от контейнера workflow: ') + force_text(response_code)
                    )

                if response_code in (WORKFLOW_CONTAINER_RESPONSE.FINISH, WORKFLOW_CONTAINER_RESPONSE.ERROR):
                    if response_code == WORKFLOW_CONTAINER_RESPONSE.ERROR:
                        exception_raised = True
                    serialized_result = response
                    break

                if time.time() - start_time >= settings.IDM_WORKFLOW_TIMEOUT:
                    raise WorkflowContainerTimeoutError(_('Выполнение workflow не было завершено в срок'))

                if response_code == WORKFLOW_CONTAINER_RESPONSE.REQUEST:
                    self.process_request(response, container)

        assert serialized_result is not None
        result = deserialize(serialized_result, cached_objects=self.cached_objects)
        self.cached_objects.clear()
        if exception_raised:
            raise result
        return result

    def process_request(self, request, container):
        process_start = time.time()
        if not isinstance(request, dict):
            raise WorkflowContainerProtocolError(
                _('Некорректный формат запроса от контейнера workflow (некорректный тип)')
            )

        if sorted(request.keys()) != ['type', 'value']:
            raise WorkflowContainerProtocolError(
                _('Некорректный формат запроса от контейнера workflow (некорректная структура json)')
            )
        if sorted(request['value']) != ['args', 'instance', 'is_callable', 'kwargs', 'name']:
            raise WorkflowContainerProtocolError(
                _('Некорректный формат запроса от контейнера workflow (некорректный список ключей)')
            )
        instance = deserialize(request['value']['instance'], self.context, cached_objects=self.cached_objects) if \
            request['value']['instance'] else None
        args = deserialize(request['value']['args'], self.context, cached_objects=self.cached_objects)
        kwargs = deserialize(request['value']['kwargs'], self.context, cached_objects=self.cached_objects)
        name = request['value']['name']
        is_callable = request['value']['is_callable']
        exception_raised = False
        if instance:
            assert isinstance(instance, SandboxedMixin)
            try:
                result = instance._call_from_container(name, args, kwargs, is_callable)
            except WorkflowError as err:
                result = err
                exception_raised = True
        else:
            try:
                try:
                    attribute = self.context.builtins[name]
                except KeyError:
                    raise WorkflowForbiddenOperationError(
                        'Function or attribute is not available from within workflow'
                    )
                if is_callable and callable(attribute):
                    result = attribute(*args, **kwargs)
                else:
                    result = attribute
            except WorkflowError as err:
                result = err
                exception_raised = True

        serialized_response = serialize(result)
        if exception_raised:
            response_code = WORKFLOW_IDM_RESPONSE.RESPONSE_EXCEPTION
        else:
            response_code = WORKFLOW_IDM_RESPONSE.RESPONSE_OK

        process_finish = time.time()
        function_logged_name = '{}{}{}{}'.format(
            instance.__class__.__name__ if instance else '',
            '.' if instance else '',
            name,
            '()' if is_callable else '',
        )
        self.container_function_calls[function_logged_name]['_calls'].append(process_finish - process_start)

        container.send_data(response_code, serialized_response)

    def run(self, **kwargs):
        reason = kwargs.get('reason')
        system = kwargs['system']
        system_slug = system.slug if system else None

        self.context.preprocess(**kwargs)

        with log_subcontext('workflow', system=system_slug, reason=reason, log_name='sandboxed_workflow'):
            with WorkflowExecutionProfiler(
                    self,
                    code_block_name='workflow; sandboxed; system %s; reason %s' % (system_slug, reason),
                    code_block_type='sandboxed_workflow',
                    threshold=0
            ):
                result = self._run(**kwargs)

        self.context.update(result)
        self.context.postprocess(**kwargs)
        return self.context

    def run_catch_exceptions(self, **kwargs):
        raise NotImplementedError()  # Доктесты прогоняются прямо в песочнице

    def check(self, **kwargs):
        raise NotImplementedError()  # Доктесты прогоняются прямо в песочнице

    def run_doctests(self, **kwargs):
        system = self.system
        self.context['system'] = SystemWrapper(system)

        serialized_result = None
        start_time = time.time()
        with self.create_container() as container:
            input_data = serialize({
                'code': self.code,
                'system': SystemWrapper(system, self.context),
            })
            container.send_data(WORKFLOW_IDM_RESPONSE.RUN_DOCTESTS, input_data)
            while True:
                try:
                    response_code, response = container.get_data()
                except socket.timeout:
                    raise WorkflowContainerTimeoutError(_('Ответ от контейнера workflow не был получен в срок'))

                accepted_codes = [WORKFLOW_CONTAINER_RESPONSE.FINISH,
                                  WORKFLOW_CONTAINER_RESPONSE.REQUEST,
                                  WORKFLOW_CONTAINER_RESPONSE.POSTPROCESS_REQUEST]
                if response_code not in accepted_codes:
                    raise WorkflowContainerProtocolError(
                        _('Некорректный код ответа от контейнера workflow: ') + force_text(response_code)
                    )

                if response_code == WORKFLOW_CONTAINER_RESPONSE.FINISH:
                    serialized_result = response
                    break

                if time.time() - start_time >= settings.IDM_WORKFLOW_TIMEOUT:
                    raise WorkflowContainerTimeoutError(_('Выполнение workflow не было завершено в срок'))

                if response_code == WORKFLOW_CONTAINER_RESPONSE.REQUEST:
                    self.process_request(response, container)

                if response_code == WORKFLOW_CONTAINER_RESPONSE.POSTPROCESS_REQUEST:
                    assert response is not None
                    result = deserialize(response)
                    self.context = self.context_class()
                    self.context.update(result)
                    self.context.postprocess(self.context['node'], system, self.context['requester'])
                    expected_fields = self.context['__expected_fields']
                    if not expected_fields:
                        response = repr(self.context.get('approvers'))
                    else:
                        response = repr([self.context.get(field) for field in expected_fields])
                    response_code = WORKFLOW_IDM_RESPONSE.RESPONSE_OK
                    container.send_data(response_code, response)

        assert serialized_result is not None
        result = deserialize(serialized_result)
        status = result['status']
        messages = result['messages']
        return status, b'\n'.join([force_bytes(x) for x in messages])


class SandboxedUserWorkflowExecutor(SandboxedWorkflowExecutor):
    context_class = UserWorkflowContext


class SandboxedGroupWorkflowExecutor(SandboxedWorkflowExecutor):
    context_class = GroupWorkflowContext
