# coding: utf-8


from django.utils.encoding import force_text

from idm.core.workflow.plain.node import NodeWrapper as PlainNodeWrapper
from idm.core.workflow.sandbox.manager.sanboxed import SandboxedMixin
from idm.core.workflow.sandbox.manager.user import UserWrapper
from idm.core.workflow.sandbox.serializable import Serializable


def nodify(thing):
    from idm.core.models import RoleNode
    if isinstance(thing, NodeWrapper):
        result = thing
    elif isinstance(thing, RoleNode):
        result = NodeWrapper(thing)
    else:
        raise ValueError('Cannot nodify')
    return result


class NodeWrapper(SandboxedMixin, Serializable, PlainNodeWrapper):
    ALLOWED_METHODS = [
        'get_parent', 'is_root', 'get_root', 'get_ancestor', 'get_ancestors', 'is_descendant_of', 'get_children',
        'get_aliases', 'get_responsibilities', 'get_owners',
    ]
    ALLOWED_PROPERTIES = [
        'parent', 'name', 'name_en', 'data', 'fullname', 'description', 'description_en',
        'slug', 'slug_path', 'value_path', 'parent_path',
    ]
    INITIAL_PROPERTIES = [
        'slug', 'slug_path', 'value_path', 'parent_path',
    ]

    def __init__(self, node, context=None):
        super(NodeWrapper, self).__init__(node)
        self.context = context

    @classmethod
    def from_dict(cls, data, context=None):
        from idm.core.models import RoleNode
        node_id = data['id']
        node = RoleNode.objects.get(id=node_id)
        return cls(node, context=context)

    def _as_dict(self):
        return {
            'id': self.node.id,
        }

    def _check_permissions(self):
        if not getattr(self, 'context', None):
            raise ValueError('context variable should be specified')
        inspecting_system = self.context['system']._system
        inspected_system = self.node.system
        self._check_systems(inspecting_system, inspected_system)

    def _call_from_container(self, *args, **kwargs):
        self._check_permissions()
        return super(NodeWrapper, self)._call_from_container(*args, **kwargs)

    def get_aliases(self, type=None):
        return [AliasWrapper(alias) for alias in super(NodeWrapper, self).get_aliases(type=type)]

    def get_responsibilities(self):
        return [ResponsibilityWrapper(resp) for resp in super(NodeWrapper, self).get_responsibilities()]

    def get_owners(self, *args, **kwargs):
        return super(NodeWrapper, self).get_owners(*args, **kwargs)


class AliasWrapper(SandboxedMixin, Serializable):
    ALLOWED_METHODS = []
    ALLOWED_PROPERTIES = ['node', 'type', 'name', 'name_en', 'is_active']
    INITIAL_PROPERTIES = ['node', 'type', 'name', 'name_en', 'is_active']

    def __init__(self, alias, context=None):
        self.alias = alias
        self.context = context

    def __eq__(self, other):
        if isinstance(other, type(self)):
            return self.alias == other.alias
        else:
            return self.alias == other

    def __str__(self):
        return force_text(self.alias)

    def __getattr__(self, name):
        return getattr(self.alias, name)

    @classmethod
    def from_dict(cls, data, context=None):
        from idm.core.models import RoleAlias
        alias_id = data['id']
        alias = RoleAlias.objects.get(id=alias_id)
        return cls(alias, context=context)

    def _as_dict(self):
        return {
            'id': self.alias.id,
        }

    def _check_permissions(self):
        if not getattr(self, 'context', None):
            raise ValueError('context variable should be specified')
        inspecting_system = self.context['system']._system
        inspected_system = self.alias.node.system
        self._check_systems(inspecting_system, inspected_system)

    def _call_from_container(self, *args, **kwargs):
        self._check_permissions()
        return super(AliasWrapper, self)._call_from_container(*args, **kwargs)

    @property
    def node(self):
        return NodeWrapper(self.alias.node)


class ResponsibilityWrapper(SandboxedMixin, Serializable):
    ALLOWED_METHODS = []
    ALLOWED_PROPERTIES = ['node', 'is_active', 'user', 'notify']
    INITIAL_PROPERTIES = ['node', 'is_active', 'user', 'notify']

    def __init__(self, responsibility, context=None):
        self.responsibility = responsibility
        self.context = context

    def __eq__(self, other):
        if isinstance(other, type(self)):
            return self.responsibility == other.responsibility
        else:
            return self.responsibility == other

    def __str__(self):
        return force_text(self.responsibility)

    def __getattr__(self, name):
        return getattr(self.responsibility, name)

    @classmethod
    def from_dict(cls, data, context=None):
        from idm.core.models import NodeResponsibility
        responsibility_id = data['id']
        responsibility = NodeResponsibility.objects.get(id=responsibility_id)
        return cls(responsibility, context=context)

    def _as_dict(self):
        return {
            'id': self.responsibility.id,
        }

    def _check_permissions(self):
        if not getattr(self, 'context', None):
            raise ValueError('context variable should be specified')
        inspecting_system = self.context['system']._system
        inspected_system = self.responsibility.node.system
        self._check_systems(inspecting_system, inspected_system)

    def _call_from_container(self, *args, **kwargs):
        self._check_permissions()
        return super(ResponsibilityWrapper, self)._call_from_container(*args, **kwargs)

    @property
    def node(self):
        return NodeWrapper(self.responsibility.node)

    @property
    def user(self):
        return UserWrapper(self.responsibility.user)

