# coding: utf-8


import abc

from django.conf import settings

from idm.core.workflow.exceptions import WorkflowForbiddenOperationError, WorkflowForbiddenSystemError
from idm.core.workflow.sandbox.serializable import serialize


class SandboxedMixin(object, metaclass=abc.ABCMeta):
    @property
    @abc.abstractmethod
    def ALLOWED_PROPERTIES(self):
        pass

    @property
    @abc.abstractmethod
    def ALLOWED_METHODS(self):
        pass

    @property
    @abc.abstractmethod
    def INITIAL_PROPERTIES(self):
        pass

    @staticmethod
    def _check_systems(inspecting_system, inspected_system):
        if inspecting_system.slug == settings.IDM_SYSTEM_SLUG:
            return
        if inspected_system == inspecting_system:
            return
        if inspected_system.slug in settings.IDM_WORKFLOW_OPEN_SYSTEMS:
            return
        if inspected_system.systems_that_can_use_this_in_workflow.filter(pk=inspecting_system.pk).exists():
            return
        raise WorkflowForbiddenSystemError(
            'System {} is not allowed to inspect system {}'.format(
                inspecting_system.slug,
                inspected_system.slug,
            )
        )

    def _call_from_container(self, name, args, kwargs, is_callable):
        error_message = 'Method or attribute is not available from within workflow'
        if is_callable and name in self.ALLOWED_METHODS:
            try:
                func = getattr(self, name)
            except AttributeError:
                raise WorkflowForbiddenOperationError(error_message)
            return func(*args, **kwargs)
        elif not is_callable and name in self.ALLOWED_PROPERTIES:
            try:
                attr = getattr(self, name)
            except AttributeError:
                raise WorkflowForbiddenOperationError(error_message)
            return attr
        else:
            raise WorkflowForbiddenOperationError(error_message)

    def as_dict(self):
        result = self._as_dict()

        if self.INITIAL_PROPERTIES:
            result['initial_properties'] = serialize({
                attr: getattr(self, attr) for attr in self.INITIAL_PROPERTIES
            })

        return result
