# coding: utf-8


import sys
import traceback
from time import time

from django.conf import settings
from django.db.backends.utils import logger, CursorWrapper, CursorDebugWrapper
from django.db.utils import DatabaseError, InterfaceError
from django_pgaas.utils import can_retry_error, get_retry_range_for


def _retry(f):
    def wrapper(self, *args, **kwargs):
        retry_range = get_retry_range_for('Query')
        tb = None
        exc_val = None

        for _ in retry_range:
            try:
                return f(self, *args, **kwargs)
            except (DatabaseError, InterfaceError) as e:
                if self.db.in_atomic_block or not can_retry_error(e):
                    raise
                _, exc_val, tb = sys.exc_info()
                self.db.close()  # to avoid 'InterfaceError: connection already closed'
                self.db.ensure_connection()

        setattr(exc_val, '_django_pgaas_retried', True)
        raise exc_val.with_traceback(tb)

    return wrapper


class RetryCursorWrapper(CursorWrapper):
    @_retry
    def execute(self, sql, params=None):
        super().execute(sql, params=params)


class RetryCursorDebugWrapper(CursorDebugWrapper):
    @_retry
    def execute(self, sql, params=None):
        super().execute(sql, params=params)


class BetterCursorDebugWrapper(RetryCursorDebugWrapper):
    def execute(self, sql, params=None):
        start = time()
        # 8 уровней снизу достаточно для нас, чтобы понять, что происходит
        formatted_stack = traceback.format_stack(limit=8)

        try:
            return super(BetterCursorDebugWrapper, self).execute(sql, params)
        finally:
            stop = time()
            duration = stop - start
            sql = self.db.ops.last_executed_query(self.cursor, sql, params)
            self.db.queries_log.append({
                'sql': sql,
                'time': "%.3f" % duration,
            })
            if duration > settings.IDM_SQL_THRESHOLD:
                logger.debug('(%.3f) %s; args=%s; stack: %s' % (duration, sql, params, formatted_stack))
