from django.db.models import Func
from django.core.exceptions import FieldError

# Copypasted from Django 1.9 source
class Least(Func):
    """
    Chooses the minimum expression and returns it.

    If any expression is null the return value is database-specific:
    On Postgres, the minimum not-null expression is returned.
    On MySQL, Oracle, and SQLite, if any expression is null, null is returned.
    """
    function = 'LEAST'

    def __init__(self, *expressions, **extra):
        if len(expressions) < 2:
            raise ValueError('Least must take at least two expressions')
        super(Least, self).__init__(*expressions, **extra)

    def as_sqlite(self, compiler, connection):
        """Use the MIN function on SQLite."""
        return self.as_sqlite_super(compiler, connection, function='MIN')

    # From 1.9 Func.as_sqlite
    def as_sqlite_super(self, *args, **kwargs):
        sql, params = self.as_sql(*args, **kwargs)
        try:
            if self.output_field.get_internal_type() == 'DecimalField':
                sql = 'CAST(%s AS NUMERIC)' % sql
        except FieldError:
            pass
        return sql, params
