# coding: utf-8
"""
Базовые классы, использующиеся в admin.py файлах проекта
"""


from django.contrib.admin.options import ModelAdmin
from django.http import HttpResponseRedirect


class IdmModelAdminMixin(object):
    """
    Model Admin класс
    Отличия:
     - позволяет членам staff просматривать данные в админке;
     - опционально запрещает редактировать через админку вообще все;
     - редактировать в любом случае могут лишь суперпользователи.
    """
    editable = True  # могут ли суперпользователи редактировать модель в админке
    readonly_fields = []
    exclude = []

    class Media:
        js = (
            'https://yastatic.net/jquery/1.9.1/jquery.min.js',
            'https://yastatic.net/jquery-ui/1.10.3/jquery-ui.min.js',
        )
        css = {
            'all': ('https://yastatic.net/jquery-ui/1.10.3/themes/smoothness/jquery-ui.min.css',)
        }

    def get_readonly_fields(self, request, obj=None):
        """
        Кроме суперпользователей, для всех пользвователей все поля - read-only
        """
        if request.user.is_superuser and self.editable:
            readonly_fields = list(self.readonly_fields)
        else:
            readonly_fields = [field.name for field in self.model._meta.fields + self.model._meta.many_to_many
                               if field.name not in self.exclude]
        return readonly_fields

    def has_change_permission(self, request, obj=None):
        """
        Здесь True для членов staff, чтобы они видели списки объектов в админке
        """
        if request.method == 'GET' and request.user.is_active and request.user.is_staff:
            return True
        return super(IdmModelAdminMixin, self).has_change_permission(request, obj)

    def has_add_permission(self, request, obj=None):
        if self.editable and request.user.is_superuser:
            return super(IdmModelAdminMixin, self).has_add_permission(request)
        return False

    def has_delete_permission(self, request, obj=None):
        if self.editable and request.user.is_superuser:
            return super(IdmModelAdminMixin, self).has_delete_permission(request, obj)
        return False

    def has_module_permission(self, request):
        return request.user.is_staff or request.user.is_superuser

    def _redirect_back(self, request):
        if request.POST.get('referer'):
            return HttpResponseRedirect(request.POST['referer'])
        if request.META.get('HTTP_REFERER'):
            return HttpResponseRedirect(request.META['HTTP_REFERER'])


class DefaultIdmModelAdmin(IdmModelAdminMixin, ModelAdmin):
    pass
