from json import dumps

from psycopg2.extras import Json
from django.core.serializers.json import DjangoJSONEncoder
from django.contrib.postgres.fields import JSONField
from django.contrib.postgres.lookups import JSONExact
from django.db import models
from django.db.models.expressions import Col
from django.db.models.fields.related_descriptors import ForwardManyToOneDescriptor

import yenv


class RelationNotLoaded(Exception):
    pass


class StrictForwardManyToOne(ForwardManyToOneDescriptor):
    def __get__(self, instance, cls=None):
        if yenv.type != 'development.unittest':
            return super().__get__(instance, cls)

        if instance is None:
            return self

        if getattr(instance, self.field.attname) is None:
            return None

        if self.is_cached(instance):
            return self.field.get_cached_value(instance)

        raise RelationNotLoaded(
            'Relation `{rel}` not loaded. Use `select_related` or '
            '`fetch_{rel}`'.format(rel=self.field.name)
        )

    def explicit_get(self, instance, cls=None):
        return super().__get__(instance, cls)


class StrictForeignKey(models.ForeignKey):

    def contribute_to_class(self, cls, name, **kwargs):
        super().contribute_to_class(cls, name, **kwargs)
        #  Override the descriptor defined by ForeignObject
        descriptor = StrictForwardManyToOne(self)
        setattr(cls, self.name, descriptor)
        #  Add a method so you don't always have to use select_related
        fetch_name = 'fetch_{rel}'.format(rel=self.name)
        setattr(cls, fetch_name, lambda inst: descriptor.explicit_get(inst))


def _dumps(value):
    return dumps(value, cls=DjangoJSONEncoder)


class JSONFieldDjangoEncoder(JSONField):
    def get_prep_value(self, value):
        if value is not None:
            return Json(value, dumps=_dumps)
        return value


class NullJSONField(JSONField):
    # Хотим IS NULL запросы, которые выпилили
    # https://code.djangoproject.com/ticket/31324
    pass


class NullJsonExact(JSONExact):
    @property
    def can_use_none_as_rhs(self):
        # Treat None lookup values as null only in queries by json key
        return not isinstance(self.lhs, Col)

    def process_rhs(self, compiler, connection):
        result = super().process_rhs(compiler, connection)
        if self.can_use_none_as_rhs and result == ('%s', [None]):
            return "'null'", []

        return result


NullJSONField.register_lookup(NullJsonExact)
