# -*- coding: utf-8 -*-
import copy
import logging
import resource
import time

from django.conf import settings
from django.db import DatabaseError, InterfaceError, InternalError
from django.utils import timezone
from django.utils.encoding import force_text
from django_tools_log_context.celery import CtxAwareMixin
from django_tools_log_context.profiler import execution_profiler
from ylog.context import log_context

from idm.celery_app import app as celery_app
from idm.utils.lock import LockAlreadyAcquired
from idm.utils.cleansing import filter_password
from idm.core.utils import create_or_update_model


DATETIME_FORMAT_STRING = '%Y-%m-%d %H:%M:%S%z'


class UnrecoverableError(RuntimeError):
    def __init__(self, *args, action_id=None, **kwargs):
        self.action_id = action_id
        super().__init__(*args, **kwargs)


class DelayingError(RuntimeError):
    """Просто исключение, чтобы явно перезапустить таск"""


class IdmCtxAwareMixin(CtxAwareMixin):
    def __call__(self, *args, **kwargs):
        # Метод __call__ переопределен, чтобы вызывать внутри себя run вместо __call__.
        # А также писать в логи дополнительные данные - все параметры тасок, которые заканчиваются на _id.
        log_context_data = getattr(self.request, 'log_context', None)
        if log_context_data is None:
            headers = self.request.headers or {}
            log_context_data = headers.get('log_context', {})
        log_context_data['celery_task'] = {'name': self.name}
        for k, v in kwargs.items():
            if k.endswith('_id'):
                log_context_data[k] = v
        with log_context(**log_context_data):
            with execution_profiler(self.name, code_block_type='celery task', threshold=self._execution_threshold):
                self.log.info('Started celery task %s', self.name)
                return self.run(*args, **kwargs)


class BaseTask(IdmCtxAwareMixin, celery_app.Task):
    """Этот класс служит для надежного выполнения задач, состоящих из ряда шагов.

    Для реализации конкретной задачи, следует унаследовать от этого класса.
    У класса наследника должен быть метод `init`, который реализует первый
    шаг. Метод `init` должен возвращать dict c одним обязательным ключом step,
    содержащем имя следующего метода. Так же, этот dict может содержать
    дополнительные параметры для этого метода. Если же метод возвращает None,
    то считается, что он в списке последний и задача выполнена.

    В случае, если выполнение задачи прерывается из-за ошибки, она откладывается
    на `default_retry_delay` секунд и будет продолжена с того же метода на котором
    прервалась.
    """
    max_retries = 6
    default_retry_delay = 2
    retry_queue = None
    throws = (UnrecoverableError,)
    # Strategy = 'idm.framework.celery.strategy:better_strategy'  TODO: можно попробовать починить
    monitor_success = True  # Определяет, надо ли записивать в БД время последнего успешного завершения таски

    def __init__(self, *args, **kwargs):
        celery_app.Task.__init__(self, *args, **kwargs)
        self.log = logging.getLogger('idm.tasks.' + self.__class__.__name__)

    def run(self, step='init', *args, **kwargs):
        task_start = kwargs.pop('_monitoring_start_datetime', None)
        if task_start is not None:
            task_start = timezone.datetime.strptime(task_start, DATETIME_FORMAT_STRING)
        else:
            task_start = timezone.now()

        initial_usage = resource.getrusage(resource.RUSAGE_SELF)
        self.log.info(
            'Processing the task: step=%r, kwargs=%r, maxrss=%s MB', step, kwargs,
            initial_usage.ru_maxrss // 1024
        )

        func = getattr(self, str(step), None)
        countdown = kwargs.pop('countdown', 1)
        if func is None:
            message = 'Method "%s.%s" not found.' % (type(self), step)
            self.log.error(message)
            e = UnrecoverableError(message)
            self._failed(e, kwargs)
            raise e

        while func is not None:
            self.log.info('Running "%s". kwargs=%r', step, kwargs)
            start = time.time()

            try:
                try:
                    result = func(**kwargs)
                except (DatabaseError, InterfaceError, InternalError) as e:
                    raise DelayingError(e)

                self.log.debug('Result "%r".', result)
                kwargs_for_log = copy.deepcopy(kwargs)
                kwargs_for_log.pop('secret_context', None)
                self.log.info(
                    'Finish "%s": %0.2f sec. kwargs=%r', step,
                    time.time() - start, kwargs_for_log)

                if not result:
                    break
                if not isinstance(result, dict):
                    raise UnrecoverableError(f'Step "{step}" return bad result: {result}')

                step = result.pop('step', '')
                func = getattr(self, step, None)
                kwargs = result
            except UnrecoverableError as e:
                self.log.warning('UnrecoverableError during: "%s", kwargs=%r',
                                 step, filter_password(kwargs), exc_info=1)

                if getattr(e, 'action_id', None):
                    kwargs['action_id'] = e.action_id

                self._failed(e, kwargs)
                return
            except DelayingError as e:
                self.log.warning('DelayingError during: "%s", kwargs=%r',
                                 step, filter_password(kwargs), exc_info=1)
                kwargs['step'] = step
                kwargs['countdown'] = 2 * countdown
                kwargs['_monitoring_start_datetime'] = task_start.strftime(DATETIME_FORMAT_STRING)
                try:
                    self.retry(args, kwargs, countdown=countdown, queue=self.retry_queue)
                except self.MaxRetriesExceededError:
                    # все, больше этот таск не повторяем
                    del kwargs['_monitoring_start_datetime']
                    self.log.warning('MaxRetriesExceededError during: "%s", kwargs=%r',
                                     step, filter_password(kwargs), exc_info=1)
                    self._failed(e, kwargs, retry=True)
                return
            except LockAlreadyAcquired:
                return
            except Exception as e:
                self.log.exception(
                    'Task %s, step %s failed with an unexpected exception, args: %r, kwargs: %r',
                    self.__class__.__name__, step, args, filter_password(kwargs)
                )
                self._failed(e, kwargs, retry=True)
                return

        if self.monitor_success and settings.ENABLE_COMMAND_MONITORING:
            from idm.core.models import CommandTimestamp
            try:
                create_or_update_model(
                    model=CommandTimestamp,
                    obj_filter={'command': self.task_name},
                    defaults={
                        'last_success_start': task_start,
                        'last_success_finish': timezone.now(),
                    }
                )
            except DatabaseError:
                self.log.exception('Can\'t save timestamp for task {} to database'.format(self.task_name))

        final_usage = resource.getrusage(resource.RUSAGE_SELF)
        self.log.info(
            'Processing the task: done. Final step=%r args=%r, kwargs=%r, maxrss=%s MB, task has eaten %s Mb',
            step, args, kwargs,
            final_usage.ru_maxrss // 1024,
            (final_usage.ru_maxrss - initial_usage.ru_maxrss) // 1024,
        )

    def _failed(self, exc_val, kwargs, retry=False):
        """ Этот обработчик вызывается, если в процессе обработки
            задачи было выброшено исключение
            :param exc_val: исключение
            :param kwargs: аргументы таски
            :param retry: нужно будет перезапустить через некоторое время
        """
        pass

    @property
    def task_name(self):
        return 'tasks.{}'.format(self.__class__.__name__)
